package TJ;

import robocode.*;
import robocode.util.Utils;
import java.awt.geom.*;     // for Point2D's
import java.lang.*;         // for Double and Integer objects
import java.util.*; 		// for collection of waves
import java.awt.Color;

/**
 *	Exupery v1.38
 *	- Creator: TJ
 *	- Released: 2/14/10
 *	- Credits: 
 *		   - Voidious, for creating the WaveSurfing tutorial and code.
 *		   - Kawigi, for creating the GuessFactor Targeting tutorial and code.
 *		   - Anyone else who has contributed to the wiki.
 *	
 *	- Description: 
 *		     - Exupery is only a 1v1 bot.
 *		     - Movement is wave surfing, with distance-to-enemy segmentation.
 *		     - Targeting is GF Targeting, with distance-to-enemy segmentation.
 */
public class Exupery extends AdvancedRobot
{
    public static int BINS = 47; /**<The maximum number of waves to keep track of. */
    public static double _surfStats[][][] = new double[4][25][BINS]; /**<Keep track of dangeous areas on the waves.*/
    public Point2D.Double _myLocation; /**<Our bot's location. */
    public Point2D.Double _enemyLocation; /**<Enemy bot's location. */
 
    public ArrayList _enemyWaves; /**<A collection of waves, to surf and gather stats on. */
    public ArrayList _surfDirections; /**<A collection of our direction in relation to the enemy in past ticks.*/
    public ArrayList _surfAbsBearings; /**<A collection of past absolute bearings to enemy.*/
    
    public static double _oppEnergy = 100.0; /**<We must keep track of the enemy's energy level to detect EnergyDrop, indicating a bullet is fired*/
    
    public static Rectangle2D.Double _fieldRect = new java.awt.geom.Rectangle2D.Double(18, 18, 764, 564); /**<This is a rectangle that represents an 800x600 battle field, used for a simple, iterative WallSmoothing method (by Kawigi).*/
    public static double WALL_STICK = 160; /**<The wall stick indicates the amount of space we try to always have on either end of the tank (extending straight out the front or back) before touching a wall. */
    
	public List<WaveBullet> waves = new ArrayList<WaveBullet>();
	public static int[][] stats = new int[25][31]; // 31 is the number of unique GuessFactors we're using Note: this must be odd number so we can get GuessFactor 0 at middle.
    public int direction = 1;

    //start: run method
    public void run()
    {
	/**
	 * \brief Set up the basics, variables and a portion of the radar code.
	 *
	 *The main point of interest in this method is the narrow lock radar, enemies cannot escape this radar lock unless another enemy comes into focus and draws Survivor's attention.
	 */
	setScanColor(Color.red); //Layzar!

        _enemyWaves = new ArrayList();
        _surfDirections = new ArrayList();
        _surfAbsBearings = new ArrayList();
 
        setAdjustGunForRobotTurn(true);
        setAdjustRadarForGunTurn(true);
 
        do
	{
            // infinity lock/spinning radar
            turnRadarRightRadians(Double.POSITIVE_INFINITY);
        } while (true);
    }
    //end: run method
 
    //start: onScannedRobot method
    public void onScannedRobot(ScannedRobotEvent e)
    {
	/**
	 * \brief Contains algorithm for gun and bullet detection.
	 *
	 * - Whenever the enemy fires a bullet, create a new wave to track that bullet.
	 * - Gun algorithm uses iterative linear targeting. Will only fire if bot's energy is under 50.0 or if the battle is 1v1.
	 * - Gun will lower power when energy is 20.0 or below.
	 */
        _myLocation = new Point2D.Double(getX(), getY());
 
        double lateralVelocity = getVelocity()*Math.sin(e.getBearingRadians());
        double absBearing = e.getBearingRadians() + getHeadingRadians();
	
        setTurnRadarRightRadians(Utils.normalRelativeAngle(absBearing - getRadarHeadingRadians()) * 2);
		
        _surfDirections.add(0, new Integer((lateralVelocity >= 0) ? 1 : -1));
        _surfAbsBearings.add(0, new Double(absBearing + Math.PI));
 
 
        double bulletPower = _oppEnergy - e.getEnergy();
        if (bulletPower < 3.01 && bulletPower > 0.09 && _surfDirections.size() > 2)
	{
            EnemyWave ew = new EnemyWave();
            ew.fireTime = getTime() - 1; //bullet is detected the tick AFTER it is fired
            ew.bulletVelocity = bulletVelocity(bulletPower);
            ew.distanceTraveled = bulletVelocity(bulletPower);
            ew.direction = ((Integer)_surfDirections.get(2)).intValue();
            ew.directAngle = ((Double)_surfAbsBearings.get(2)).doubleValue();
            ew.fireLocation = (Point2D.Double)_enemyLocation.clone(); //bullet comes from the enemy's location on the last tick
			ew.bulletPow = bulletPower;
 
            _enemyWaves.add(ew);
        }
 
        _oppEnergy = e.getEnergy();
 
        //Update after EnemyWave detection, because that needs the previous enemy location as the source of the wave.
	 _enemyLocation = project(_myLocation, absBearing, e.getDistance());
 
        updateWaves();
        doSurfing();

	//Start - gun algorithm
	 
	// find our enemy's location:
	double ex = getX() + Math.sin(absBearing) * e.getDistance();
	double ey = getY() + Math.cos(absBearing) * e.getDistance();
 
	// Let's process the waves now:
	for (int i=0; i < waves.size(); i++)
	{
		WaveBullet currentWave = (WaveBullet)waves.get(i);
		if (currentWave.checkHit(ex, ey, getTime()))
		{
			waves.remove(currentWave);
			i--;
		}
	}

	//********************Basic Energy Management*****************
	double power = Math.min(2.5, (e.getEnergy()/4.0)); //Choose base power of 2, or max needed to kill enemy (enemy energy/4)
	//If the enemy is far away or bot is low on energy, lower power
	if (getEnergy() < 20.0 || e.getDistance() > 800)
	{
		power = 1.0;
	}
	//If enemy is close, raise to MAX power!
	else if (e.getDistance() < 200)
	{
		power = 3.0;
	}

	// don't try to figure out the direction they're moving 
	// they're not moving, just use the direction we had before
	if (e.getVelocity() != 0)
	{
		if (Math.sin(e.getHeadingRadians()-absBearing)*e.getVelocity() < 0)
			direction = -1;
		else
			direction = 1;
	}

	int flightTime = (int)(e.getDistance()/power); //bullet flight time

	int[] currentStats = stats[(int)(flightTime/10.0)]; ////////////////////////////////////SEGMENTATION
	
	WaveBullet newWave = new WaveBullet(getX(), getY(), absBearing, power, direction, getTime(), currentStats);

	int bestindex = 15;	// initialize it to be in the middle, guessfactor 0.
	for (int i=0; i<31; i++)
	{
		if (currentStats[bestindex] < currentStats[i])
		{
			bestindex = i;
		}
	}
 
	// this should do the opposite of the math in the WaveBullet:
	double guessfactor = (double)(bestindex - (currentStats.length - 1) / 2) / ((currentStats.length - 1) / 2);
	double angleOffset = direction * guessfactor * newWave.maxEscapeAngle();
    double gunAdjust = Utils.normalRelativeAngle(absBearing - getGunHeadingRadians() + angleOffset);
    setTurnGunRightRadians(gunAdjust);
    	
	//FIRE!!
	if (getGunHeat() == 0 && gunAdjust < Math.atan2(9, e.getDistance()))
	{
		setFire(power);
	}
	else
	{
		//don't fire if the gun has to turn more than half the robot (9 pixels)
	}

	waves.add(newWave);

	//End - gun algorithm
    }
    //end: onScannedRobot method
 
    //start: updateWaves method
    public void updateWaves()
    {
	/**
	 * \brief Keep waves up to date each tick.
	 *
	 * - Update distance each wave has travelled from source.
	 * - Clear waves once they have passed bot (as opposed to passed bot's center).
	 */
        for (int x = 0; x < _enemyWaves.size(); x++)
	{
            EnemyWave ew = (EnemyWave)_enemyWaves.get(x);
 
            ew.distanceTraveled = (getTime() - ew.fireTime) * ew.bulletVelocity;
            if (ew.distanceTraveled > _myLocation.distance(ew.fireLocation) + 50) //extra 50 is just a caution, bot could still get hit after bullet has passed center.
	    {
                _enemyWaves.remove(x);
                x--;
            }
        }
    }
    //end: updateWaves method
 
    //start: getClosestSurfableWave method
    public EnemyWave getClosestSurfableWave()
    {
	/**
	 * \brief Return the closest wave that hasn't completely passed bot.
	 */
        double closestDistance = 50000; // I juse use some very big number here
        EnemyWave surfWave = null;
 
        for (int x = 0; x < _enemyWaves.size(); x++)
	{
            EnemyWave ew = (EnemyWave)_enemyWaves.get(x);
            double distance = _myLocation.distance(ew.fireLocation) - ew.distanceTraveled;
 
            if (distance > ew.bulletVelocity && distance < closestDistance)
	    {
                surfWave = ew;
                closestDistance = distance;
            }
        }
 
        return surfWave;
    }
    //end: getClosestSurfableWave method
 
    //start: getFactorIndex method
    public static int getFactorIndex(EnemyWave ew, Point2D.Double targetLocation)
    {
	/**
	 * \brief Get location (index) of the bullet on this wave (array).
	 */
        double offsetAngle = (absoluteBearing(ew.fireLocation, targetLocation) - ew.directAngle); //relative angle that enemy aimed at to hit us; current angle from bot to wave source minus original angle (at fire time)
        double factor = Utils.normalRelativeAngle(offsetAngle) / maxEscapeAngle(ew.bulletVelocity) * ew.direction;
 
        return (int)limit(0, (factor * ((BINS - 1) / 2)) + ((BINS - 1) / 2), BINS - 1);
    }
    //end: getFactorIndex method
    
    //start: logHit method
    public void logHit(EnemyWave ew, Point2D.Double targetLocation, int dist)
    {
	/**
	 * \brief Update stat array to reflect danger in that area.
	 */
        int index = getFactorIndex(ew, targetLocation);
 
        for (int x = 0; x < BINS; x++)
		{
            // for the spot bin that we were hit on, add 1;
            // for the bins next to it, add 1 / 2;
            // the next one, add 1 / 5; and so on...
            _surfStats[(int)(ew.bulletPow)][dist][x] += 1.0 / (Math.pow(index - x, 2) + 1);
        }
    }
    //end: logHit method

    //start: onHitByBullet method
    public void onHitByBullet(HitByBulletEvent e)
    {
	/**
	 * \brief Update surfStats when hit by a bullet.
	 *
	 * - Adjust oppEnergy (accurate energy tracking is directly proportional to wave surfing prowess).
	 * - Figure out which wave hit us, update stats, remove wave.
	 */
	
	_oppEnergy += 3*(e.getPower());

	double bulletPow = e.getBullet().getPower();
        // If the _enemyWaves collection is empty, we must have missed the detection of this wave somehow.
	if (!_enemyWaves.isEmpty())
	{
            Point2D.Double hitBulletLocation = new Point2D.Double(e.getBullet().getX(), e.getBullet().getY());
            EnemyWave hitWave = null;
 
            // look through the EnemyWaves, and find one that could've hit us.
            for (int x = 0; x < _enemyWaves.size(); x++)
	    	{
                EnemyWave ew = (EnemyWave)_enemyWaves.get(x);
 		
				//check if distance wave has travelled is within 50 units of our current distance from it's source and if the velocity is the same as the bullet we were hit by.
                if (Math.abs(ew.distanceTraveled - _myLocation.distance(ew.fireLocation)) < 45 && Math.abs(bulletVelocity(bulletPow) - ew.bulletVelocity) < 0.001)
				{
                    hitWave = ew;
                    break;
                }
            }
 
            if (hitWave != null)
	    	{
                logHit(hitWave, hitBulletLocation, (int)(_myLocation.distance(hitWave.fireLocation)/50));
 
                // We can remove this wave now, of course.
                _enemyWaves.remove(_enemyWaves.lastIndexOf(hitWave));
            }
        }
    }
    //end: onHitByBullet method
 
    //start: onBulletHit method
    public void onBulletHit(BulletHitEvent e)
    {
	    /**
	     * \brief Ajust oppEnergy to account for hitting opponent with bullet.
	     *
	     * - Acuurate energy tracking is directly proportional to wave surfing prowess.
	     */
	 
	    _oppEnergy = e.getEnergy();
    }
    //end: onBulletHit method
    
    //start: onBulletHitBullet method
    public void onBulletHitBullet(BulletHitBulletEvent e)
    {
	/**
	 * \brief Update surfStats when our bullet is hit by a bullet.
	 *
	 * - Figure out which wave hit our bullet, update stats, remove wave.
	 */
	double bulletPow = e.getHitBullet().getPower();

        if (!_enemyWaves.isEmpty())
	{
            Point2D.Double hitBulletLocation = new Point2D.Double(e.getBullet().getX(), e.getBullet().getY());
            EnemyWave hitWave = null;
 
            // look through the EnemyWaves, and find one that could've hit our bullet.
            for (int x = 0; x < _enemyWaves.size(); x++)
	    {
                EnemyWave ew = (EnemyWave)_enemyWaves.get(x);
 		
		//check if distance wave has travelled is within 15 units of our bullet's distance from it's source and if the velocity is the same as the bullet ours was hit by.
                if ((Math.abs(ew.distanceTraveled - hitBulletLocation.distance(ew.fireLocation))) < 15 && Math.abs(bulletVelocity(bulletPow) - ew.bulletVelocity) < 0.001)
		{
                    hitWave = ew;
                    break;
                }
            }
 
            if (hitWave != null)
	    {
                logHit(hitWave, hitBulletLocation, (int)(hitBulletLocation.distance(hitWave.fireLocation)/50));
 
                // We can remove this wave now, of course.
                _enemyWaves.remove(_enemyWaves.lastIndexOf(hitWave));
            }
        }
    }
    //end: onBulletHitBullet method

    //start: predictPosition method
    // CREDIT: mini sized predictor from Apollon, by rozu
    // http://robowiki.net?Apollon
    public Point2D.Double predictPosition(EnemyWave surfWave, int dir)
    {
	/**
	 * \brief Predicts were our bot would be when the wave intercepts it.
	 *
	 * Given robocode physics, the wave we are surfing and the direction we predict, this method will predict where the bot will be when the wave intercepts it.
	 */
    	Point2D.Double predictedPosition = (Point2D.Double)_myLocation.clone();
    	double predictedVelocity = getVelocity();
    	double predictedHeading = getHeadingRadians();
    	double maxTurning, moveAngle, moveDir;
 
        int counter = 0; // number of ticks in the future
        boolean intercepted = false;
 
    	do
		{
    		moveAngle = wallSmoothing(predictedPosition, absoluteBearing(surfWave.fireLocation, predictedPosition) + (dir * (Math.PI/2)), dir) - predictedHeading;
    		moveDir = 1;
 
    		if(Math.cos(moveAngle) < 0)
			{
    			moveAngle += Math.PI;
    			moveDir = -1;
    		}
 
    		moveAngle = Utils.normalRelativeAngle(moveAngle);
 
    		// maxTurning is built in like this, you can't turn more than this in one tick
    		maxTurning = Math.PI/720d*(40d - 3d*Math.abs(predictedVelocity));
    		predictedHeading = Utils.normalRelativeAngle(predictedHeading + limit(-maxTurning, moveAngle, maxTurning));
 
    		// this one is nice ;). if predictedVelocity and moveDir have
            	// different signs you want to break down
    		// otherwise you want to accelerate (look at the factor "2")
    		predictedVelocity += (predictedVelocity * moveDir < 0 ? 2*moveDir : moveDir);
    		predictedVelocity = limit(-8, predictedVelocity, 8);
 
    		// calculate the new predicted position
    		predictedPosition = project(predictedPosition, predictedHeading, predictedVelocity);
 
            	counter++;
 
            	if (predictedPosition.distance(surfWave.fireLocation) < surfWave.distanceTraveled + (counter * surfWave.bulletVelocity) + surfWave.bulletVelocity)
		{
                	intercepted = true;
            	}
    	} while(!intercepted && counter < 500);
 
    	return predictedPosition;
    }
    //end: predictPosition method
 
    //start: checkDanger method
    public double checkDanger(EnemyWave surfWave, int dir)
    {
	/*
	 * \brief Calculate a danger index for a given position.
	 */
		int index = getFactorIndex(surfWave, predictPosition(surfWave, dir));
 
        return _surfStats[(int)(surfWave.bulletPow)][(int)(_myLocation.distance(surfWave.fireLocation)/50)][index];
    }
    //end: checkDanger method
 
    //start: doSurfing method
    public void doSurfing()
    {
	/** 
	 * \brief Find which direction would be safest to move.
	 *
	 * - Compare moving clockwise and counterclockwise.
	 * - Move in the direction with the lowest danger.
	 */
        EnemyWave surfWave = getClosestSurfableWave();
 
        if (surfWave == null) { return; }
 
        double dangerLeft = checkDanger(surfWave, -1);
        double dangerRight = checkDanger(surfWave, 1);
 
        double goAngle = absoluteBearing(surfWave.fireLocation, _myLocation);

		if (dangerLeft < dangerRight)
		{
            goAngle = wallSmoothing(_myLocation, goAngle - (Math.PI/2), -1);
        }
		else
		{
            goAngle = wallSmoothing(_myLocation, goAngle + (Math.PI/2), 1);
        }
	
        setBackAsFront(this, goAngle);
    }
    //end: doSurfing method


    //***********UTILITY CLASSES AND METHODS********************

    //start: EnemyWave class
    //used to define/store waves.
    class EnemyWave
    {
        Point2D.Double fireLocation;
        long fireTime;
        double bulletVelocity, directAngle, distanceTraveled, bulletPow;
        int direction;
 
        public EnemyWave() { }
    }
    //end: EnemyWave class
 
    //start: wallSmoothing method
    // CREDIT: Iterative WallSmoothing by Kawigi
    //   - return absolute angle to move at after account for WallSmoothing
    // robowiki.net?WallSmoothing
    public double wallSmoothing(Point2D.Double botLocation, double angle, int orientation)
    {
        while (!_fieldRect.contains(project(botLocation, angle, WALL_STICK)))
	{
            angle += orientation*0.05;
        }
        return angle;
    }
    //end: wallSmoothing method
 
    //start: project method
    // CREDIT: from CassiusClay, by PEZ
    //   - returns point length away from sourceLocation, at angle
    // robowiki.net?CassiusClay
    public static Point2D.Double project(Point2D.Double sourceLocation, double angle, double length)
    {
        return new Point2D.Double(sourceLocation.x + Math.sin(angle) * length, sourceLocation.y + Math.cos(angle) * length);
    }
    //end: project method
 
    //start: absoluteBearing method
    // got this from RaikoMicro, by Jamougha, but I think it's used by many authors
    //  - returns the absolute angle (in radians) from source to target points
    public static double absoluteBearing(Point2D.Double source, Point2D.Double target)
    {
        return Math.atan2(target.x - source.x, target.y - source.y);
    }
    //end: absoluteBearing method

    //start: limit method
    public static double limit(double min, double value, double max)
    {
        return Math.max(min, Math.min(value, max));
    }
    //end: limit method

    //start: bulletVelocity method
    public static double bulletVelocity(double power)
    {
        return (20D - (3D*power));
    }
    //end: bulletVelocity method
 
    //start: maxEscapeAngle method
    public static double maxEscapeAngle(double velocity)
    {
        return Math.asin(8.0/velocity);
    }
    //end: maxEscapeAngle method
    
    //start: setBackAsFront method
    public static void setBackAsFront(AdvancedRobot robot, double goAngle)
    {
        double angle = Utils.normalRelativeAngle(goAngle - robot.getHeadingRadians());
        if (Math.abs(angle) > (Math.PI/2))
	{
            if (angle < 0)
	    {
                robot.setTurnRightRadians(Math.PI + angle);
            }
	    else
	    {
                robot.setTurnLeftRadians(Math.PI - angle);
            }
            robot.setBack(100);
        }
	else
	{
            if (angle < 0)
	    {
                robot.setTurnLeftRadians(-1*angle);
            }
	    else
	    {
                robot.setTurnRightRadians(angle);
            }
            robot.setAhead(100);
        }
    }
    //end: setBackAsFront method
 
    //start: onPaint method
    public void onPaint(java.awt.Graphics2D g)
    {
	 /** 
	  * \brief When paint is turn on, this method will paint the waves that the bot detects.
	  */
         g.setColor(java.awt.Color.red);
         for(int i = 0; i < _enemyWaves.size(); i++)
 	 {
             EnemyWave w = (EnemyWave)(_enemyWaves.get(i));
             Point2D.Double center = w.fireLocation;
 
             int radius = (int)w.distanceTraveled;
 
             if(radius - 40 < center.distance(_myLocation))
       	     {
                g.drawOval((int)(center.x - radius ), (int)(center.y - radius), radius*2, radius*2);
	     }
         }
    }
    //end: onPaint method
 
    //start: WaveBullet class
    public class WaveBullet
    {
		private double startX, startY, startBearing, power;
		private long   fireTime;
		private int    direction;
		private int[]  returnSegment;
 	
		public WaveBullet(double x, double y, double bearing, double power, int direction, long time, int[] segment)
		{
			startX         = x;
			startY         = y;
			startBearing   = bearing;
			this.power     = power;
			this.direction = direction;
			fireTime       = time;
			returnSegment  = segment;
		}

	    public double getBulletSpeed()
		{
			return 20 - power * 3;
		}
 
		public double maxEscapeAngle()
		{
			return Math.asin(8 / getBulletSpeed());
		}
	
		public boolean checkHit(double enemyX, double enemyY, long currentTime)
		{
			// if the distance from the wave origin to our enemy has passed
			// the distance the bullet would have traveled...
			if (Point2D.distance(startX, startY, enemyX, enemyY) <= (currentTime - fireTime) * getBulletSpeed())
			{
				double desiredDirection = Math.atan2(enemyX - startX, enemyY - startY);
				double angleOffset = Utils.normalRelativeAngle(desiredDirection - startBearing);
				double guessFactor = Math.max(-1, Math.min(1, angleOffset / maxEscapeAngle())) * direction;
				int index = (int) Math.round((returnSegment.length - 1) /2 * (guessFactor + 1));
				returnSegment[index]++;
				return true;
			}
			return false;
		}
    }
    //end: WaveBullet class
}
