/*
 * PatternAnalyzer - simple pattern analyzer for BlotBot
 * Copyright (C) 2002  Joachim Hofer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * You can contact the author via email (qohnil@johoop.de) or write to
 * Joachim Hofer, Feldstr. 12, D-91052 Erlangen, Germany.
 */
package qohnil.blot;

import qohnil.util.ObjectBuffer;
import qohnil.util.BotMath;
import qohnil.util.Polar;
import qohnil.util.Logger;
import qohnil.neural.NeuralNet;
import qohnil.neural.TrainingPattern;
import qohnil.neural.util.Preprocessor;

import java.util.ArrayList;

/**
 * TODO:
 *
 */
public class NeuralPatternAnalyzer {
    ArrayList historyList = new ArrayList(100);
    int patternIdx = 0;
    ArrayList trainingPatterns = new ArrayList(10000);
    ArrayList trainingPatterns16 = new ArrayList(10000);
    ArrayList trainingPatterns24 = new ArrayList(10000);
    ObjectBuffer history = null;

    NeuralNet nn8 = new NeuralNet(32, 10, 0, 3);
    NeuralNet nn16 = new NeuralNet(64, 14, 0, 3);
    NeuralNet nn24 = new NeuralNet(96, 17, 0, 3);

    double error = Double.POSITIVE_INFINITY;

    public NeuralPatternAnalyzer() {}

    public void addHistory(ObjectBuffer history) {
        historyList.add(history);
    }

    private void chooseIndex(int minPatternLength) {
        int historyIdx = BotMath.random.nextInt(historyList.size());
        history = (ObjectBuffer)historyList.get(historyIdx);
        int numPatterns = history.size();
        patternIdx = BotMath.random.nextInt(numPatterns - minPatternLength - 1);
    }

    public double train(int numIterations) {
        if (trainingPatterns.size() > (96 * 17 + 17 * 3) * 2) {
            error = 0.0;
            for (int i = 0; i < numIterations; i++) {
                // train the network that predicts 8 steps ahead
                int idx = BotMath.random.nextInt(trainingPatterns.size());
                TrainingPattern tp = (TrainingPattern)trainingPatterns.get(idx);
                error += nn8.learnSingle(tp, 0.1, 0.7);
            }
            error /= numIterations;

            double error2 = 0.0;
            for (int i = 0; i < numIterations; i++) {
                // train the network that predicts 16 steps ahead
                int idx = BotMath.random.nextInt(trainingPatterns.size());
                TrainingPattern tp = (TrainingPattern)trainingPatterns16.get(idx);
                error2 += nn16.learnSingle(tp, 0.1, 0.7);
            }
            error2 /= numIterations;

            double error3 = 0.0;
            for (int i = 0; i < numIterations; i++) {
                // train the network that predicts 24 steps ahead
                int idx = BotMath.random.nextInt(trainingPatterns.size());
                TrainingPattern tp = (TrainingPattern)trainingPatterns24.get(idx);
                error3 += nn24.learnSingle(tp, 0.1, 0.7);
            }
            error3 /= numIterations;

            error += error2 + error3;
            error /= 3;
        }

        return error;
    }

    public void update() {
        // retrieve history of current round
        history = (ObjectBuffer)historyList.get(historyList.size() - 1);
        if (history.size() < 73) {
            return;
        }

        // create a new training pattern for the current enemy movement data
        double[] inps = new double[32];
        int j = 0;
        for (int i = 0; i < 16; i++) {
            MovableData bot = (MovableData)history.get(8 + i);
            inps[j++] = Preprocessor.normalize(bot.velocity, -8.0, 8.0);
            inps[j++] = Preprocessor.normalize(bot.turnVelocity, Math.toRadians(-10.0), Math.toRadians(10.0));
        }
        double[] target = new double[3];
        Polar p = getReaction(8, 8);
        target[0] = Preprocessor.normalize(p.getDistance(), 0.0, 64.0, 0.2);
        target[1] = Preprocessor.normalize(Math.sin(p.getBearing()), -1.0, 1.0, 0.2);
        target[2] = Preprocessor.normalize(Math.cos(p.getBearing()), -1.0, 1.0, 0.2);

        TrainingPattern pattern = new TrainingPattern(inps, target);
        trainingPatterns.add(pattern);

        // create a new training pattern for the current enemy movement data
        inps = new double[64];
        j = 0;
        for (int i = 0; i < 32; i++) {
            MovableData bot = (MovableData)history.get(16 + i);
            inps[j++] = Preprocessor.normalize(bot.velocity, -8.0, 8.0);
            inps[j++] = Preprocessor.normalize(bot.turnVelocity, Math.toRadians(-10.0), Math.toRadians(10.0));
        }
        target = new double[3];
        p = getReaction(16, 16);
        target[0] = Preprocessor.normalize(p.getDistance(), 0.0, 128.0, 0.2);
        target[1] = Preprocessor.normalize(Math.sin(p.getBearing()), -1.0, 1.0, 0.2);
        target[2] = Preprocessor.normalize(Math.cos(p.getBearing()), -1.0, 1.0, 0.2);

        pattern = new TrainingPattern(inps, target);
        trainingPatterns16.add(pattern);

        // create a new training pattern for the current enemy movement data
        inps = new double[96];
        j = 0;
        for (int i = 0; i < 48; i++) {
            MovableData bot = (MovableData)history.get(24 + i);
            inps[j++] = Preprocessor.normalize(bot.velocity, -8.0, 8.0);
            inps[j++] = Preprocessor.normalize(bot.turnVelocity, Math.toRadians(-10.0), Math.toRadians(10.0));
        }
        target = new double[3];
        p = getReaction(24, 24);
        target[0] = Preprocessor.normalize(p.getDistance(), 0.0, 192.0, 0.2);
        target[1] = Preprocessor.normalize(Math.sin(p.getBearing()), -1.0, 1.0, 0.2);
        target[2] = Preprocessor.normalize(Math.cos(p.getBearing()), -1.0, 1.0, 0.2);

        pattern = new TrainingPattern(inps, target);
        trainingPatterns24.add(pattern);
    }

    /**
     * Precondition: history is set to the history in question!
     */
    private Polar getReaction(int currentIdx, int lookAheadTime) {
        MovableData previous = (MovableData)history.get(currentIdx);
        MovableData reaction = (MovableData)history.get(currentIdx - lookAheadTime);

        // compute bearing and distance
        Polar p = BotMath.coordsToPolar(reaction.coords, previous.coords);
        double angle = p.getBearing() - previous.heading;
        double velocity = p.getDistance();

        return new Polar(BotMath.normalizeRelativeAngle(angle), velocity);
    }

    public Polar getPrediction(int lookAhead) {
        if (history.size() < 48) {
            return new Polar(0.0, 1.0);
        }

        double[] inps = new double[32];
        int j = 0;
        for (int i = 0; i < 16; i++) {
            MovableData bot = (MovableData)history.get(patternIdx + i);
            inps[j++] = Preprocessor.normalize(bot.velocity, -8.0, 8.0);
            inps[j++] = Preprocessor.normalize(bot.turnVelocity, Math.toRadians(-10.0), Math.toRadians(10.0));
        }
        double[] results = nn8.propagate(inps);

        double predictedDistance8 = Preprocessor.denormalize(results[0], 0.0, 64.0, 0.2);

        double sinRelHeading = Preprocessor.denormalize(results[1], -1.0, 1.0, 0.2);
        double cosRelHeading = Preprocessor.denormalize(results[2], -1.0, 1.0, 0.2);
        double norm = Math.sqrt(sinRelHeading * sinRelHeading + cosRelHeading * cosRelHeading);
        double predictedAngle8 = Math.acos(cosRelHeading / norm);
        if (sinRelHeading < 0) {
            predictedAngle8 = BotMath.PI2 - predictedAngle8;
        }

        // adapt linearly to actual time
        if (lookAhead <= 8.0) {
            predictedAngle8 *= lookAhead / 8.0;
            predictedDistance8 *= lookAhead / 8.0;
            return new Polar(predictedAngle8, predictedDistance8);
        }

        inps = new double[64];
        j = 0;
        for (int i = 0; i < 32; i++) {
            MovableData bot = (MovableData)history.get(patternIdx + i);
            inps[j++] = Preprocessor.normalize(bot.velocity, -8.0, 8.0);
            inps[j++] = Preprocessor.normalize(bot.turnVelocity, Math.toRadians(-10.0), Math.toRadians(10.0));
        }
        results = nn16.propagate(inps);

        double predictedDistance16 = Preprocessor.denormalize(results[0], 0.0, 128.0, 0.2);

        sinRelHeading = Preprocessor.denormalize(results[1], -1.0, 1.0, 0.2);
        cosRelHeading = Preprocessor.denormalize(results[2], -1.0, 1.0, 0.2);
        norm = Math.sqrt(sinRelHeading * sinRelHeading + cosRelHeading * cosRelHeading);
        double predictedAngle16 = Math.acos(cosRelHeading / norm);
        if (sinRelHeading < 0) {
            predictedAngle16 = BotMath.PI2 - predictedAngle16;
        }

        // adapt linearly to actual time
        if (lookAhead <= 16.0) {
            double predictedAngle
                    =  ((lookAhead - 8) * predictedAngle16
                    + ((16 - lookAhead) * predictedAngle8))
                            / 8.0;
            double predictedDistance
                    =  ((lookAhead - 8) * predictedDistance16
                    + ((16 - lookAhead) * predictedDistance8))
                            / 8.0;

            return new Polar(predictedAngle, predictedDistance);
        }

        inps = new double[96];
        j = 0;
        for (int i = 0; i < 48; i++) {
            MovableData bot = (MovableData)history.get(patternIdx + i);
            inps[j++] = Preprocessor.normalize(bot.velocity, -8.0, 8.0);
            inps[j++] = Preprocessor.normalize(bot.turnVelocity, Math.toRadians(-10.0), Math.toRadians(10.0));
        }
        results = nn24.propagate(inps);

        double predictedDistance24 = Preprocessor.denormalize(results[0], 0.0, 192.0, 0.2);

        sinRelHeading = Preprocessor.denormalize(results[1], -1.0, 1.0, 0.2);
        cosRelHeading = Preprocessor.denormalize(results[2], -1.0, 1.0, 0.2);
        norm = Math.sqrt(sinRelHeading * sinRelHeading + cosRelHeading * cosRelHeading);
        double predictedAngle24 = Math.acos(cosRelHeading / norm);
        if (sinRelHeading < 0) {
            predictedAngle24 = BotMath.PI2 - predictedAngle24;
        }

        // adapt linearly to actual time
        double predictedAngle
                =  ((lookAhead - 16) * predictedAngle24
                + ((24 - lookAhead) * predictedAngle16))
                        / 8.0;
        double predictedDistance
                =  ((lookAhead - 16) * predictedDistance24
                + ((24 - lookAhead) * predictedDistance16))
                        / 8.0;

        return new Polar(predictedAngle, predictedDistance);
    }

    public double getError() {
        return error;
    }

    public int getNumPatterns() {
        int numPatterns = 0;
        for (int i = 0; i < historyList.size(); i++) {
            numPatterns += ((ObjectBuffer)historyList.get(i)).size();
        }
        return numPatterns;
    }
}
