package bvh.fry;

import robocode.*;
import robocode.util.Utils;
import java.awt.geom.Point2D;
import java.awt.Color;
import java.util.*;
import java.io.*;


/**
 * <h2>Freya</h2> <h3>- a robot by Bart "Loki" van Hest</h3>
 *
 * <p>Freya was the goddess of love, marriage, and fertility. Her identity and
 *   attributes were often confused with those of the goddess Frigg. As a deity
 *   of the dead, Freya was entitled to half the warriors killed in battle,
 *   the other half going to Odin.
 *   She was the daughter of Njord and the sister of the god Frey and originally
 *   one of the Vanir. She was frequently represented as riding in a chariot
 *   drawn by cats."</p>
 *
 1st: abc.tron3.Shadow 3.69	600175	420150	32490	137612	9904	17	0	362	170	139
2nd: rz.Aleph 0.34	546626	389800	17460	130462	8891	13	0	198	195	171
3rd: kawigi.mini.Coriantumr 1.1	534838	371500	16290	137664	8907	476	0	187	173	167
4th: pe.SandboxDT 2.71m	500739	371800	15210	108127	5527	74	0	172	81	176
5th: bvh.fry.Freya 0.81	476688	337950	8100	123638	6876	123	0	93	196	184
6th: bvh.fry.Freya 0.82m4k5	475521	335250	7290	125443	7023	492	22	84	194	187
7th: bvh.fry.Freya 0.82m0k5	468334	331450	9000	121073	6802	8	0	108	195	187
8th: florent.XSeries.X2 0.9	384960	263100	4680	111048	6086	38	6	52	65	97
9th: kid.Gladiator .6.3	384501	266350	13950	100188	3792	220	0	160	99	79
10th: gh.Griezel 0.4	379134	284300	10170	81794	2611	258	0	114	128	115
 *
 * @version 0.82
 * @author Bart van Hest
 */
public class Freya extends AdvancedRobot implements Constanten {
   public static boolean opslaanStatistiek;
   public static boolean debugmode;
   public static Hashtable opponenten;

   public Point2D.Double positie;
   public Hildisvin beweging;
   public Mjolnir kanon;
   public long beurtOvergeslagen;

   /**
    * run: Freya's hoofdroutine:
    */
   public void run() {
      initialiseerRobot();

      setTurnRadarRightRadians(Double.POSITIVE_INFINITY);

      do {
         log("run(): nieuwe loop ---------->");
         positie = new Point2D.Double(getX(), getY());

         if (kanon.doel != null) {
            stuurRadar(kanon.doel);
            beweging.uitvoeren(positie);

            if (kanon.doel.actief) {
               kanon.uitvoeren();
            }
         } else {
            stop();
            setTurnRadarRightRadians(Double.POSITIVE_INFINITY);
         }

         execute();
      } while (true);
   }

   /**
    * onScannedRobot: What to do when you see another robot
    * let op: van alle scanned bots tijdens een radar-sweep wordt de dichtstbijzijnde bot doorgegegeven!
    */
   public void onScannedRobot(ScannedRobotEvent evt) {
      log("onScannedRobot():");

      String naam = evt.getName();
      Doel d = null;
      d = (Doel) opponenten.get(naam);

      if (d == null) {
         d = new Doel(naam);
         opponenten.put(naam, d);
      }

      d.bijwerken(evt, this, getTime(), getHeadingRadians());

      kanon.onScannedRobot(d);
   }

   /**
    * Wanneer een tegenstander wordt uitgeschakeld moet deze tegenstander op non-actief
    * worden gezet in de hash-tabel met tegenstanders om te voorkomen dat data van deze
    * tegenstander nog wordt gebruikt bij bijv. navigatie.
    * Omdat het huidige doel een referentie is naar een Doel-object uit de hash-table
    * wordt deze automatisch bijgewerkt.
    *
    * @param evt Het RobotDeathEvent dat naar de robot wordt gestuurd als een andere
    * robot wordt uitgeschakeld.
    */
   public void onRobotDeath(RobotDeathEvent evt) {
      String naam = evt.getName();

      if (opponenten.containsKey(naam)) {
         Doel d = (Doel) opponenten.get(naam);
         d.actief = false;

         if (kanon.doel == d) {
            kanon.doel = null;
         }
      }

      // compacter:
      // ((Doel)opponenten.get(evt.getName())).actief = false;
   }

   /**
    * onHitByBullet: n.a.v. een treffer door de tegenstander worden de volgende gegevens bijgewerkt:
    * <ul>
    *  <li>treffers door de tegenstander.</li>
    *  <li>energie van de tegenstander (deze wordt opgehoogd met 3 keer de kogel vuurkracht).</li>
    *  <li>eventueel bijwerken doelgolf-statistiek.</li>
    *  </ul>:
    *
    * @param evt Het HitByBulletEvent dat naar de robot wordt gestuurd als de robot
    * wordt geraakt door een vijandelijke kogel.
    */
   public void onHitByBullet(HitByBulletEvent evt) {
      String naam = evt.getName();

      if (opponenten.containsKey(naam)) {
         Doel d = (Doel) opponenten.get(naam);
         d.treffers++;
         d.energie += evt.getPower() * 3D;
      }
   }

   /**
    * onBulletHit: n.a.v. een treffer op de tegenstander worden de volgende gegevens bijgewerkt:
    * <ul>
    *  <li>treffers op de tegenstander.</li>
    *  <li>energie van de tegenstander.</li>
    *  </ul>
    *
    * @param evt Het BulletHitEvent dat naar de robot wordt gestuurd als een kogel
    * een tegenstander raakt.
    */
   public void onBulletHit(BulletHitEvent evt) {
      String naam = evt.getName();

      if (opponenten.containsKey(naam)) {
         Doel d = (Doel) opponenten.get(naam);
         d.treffersFreya++;
         d.energie = evt.getEnergy();
      }
   }

   /**
    * N.a.v. een botsing met een tegenstander worden de volgende gegevens bijgewerkt:
    * <ul>
    *  <li>energie van de tegenstander.</li>
    *  </ul>
    *
    * @param evt Het HitRobotEvent dat naar de robot wordt gestuurd als deze met
    * een tegenstander botst.
    */
   public void onHitRobot(HitRobotEvent evt) {
      String naam = evt.getName();

      if (opponenten.containsKey(naam)) {
         Doel d = (Doel) opponenten.get(naam);
         d.energie -= 0.6D;
      }
   }

   /**
    * DOCUMENT ME!
    *
    * @param ev DOCUMENT ME!
    */
   public void onHitWall(HitWallEvent ev) {
      ++beweging.aantalBotsingenMetWand;
   }

   /**
    * De tijdens het gevecht verzamelde statistiek wordt opgeslagen.
    *
    * @param evt Het WinEvent dat naar de robot wordt gestuurd als de robot een slag wint.
    */
   public void onWin(WinEvent evt) {
      opslaanDoelStatistiek();
      log("GEWONNEN: " + beurtOvergeslagen + " keren beurt overgeslagen.");
      log("          " + beweging.aantalBotsingenMetWand + " de wanden geraakt.");
   }

   /**
    * Alle opponenten worden op inactief gezet als initiatie voor de volgende ronde. Verder
    * wordt de tijdens het gevecht verzamelde statistiek opgeslagen.
    *
    * @param evt Het DeathEvent dat naar de robot wordt gestuurd als de robot wordt
    * uitgeschakeld.
    */
   public void onDeath(DeathEvent evt) {
      opslaanDoelStatistiek();
      log("....DOOD: " + beurtOvergeslagen + " keren beurt overgeslagen.");
      log("          " + beweging.aantalBotsingenMetWand + " de wanden geraakt.");
   }

   /**
    * onSkippedTurn:  telt aantal keren dat beurt wordt overgeslagen;
    *
    * @param evt Het SkippedTurnEvent dat naar de robot wordt gestuurd als de binnen
    * de aan de robot toegekende tijd teveel bewerkingen worden uitgevoerd.
    */
   public void onSkippedTurn(SkippedTurnEvent evt) {
      beurtOvergeslagen += 1;
   }

   private void initialiseeropponenten() {
      log("initialiseeropponenten(): aantal opponenten = " + opponenten.size());

      for (Enumeration enm = opponenten.elements(); enm.hasMoreElements();) {
         Doel d = (Doel) enm.nextElement();
         log("initialiseeropponenten(): initialiseer doel = " + d.naam);
         d.actief = false;
         d.resetGolven();
      }
   }

   public Point2D.Double geefVolgendePositie() {
      double snelheid = getVelocity();

      double rotatie;
      if (getTurnRemaining() >= 0) {
         rotatie = Math.min(getTurnRemaining(), 10-0.75*Math.abs(snelheid));
      } else {
         rotatie = Math.max(getTurnRemaining(),-10+0.75*Math.abs(snelheid));
      }
      double nieuweBewegingsrichting = Math.toRadians(getHeading() + rotatie);
   
      return (new Point2D.Double(this.getX() + Math.sin(nieuweBewegingsrichting) * snelheid,
         this.getY() + Math.cos(nieuweBewegingsrichting) * snelheid));
   }

   /**
    * In melee-mode draait de radar rond, tenzij het kanon bijna klaar is om te schieten:
    * in dat geval wordt de radar gelocked op de tegenstander.
    * In 1-vs-1 gevechten wordt de radar altijd gelocked op de tegenstander, tenzij het
    * doel niet langer actief is (dus is vernietig, waarna een lock niet meer van belang
    * is).
    */
   public void stuurRadar(Doel d) {
      if (d.actief && (getOthers() == 1 || (getGunHeat() / getGunCoolingRate()) < 5D)) {
         setTurnRadarRightRadians(2.0D * Utils.normalRelativeAngle(BotUtils.bepaalRichting(positie, d)
               - getRadarHeadingRadians()));
         log("stuurRadar(): locked modus");
      } else {
         setTurnRadarRightRadians(Double.POSITIVE_INFINITY);
         log("stuurRadar(): search modus");
      }
   }

   /**
    * indien de variabele debugmode groter is dan nul, stuurt deze
    * functie de opgegeven tekst naar standaard uitvoer.
    *
    * @param s de te printen tekst-string.
    */
   public void log(String s) {
      if (debugmode) {
         out.println(s);
      }
   }

   public void initialiseerRobot() {
      log("initialiseerRobot()");
      setColors(Color.pink, Color.pink, Color.pink);

      setAdjustGunForRobotTurn(true);
      setAdjustRadarForGunTurn(true);

      double xMax = getBattleFieldWidth();
      double yMax = getBattleFieldHeight();
      
      beweging = new Hildisvin(this, xMax, yMax);
      kanon = new Mjolnir(this, xMax, yMax);

      // initialiseer Freya's gedrag:
      inlezenInitialisatieBestand();

      if (opponenten == null) {
         log("initialiseerRobot(): initialiseer Hashtable met opponenten.");
         opponenten = new Hashtable();
      }
      initialiseeropponenten();
   }

   /**
    * <p>Deze functie leest het initialisatiebestand voor deze robot ("Freya.ini") en
    * leest hieruit de gedrags-parameter "behaviour" en de logging-parameters "dataSaving"
    * en "logging".</p>
    * <p>De gedrags-parameter "behaviour" mag de volgende waarden hebben:
    * <ul>
    * <li>normal : beweegt en schiet (default-waarde).</li>
    * <li>reference: beweegt niet, schiet kogels met kracht 3.0.</li>
    * <li>challenge: beweegt, schiet niet.</li>
    * </ul></p>
    * <p>De logging-parameter "dataSaving" mag de volgende waarden hebben:
    * <ul>
    * <li>true : slaat statistiek van de tegenstander(s) op.</li>
    * <li>false: slaat geen statistiek op (default-waarde).</li>
    * </ul></p>
    * <p>De logging-parameter "logging" mag de volgende waarden hebben:
    * <ul>
    * <li>true : toont debug-teksten in standaard uitvoer.</li>
    * <li>false: toont geen debug-teksten (default-waarde).</li>
    * </ul></p>
    */
   private void inlezenInitialisatieBestand() {
      if (getRoundNum() == 0) {
         try {
            BufferedReader bufferedreader = new BufferedReader(new FileReader(getDataFile("Freya.ini")));
            String s;
            String gedrag = null;

            while ((s = bufferedreader.readLine()) != null) {
               s = s.trim();

               if (!s.substring(0, 1).equals("#")) {
                  if (s.length() > 11 && s.substring(0, 10).equalsIgnoreCase("behaviour=")) {
                     gedrag = s.substring(10).trim();
                  }

                  if (s.length() > 12 && s.substring(0, 11).equalsIgnoreCase("dataSaving=")) {
                     opslaanStatistiek = s.substring(11).trim().equalsIgnoreCase("false") ^ true;
                  }

                  if (s.length() > 9 && s.substring(0, 8).equalsIgnoreCase("logging=")) {
                     debugmode = s.substring(8).trim().equalsIgnoreCase("false") ^ true;
                  }
               }
            }

            if (gedrag != null) {
               beweging.targettingChallenge = gedrag.equalsIgnoreCase("reference") ? true : false;
               kanon.targettingChallenge = gedrag.equalsIgnoreCase("reference") ? true : false;
               kanon.movementChallenge = gedrag.equalsIgnoreCase("challenge") ? true : false;
            }
         } catch (IOException ioexception) {
            out.println("inlezenInitialisatieBestand(): IOException bij inlezen data Freya.ini: " + ioexception);
         }

         log("inlezenInitialisatieBestand(): gedrag = "
            + (kanon.targettingChallenge ? "targettingChallenge" : (kanon.movementChallenge ? "movementChallenge" : "normaal")));
         log("inlezenInitialisatieBestand(): doel-statistiek wordt " + (opslaanStatistiek ? "wel" : "niet")
            + " opgeslagen.");
      }
   }

   /**
    * Method die per tegenstander de opgebouwde statistiek wegschrijft naar standaard
    * uitvoer en naar een bestand.
    */
   private void opslaanDoelStatistiek() {
      if (opslaanStatistiek && (getRoundNum() == getNumRounds() - 1)) {
         for (Enumeration enm = opponenten.elements(); enm.hasMoreElements();) {
            Doel d = (Doel) enm.nextElement();

            try {
               PrintStream w = new PrintStream(new RobocodeFileOutputStream(getDataFile(d.naam + ".txt")));
               w.println(">>**************************************************<<");
               w.println("Statistiek bot " + d.naam);
               w.println(">>**************************************************<<");
               w.println("I fire bullet amount       : " + d.schotenFreya);
               w.println("My hit bullet amount       : " + d.treffersFreya);
               w.println("                 Hit ratio : " + 100.0d * d.treffersFreya / d.schotenFreya + "%");
               w.println("+-------------------------------------+");
               w.println("Enemy fire bullet amount   : " + d.schoten);
               w.println("Enemy  hit bullet amount   : " + d.treffers);
               w.println("                 Hit ratio : " + 100.0d * d.treffers / d.schoten + "%");
               w.println("+-------------------------------------+");
               w.println("Gemiddelde afstand         : " + d.gemAfstand);
               w.println("Gemiddelde snelheid        : " + d.gemSnelheid);
               w.println("Gemiddelde kogelkracht     : " + (d.gemKogelkracht / d.schoten));
               w.println("Gemiddelde kogelkrachtFreya: " + (d.gemKogelkrachtFreya / d.schotenFreya));
               w.println("+-------------------------------------+");

               log(">>**************************************************<<");
               log("Statistiek bot " + d.naam);
               log(">>**************************************************<<");
               log("I fire bullet amount       : " + d.schotenFreya);
               log("My hit bullet amount       : " + d.treffersFreya);
               log("                 Hit ratio : " + 100.0d * d.treffersFreya / d.schotenFreya + "%");
               log("+-------------------------------------+");
               log("Enemy fire bullet amount   : " + d.schoten);
               log("Enemy  hit bullet amount   : " + d.treffers);
               log("                 Hit ratio : " + 100.0d * d.treffers / d.schoten + "%");
               log("+-------------------------------------+");
               log("Gemiddelde afstand         : " + d.gemAfstand);
               log("Gemiddelde snelheid        : " + d.gemSnelheid);
               log("Gemiddelde kogelkracht     : " + (d.gemKogelkracht / d.schoten));
               log("Gemiddelde kogelkrachtFreya: " + (d.gemKogelkrachtFreya / d.schotenFreya));
               log("+-------------------------------------+");

               if (w.checkError()) {
                  log("Probleem bij wegschrijven statistiek!");
               }

               w.close();
               log(d.naam + "'s data opgeslagen");
            } catch (Exception e) {
               log("opslaanDoelStatistiek(): IOException bij opslaan data " + d.naam + " : " + e);
            }
         }
      }
   }
} // EINDE CLASS Freya
