package sch;

import robocode.*;
import java.util.*;

/**
 * Perform all enemy management related tasks
 */
 
public class EnemyManager implements Constants {
	
	private Map enemyList;
	private List targetList;
	private AdvancedRobot myBot;
	private Bot currentTarget=null;
	private List interceptMethods=null;
	
	/**
	 * Create the EnemyManager
	 * @param	myBot	the instance of my Bot
	 */
	public EnemyManager(AdvancedRobot myBot) {
		this.myBot=myBot;
		enemyList=new HashMap(myBot.getOthers());
		targetList=new ArrayList(myBot.getOthers());
		
		// Load all intercept methods
		interceptMethods=new LinkedList();		
		
		Intercept stillIntercept=new StillIntercept();
		interceptMethods.add(stillIntercept);
		
		Intercept approxIntercept=new Intercept();
		interceptMethods.add(approxIntercept);
		
		Intercept linearIntercept=new LinearIntercept();
		interceptMethods.add(linearIntercept);

		Intercept ciruclarIntercept=new CircularIntercept();
		interceptMethods.add(ciruclarIntercept);
		
		if (BotUtil.varContains(DEBUG,DEBUG_OBJECT_CREATION)) System.out.println("Enemy Manager initialized");
	}
	
	/**
	 * What to do in the first round of a battle to initialize the enemy manager<br>
	 * At the moment nothing. :-)
	 */
	public void initialize() {}

	/**
	 * What to do in all rounds of a battle different from the first one to reset the enemy manager<br>
	 */	
	public void reinitialize() {
		// Reset all robot enemy objects
		for (Iterator i=enemyList.values().iterator();i.hasNext();) {
			Bot currentBot=(Bot) i.next();
			currentBot.reinitialize();
		}		
	}
	
	/**
	 * Return the instance of the main Bot
	 * @return	main Bot
	 */
	public AdvancedRobot getMyBot() {
		return myBot;
	}

	/**
	 * Return a Map containing all scanned robot in the battle
	 * @return	list of robots
	 */
	public Map getEnemyList() {
		return enemyList;
	}

	/**
	 * Return a Map containing all intercept methods available
	 * @return	list of intercept methods
	 */
	public List getInterceptMethods() {
		return interceptMethods;
	}

	/**
	 * Return the instance of a specified bot
	 * @param	botName		name of the needed robot
	 * @return				the instance of the bot
	 */
	public Bot getBot(String botName) {
		return (Bot) enemyList.get(botName);
	}

	/**
	 * Return the current target robot
	 * @return		target robot
	 */
	public Bot getCurrentTarget() {return currentTarget;}
	
	/**
	 * Set the current target
	 * @param	botName		the name of the new target
	 */
	public void setCurrentTarget(String botName) {
		this.currentTarget=getBot(botName);
	}
	
	/**
	 * Perform all operation needed when a robot is scanned
	 * @param	scannedRobot	a ScannedRobotEvent
	 */
	public void onScannedRobot(ScannedRobotEvent scannedRobot) {
		
		//Retrieve informations from the scanned robot
		
		String botName=scannedRobot.getName();
		double heading_rad=scannedRobot.getHeadingRadians();
		double speed=scannedRobot.getVelocity();
		double distance=scannedRobot.getDistance();
		double energy=scannedRobot.getEnergy();
		double time=scannedRobot.getTime();
		
		//Retrieve myBot information used to calculate absPosition of the scanned Robot
		
		double bearing_rad=scannedRobot.getBearingRadians();
		double myX=myBot.getX();
		double myY=myBot.getY();
		double myHeading_rad=myBot.getHeadingRadians();
		
		//Calculate absolute bearing of the scanned bot
		double absbearing_rad=BotMath.normaliseHeading(bearing_rad+myHeading_rad);
		// Calculate x,y position relative to myBot pos
		Point enemyPos=BotMath.polar2Cartesian(distance,absbearing_rad);
		
		//Convert to battlefield coordinate
		enemyPos.add(myX,myY);
		
		
		Bot currBot=getBot(botName);
		//if bot is not in already scanned bot I create a new Bot and put into the enemyList
		if (currBot==null) {
			currBot=new Bot(botName, this);
			enemyList.put(botName, currBot);
			targetList.add(currBot);
		}
		
		//Update the bot with fresh informations
		currBot.update(heading_rad, speed, distance, enemyPos, energy, time);
		selectCurrentTarget();
	}

	/**
	 * Perform all operation needed when a robot dies
	 * @param	deathRobot	a ScannedRobotEvent
	 */
	public void onRobotDeath(String botName) {
		Bot currBot=(Bot) enemyList.get(botName);
		currBot.onDeath();
	}

	/**
	 * Prints the stats relative to all robots
	 */
	public void printStats() {
		//Print stats intestations
		System.out.println(" __________BOT NAME__________|_DIED_BEFORE_ME_|_____HIT_RATIO______|");
		for (Iterator i=enemyList.values().iterator();i.hasNext();) {
			Bot currentBot=(Bot) i.next();
			System.out.println("|"+BotUtil.makeFixedLength(currentBot.getName(),28,ALIGN_LEFT)+"|"+BotUtil.makeFixedLength(currentBot.getNumDiedBeforeMe()+"",16,ALIGN_CENTER)+"|"+BotUtil.makeFixedLength(currentBot.getNumHit()+"/"+currentBot.getNumShotAt()+" "+BotMath.round2(currentBot.getHitRatio())+"%",20,ALIGN_CENTER)+"|");
		}
	}

	/**
	 * Compute the stats of all robots
	 */	
	public void calculateStats() {
		for (Iterator i=enemyList.values().iterator();i.hasNext();) {
			// For each bot compute the stats
			Bot currentBot=(Bot) i.next();
			currentBot.calculateStats();
		}		
	}

	/**
	 * Save to disk all the bot statistics
	 */	
	public void saveStatsFiles() {
		for (Iterator i=enemyList.values().iterator();i.hasNext();) {
			// For each bot save to disk the stats
			Bot currentBot=(Bot) i.next();
			currentBot.writeToFile();
		}		
	}
	
	public void selectCurrentTarget() {
		Collections.sort(targetList);	
		this.currentTarget=(Bot) targetList.get(0);
		if(BotUtil.varContains(DEBUG,DEBUG_TARGET_SELECTION)) System.out.println("Target Selected : "+getCurrentTarget().getName() + "("+getCurrentTarget().getTargetingGoodness()+")");
	}
}
