package tzu.intel;

import tzu.gun.*;
import tzu.util.*;
import tzu.intel.*;
import tzu.strategy.*;
import robocode.*;

/**
 * Tracks bullets fired by an enemy robot.
 */
public class EnemyBulletManager extends AbstractManager {

    static final double DISTANCE_TOLERANCE = 50;
    static final int ARRAY_SIZE     = 500;
    EnemyBullet[] bullets           = new EnemyBullet[ARRAY_SIZE];
    int lastLiveBullet              = -1;
    double gameSpeed;
    double speed;
    double prevSpeed;
    double chgSpeed;
    double totSpeed;
    double avgSpeed;
    double prevChgSpeed;
    double heading;
    double prevHeading;
    double gameHeading;
    double prevGameHeading;
    double chgHeading;
    double prevChgHeading;
    double prevX;
    double prevY;
    long lastUpdate;
    int updateCount;


    /**
     * Create an enemy bullet manager object.
     * @param ar    Your AdvancedRobot.
     */
    public EnemyBulletManager(AdvancedRobot ar, TargetStrategyInterface tsi) {
        super(ar, tsi);
        initialize();
    }

    /**
     * Wipes out the enemy bullet array and reinitializes
     * fields that track your robot's movement.
     */
    public void reinitialize() {
        for (int i = 0; i < ARRAY_SIZE; i++) {
            bullets[i] = null;
        }
        lastLiveBullet = -1;
        initialize();
    }

    /**
     * Initialize fields that track your robot's movement.
     */
    public void initialize() {

        gameSpeed = myRobot.getVelocity();
        speed = BotMath.abs(gameSpeed);
        updateCount++;
        totSpeed += speed;
        avgSpeed = totSpeed / updateCount;
        prevSpeed = speed;
        chgSpeed = 0.0;
        prevChgSpeed = 0.0;
        gameHeading = myRobot.getHeading();
        prevGameHeading = gameHeading;
        heading = (gameSpeed < 0.0) ? (gameHeading + A_180) % A_360 : gameHeading;
        prevHeading = heading;
        chgHeading = 0.0;
        prevChgHeading = 0.0;
        prevX = myRobot.getX();
        prevY = myRobot.getY();
        lastUpdate = myRobot.getTime();

    }

    /**
     * Update bullet positions and fields that track your robot's movement.
     */
    public void takeTurn() {

        update();

        gameSpeed       = myRobot.getVelocity();
        prevSpeed       = speed;
        speed           = BotMath.abs(gameSpeed);

        updateCount++;
        totSpeed += speed;
        avgSpeed = totSpeed / updateCount;

        prevChgSpeed    = chgSpeed;
        chgSpeed        = (speed - prevSpeed) / (myRobot.getTime() - lastUpdate);
        if (chgSpeed == Double.NaN) chgSpeed = 0.0;
        prevGameHeading = gameHeading;
        gameHeading     = myRobot.getHeading();
        prevHeading     = heading;
        heading         = (gameSpeed < 0)
                        ? (gameHeading + A_180) % A_360
                        : gameHeading;
        prevChgHeading  = chgHeading;
        chgHeading      = BotMath.plusMinus180(
                          prevGameHeading, gameHeading) /
                          (myRobot.getTime() - lastUpdate);

        if (chgHeading == Double.NaN) chgHeading = 0;
        prevX           = myRobot.getX();
        prevY           = myRobot.getY();

        lastUpdate = myRobot.getTime();
    }


    /**
     * Update bullet positions.  If a bullet is out of bounds, remove it.
     */
    public void update() {

        EnemyBullet bullet;
        int lastLive = -1;
        int count = 0;
        double yourX = myRobot.getX();
        double yourY = myRobot.getY();

        for (int i = 0; i <= lastLiveBullet; i++) {
            bullet = bullets[i];

            if (bullet != null) {

                if (bullet.isNotAThreat(yourX, yourY)) {
                    bullets[i] = null;

                } else {
                    lastLive = i;
                    count++;
                }
            }
        }
        lastLiveBullet = lastLive;
    }


    /**
     * When one of our bullets hits one of theirs, remove their bullet(s)
     * from the EnemyBulletManager
     * @param bullet    the opponent's bullet that was hit by our bullet.
     */
    public void removeBullet(Bullet bullet) {

        String  name    = bullet.getName();
        double  energy  = bullet.getPower();
        double  x       = bullet.getX();
        double  y       = bullet.getY();

        EnemyBullet eb;
        long        creationTime = 0;
        int         i;

        for (i = 0; i <= lastLiveBullet; i++) {

            eb = bullets[i];
            if (eb != null) {
                if (eb.getName().equals(name)   &&
                    eb.getEnergy() == energy    &&
                    BotMath.calcDistance(
                    eb.getX(), eb.getY(), x, y) <
                    DISTANCE_TOLERANCE)         {

                    creationTime = eb.getCreationTime();
                    break;
                }
            }
        }

        if (creationTime != 0) {
            for (i = 0; i <= lastLiveBullet; i++) {

                eb = bullets[i];
                if (eb != null) {

                    if (eb.getCreationTime() == creationTime &&
                        eb.getName().equals(name) &&
                        eb.getEnergy() == energy) {

                        bullets[i] = null;
                    }
                }
            }
        }
    }


    /**
     * Add 1 or 3 bullets to the array.
     * @param shooter       the enemy robot shooting at us.
     * @param firePower     the firepower used to shoot at us.
     */
    public void addBullets(Bot shooter, double firePower) {

        int enemyCount = myRobot.getOthers();



        if (enemyCount < 4 ||
                myRobot.getTime() - shooter.getLastShotMeTime() < 50 ||
                shooter.getDistance() < BattleField.getDiagonal() / 2) {

            addLinearBullet(shooter, firePower);
        }

        if ((prevSpeed != 0.0 || speed != 0.0) && myRobot.getOthers() < 3) {
            addEstimatedBullet(shooter, firePower);
            addSpeedAveragedBullet(shooter, firePower);
        }
    }

    /**
     * Estimate where the enemy robot is aiming to hit us and add a bullet
     * that reflects that estimation.
     * @param shooter       the enemy robot shooting at us.
     * @param firePower     the firepower used to shoot at us.
     */
    public void addEstimatedBullet(Bot shooter, double firePower) {

        Motion myMotion = new Motion(
            prevX, prevY,
            prevHeading, prevSpeed,
            prevChgHeading, prevChgSpeed);

        createAndAddBullet(shooter, firePower,
                myMotion, FireSolution.AIM_ESTIMATED);
    }

    /**
     * Same as addEstimateBullet, except uses average velocity to aim,
     * and no acceleration or turning.
     * *
     * @param shooter       the enemy shooting at us.
     * @param firePower     the firepower used to shoot at us.
     */
    public void addSpeedAveragedBullet(Bot shooter, double firePower) {

        Motion myMotion = new Motion(
            prevX, prevY,
            prevHeading, avgSpeed,
            0, 0);

        createAndAddBullet(shooter, firePower,
                myMotion, FireSolution.AIM_AVERAGE);
    }


    private void createAndAddBullet(
            Bot shooter,
            double firePower,
            Motion myMotion,
            int method) {

        Motion bulletMotion = new Motion(
            shooter.getPrevX(),
            shooter.getPrevY(),
            0, firePower);

        FireSolution fs = EstimateAim.calculate(myMotion, bulletMotion);

        for (long i = shooter.getLastUpdate();
                i <= myRobot.getTime() - 1; i++) {

            EnemyBullet b = new EnemyBullet(
                    shooter.getName(), shooter.getPrevX(), shooter.getPrevY(),
                    fs.getExpectedHitX(), fs.getExpectedHitY(),
                    firePower, fs.gunAngle, myRobot, method,
                    i);

            addBullet(b);
        }
    }



    /**
     * Add a bullet assuming that the enemy robot is using linear
     * aiming (ie: no leading the target).
     * @param shooter       the enemy robot shooting at us.
     * @param firePower     the firepower used to shoot at us.
     */
    public void addLinearBullet(Bot shooter, double firePower) {

        double x = shooter.getPrevX();
        double y = shooter.getPrevY();

        for (long i = shooter.getLastUpdate();
                i <= myRobot.getTime() - 1; i++) {

            EnemyBullet b = new EnemyBullet(
                    shooter.getName(),
                    x, y, prevX, prevY,
                    firePower,
                    BotMath.calcHeading(x, y, prevX, prevY),
                    myRobot,
                    FireSolution.AIM_LINEAR,
                    i);

            addBullet(b);
        }
    }


    /**
     * Add a bullet to the array.
     * @param bullet
     */
    public void addBullet(EnemyBullet bullet) {

        int i;

        for (i = 0; i <= lastLiveBullet; i++) {

            if (bullets[i] == null) {
                bullets[i] = bullet;
                return;
            }
        }
        lastLiveBullet = i;
        bullets[lastLiveBullet] = bullet;
    }

    /**
     * Calculate the anti-gravity effect enemy bullets have on our robot.
     * @return the relative point the anti-gravity effect is pushing us towards.
     */
    public Point calcAntiGravity() {

        Point p = new Point(0,0);

        double yourX = myRobot.getX();
        double yourY = myRobot.getY();


        for (int i = 0; i < ARRAY_SIZE; i++) {
            if (bullets[i] != null) {
                p.add(bullets[i].calcAntiGravity(yourX, yourY));
            }
        }
        return p;
    }
}
