package tzu.intel;

import robocode.*;
import java.io.*;
import java.util.zip.*;
import tzu.util.*;

/**
 * Stores previous reactions by the enemy when you fired at it.
 *
 * NOTE: CHANGES TO THIS CLASS MAY IMPACT THE Bot CLASS, ESPECIALLY
 * METHODS PERTAINING TO PERSISTENCE.
 */
public final class PreviousReactions implements Constants {

    public static final int H_DIMENSION  = 24;  // Heading relative to bearing.
    public static final int S_DIMENSION  = 9;   // Speed from 0 to 8.
    public static final int A_DIMENSION  = 3;   // Deaccel/accel indicator.
    public static final int D_DIMENSION  = 9;   // Distance from target.
    public static final int FILE_VERSION = 7;

    /** The minimum number of bytes required to store this object. */
    public  static final int REACTION_FILE_SIZE =
                             H_DIMENSION *
                             S_DIMENSION *
                             A_DIMENSION *
                             D_DIMENSION *
                             2 +
                             4;

    private Reaction[][][][] pr;


    /** Create a new array to store previous reactions. */
    public PreviousReactions() {
        pr = new Reaction[H_DIMENSION][S_DIMENSION][A_DIMENSION][D_DIMENSION];
    }


    /**
     * Store a reaction by the enemy to being fired at.
     */
    public void update(double heading,
                       double bearing,
                       double speed,
                       double acceleration,
                       double distance,
                       Reaction r) {
        int h;
        int s;
        int a;
        int d;


        heading = heading - bearing;
        if (heading < 0) heading += A_360;

        h = (int)Math.round(heading / (A_360 / H_DIMENSION));
        if (h >= H_DIMENSION) h = 0;

        s = (int)Math.round(speed);

        if (acceleration < 0)       { a = 0; }
        else if (acceleration == 0) { a = 1; }
        else                        { a = 2; }

        d = (int)Math.round(distance / 50.0);
        if (d > (D_DIMENSION - 1)) d = D_DIMENSION - 1;
        pr[h][s][a][d] = r;
    }


    /**
     * Return a previous reaction by the enemy.
     * @return previous reaction; null if none recorded.
     */
    public Reaction get(double heading,
                        double bearing,
                        double speed,
                        double acceleration,
                        double distance) {
        int h;
        int s;
        int a;
        int d;

        double head = heading;
        double bear = bearing;

        heading = heading - bearing;
        if (heading < 0) heading += A_360;

        h = (int)Math.round(heading / (A_360 / H_DIMENSION));
        if (h >= H_DIMENSION) h = 0;

        s = (int)Math.round(speed);

        if (acceleration < 0)       { a = 0; }
        else if (acceleration == 0) { a = 1; }
        else                        { a = 2; }

        d = (int)Math.round(distance / 50.0);
        if (d > (D_DIMENSION - 1)) d = D_DIMENSION - 1;
        return pr[h][s][a][d];
    }


    /**
     * Returns a byte array representation of this class
     * for storing to file.
     */
    public byte[] getData() {

        int h;
        int s;
        int a;
        int d;
        int index;
        int count = 0;
        Reaction r;

        byte[] byteArray = new byte[
                H_DIMENSION * S_DIMENSION * A_DIMENSION * D_DIMENSION * 2];

        index = 0;
        for (h = 0; h < H_DIMENSION; h++) {
            for (s = 0; s < S_DIMENSION; s++) {
                for (a = 0; a < A_DIMENSION; a++) {
                    for (d = 0; d < D_DIMENSION; d++) {

                        r = pr[h][s][a][d];

                        if (r == null) {
                            byteArray[index] = -1;
                            index++;
                            byteArray[index] = -1;
                            index++;
                        } else {
                            byteArray[index] = r.getByteRelativeHeading();
                            index++;
                            byteArray[index] = r.getByteSpeed();
                            index++;
                            count++;
                        }
                    }
                }
            }
        }
        return byteArray;
    }


    /**
     * Initializes this class from a byte array representation
     * of the class (produced by getData).
     */
    public void setData(byte[] byteArray) {


        if (pr == null) {
            pr = new Reaction[H_DIMENSION][S_DIMENSION][A_DIMENSION][D_DIMENSION];
        }

        int h;
        int s;
        int a;
        int d;
        int index;
        int count = 0;
        Reaction r;
        byte heading;
        byte speed;

        index = 0;
        for (h = 0; h < H_DIMENSION; h++) {
            for (s = 0; s < S_DIMENSION; s++) {
                for (a = 0; a < A_DIMENSION; a++) {
                    for (d = 0; d < D_DIMENSION; d++) {

                        heading = byteArray[index];
                        index++;
                        speed   = byteArray[index];
                        index++;

                        if (heading == -1 && speed == -1) {
                            pr[h][s][a][d] = null;
                        } else {
                            r = new Reaction(heading, speed);
                            pr[h][s][a][d] = r;
                            count++;
                        }
                    }
                }
            }
        }
    }


    /**
     * Restore this object's state from data stored in file.
     */
    public void readObject(DataInputStream stream) throws IOException {

        byte[] byteArray = new byte[
                H_DIMENSION * S_DIMENSION * A_DIMENSION * D_DIMENSION * 2];

        if (stream.readInt() == FILE_VERSION) {
            stream.readFully(byteArray);
            setData(byteArray);
        }
    }


    /**
     * Store this object to file.
     */
    public void writeObject(DataOutputStream stream) throws IOException {

        stream.writeInt(FILE_VERSION);
        stream.write(getData());
    }
}
