package tzu.radar;

import tzu.util.*;
import tzu.intel.*;
import tzu.strategy.*;
import robocode.*;

/**
 * Manages the radar for an AdvancedRobot.
 */
public class RadarManager extends AbstractManager {


    int         direction       = FORWARD;
    String      scannedBots     = null;
    int         scanCount       = 0;
    Bot         target          = null;



    /**
     * Create a new RadarManager object.
     * @param ar    your AdvancedRobot.
     * @param tsi   object implementing the TargetStrategyInterface.
     */
    public RadarManager(AdvancedRobot ar, TargetStrategyInterface tsi) {
        super(ar, tsi);
        reinitialize();
    }


    /** Reinitialization to do in second and subsequent round. */
    public void reinitialize() {
        scannedBots = null;
        scanCount = 0;
    }


    /*
     * Get the currently selected target and set the radar to
     * scan just the target (one-on-one) or many targets.
     */
    public void takeTurn() {

        target = targets.getTarget();

        if (myRobot.getOthers() > 1) {
            scanMultipleTargets();
        } else {
            scanSingleTarget();
        }
    }


    /** Scan a battle field with multiple opponents. */
    void scanMultipleTargets() {
        if (myRobot.getRadarTurnRemaining() == 0.0) {
            scan(A_360 * direction);
        }
    }


    /**
     * Add each newly scanned bot to a String listing bots that have
     * been scanned.  When all bots have been scanned, reset the string
     * and scan in the opposite direction.
     */
    public void onScannedRobot(ScannedRobotEvent e) {

        String newBot = e.getName();

        if (scannedBots == null) {
            scannedBots = newBot;
            scanCount = 1;
        } else {
            if (scannedBots.indexOf(newBot) == -1) {
                scanCount++;
                scannedBots += newBot;
            }
            if (scanCount >= myRobot.getOthers()) {
                scannedBots = newBot;
                scanCount = 1;
                direction *= -1;
                scan(A_360 * direction);
            }
        }
    }


    /** Scan the single enemy bot. */
    void scanSingleTarget() {

        double angle    = 0.0;
        double heading  = myRobot.getRadarHeading() % A_360;

        /* There is only one bot, but we don't know where it is. */
        if (target == null) {

            scan(A_360);

        } else {

            /*
             * There is only one bot, but we haven't seen it
             * in a while or the target we have been given is
             * already dead.
             */
            if (target.getLastUpdate() == 0 ||
                target.getLastUpdate()      +
                MAX_SCAN_AGE                <
                myRobot.getTime()           ||
                target.getEnergy() == -1)   {

                scan(A_360);

            /*
             * We have a valid target and the data on it
             * is good.  Scan back and forth in the target's
             * direction.
             */
            } else {

                angle = BotMath.plusMinus180(
                        heading,
                        target.getBearing());

                angle += angle > 0 ? +HALF_SCAN_ARC : -HALF_SCAN_ARC;
                scan(angle);
            }
        }
    }


    /** Turn the radar the specified number of degrees. */
    void scan(double degrees) {
        myRobot.setTurnRadarRight(degrees);
    }

}
