package tzu.strategy;

import tzu.intel.*;
import robocode.*;
import tzu.util.*;
import tzu.intel.*;

/**
 * Provides a target choosing strategy for an AdvancedRobot.
 */
public class TargetStrategy extends AbstractStrategy
    implements TargetStrategyInterface{

    Bot     target;
    boolean isTeamPlay;

    /**
     * Create a target strategy object.
     * @param ar    your AdvancedRobot.
     * @param emi   object implementing the EnemyManagerInterface.
     */
    public TargetStrategy(AdvancedRobot ar, EnemyManagerInterface emi) {
        super(ar, emi);

        if (myRobot instanceof TeamRobot) {
            isTeamPlay = true;
        } else {
            isTeamPlay = false;
        }
    }

    /**
     * Pick the best target each turn.
     */
    public void takeTurn() {
        Bot prevTarget = target;
        target = pickTarget();
    }

    /**
     * Reinitialization for start of second and subsequent rounds of battle.
     */
    public void reinitialize() {
        target = null;
    }

    /**
     * Pick the best target for an individual or team.
     * @return enemy robot to shoot at.
     */
    public Bot pickTarget() {

        if (isTeamPlay) {
            target = pickTeamTarget(target);
        } else {
            target = pickIndividualTarget(target);
        }
        return target;
    }

    /**
     * Retrieves an enemy robot using its name.
     * @param name  name of enemy robot.
     * @return      Bot or null if not found.
     */
    public Bot get(String name) {
        return null;
    }

    /**
     * Returns the currently selected target.
     */
    public Bot getTarget() {
        return target;
    }

    /**
     * Sets the current target.
     */
    public void setTarget(Bot b) {
        target = b;
    }

    /** Evaluate all enemy bots and return the best target to shoot at. */
    private Bot pickIndividualTarget(Bot prevTarget) {

        Bot     prevBot             = null;
        Bot     currBot             = null;
        double  prevDifficulty      = Double.MAX_VALUE;
        double  currDifficulty      = 0.0;
        int     knownCount          = enemyManager.size();
        double  distance;
        long    now                 = myRobot.getTime();

        double  gunTurn             = 0.0;
        double  angleOfAttack       = 0.0;
        double  speedRatio          = 0.0;
        double  distanceRatio       = 0.0;
        double  angleRatio          = 0.0;
        double  gunTurnRatio        = 0.0;

        /*
         * For each bot alive, calculate the difficulty of shooting
         * that bot as a number between 0 and 1 (1 = most difficult).
         * Return the bot with the lowest score.
         */
        for (int i = 0; i < knownCount; i++) {

            currBot = enemyManager.get(i);

            if (currBot.isTeammate() == false               &&
                currBot.getEnergy() > -1                    &&
                currBot.getDistance() < SHOOT_MAX_DISTANCE  &&
                currBot.getLastUpdate() > 0                 &&
                currBot.getLastUpdate() + MAX_SCAN_AGE      >=
                myRobot.getTime()) {

                /** Calculate the angle of attack. */
                angleOfAttack =
                        BotMath.abs(
                        BotMath.plusMinus180(
                        currBot.getBearing(),
                        currBot.getHeading()));
                if (angleOfAttack > 90) {
                    angleOfAttack = 180 - angleOfAttack;
                }


                /** Calculate the amount of gun turn (without leading). */
                gunTurn = BotMath.abs(BotMath.plusMinus180(
                    myRobot.getGunHeading(), currBot.getBearing()));


                /*
                 * Calculate the ratios (ie: the target's speed as
                 * a ratio of the maximum speed possible; the distance
                 * as a ratio of the maximum distance possible, etc.
                 * A higher ratio indicates an easier target to hit.
                 */
                distance      = currBot.estimateDistanceAt(now);
                speedRatio    = currBot.getSpeed() / MAX_SPEED;
                distanceRatio = distance / BattleField.getDiagonal();
                angleRatio    = angleOfAttack / 90;
                gunTurnRatio  = gunTurn / 180;


                /*
                 * Average the ratios. Put more weight on the
                 * distance ratio than angle and gun turn.
                 */
                currDifficulty = (speedRatio           +
                                 (distanceRatio * 7)   +
                                  angleRatio           +
                                  gunTurnRatio)        / 10;


                /*
                 * If this type of bot is relatively easy to hit,
                 * or harder to hit, adjust the difficulty level to
                 * reflect that.
                 */
                if (currBot.getAvgSpeed() < 2) {
                    currDifficulty *= 0.25;
                }

                /* Favor the previous target over new targets. */
                if (currBot == prevTarget) {
                    currDifficulty *= 0.90;
                }

                /* Favor really close targets. */
                if (distance < HOP) {
                    currDifficulty *= .25;
                } else if (distance < JUMP) {
                    currDifficulty *= .75;
                }

                if (currDifficulty < prevDifficulty) {
                    prevDifficulty = currDifficulty;
                    prevBot = currBot;
                }
            }
        }
        return prevBot;
    }


    /**
     * Pick a target for our team.  Target preference is:
     * 1) Leaders, 2) TeamRobots, 3) Droids and non-team robots.
     */
    private Bot pickTeamTarget(Bot prevTarget) {

        Bot currBot = null;
        Bot prevBot = null;
        int type;
        int knownCount = enemyManager.size();

        for (int i = 0; i < knownCount; i++) {
            currBot = enemyManager.get(i);

            if (currBot.isTeammate() == false) {
                if (currBot.getEnergy() > -1) {
                    type = currBot.getTeamType();
                    if (type == LEADER) {
                        return currBot;
                    } else if (type == TEAMROBOT) {
                        prevBot = currBot;
                    } else if (prevBot == null) {
                        prevBot = currBot;
                    }
                }
            }
        }
        return prevBot;
    }
}
