package chase.s2.stat;

import org.csdgn.utils.VDA;

/**
 * This is a visit count stat buffer. However, it is such a buffer where it may
 * have any number of segmentation or dimensionality.
 * 
 * For those uninformed, this is part of a statistical data mining system.
 * 
 * @author Chase
 */
public class StatBuffer {
	protected static final int scanWidth = 1;

	/** Handing the VDA manually is dangerous, and could wipe out all your data */
	private VDA<float[]> _binSet;
	public int _binTotal;
	public int[] _dataIndex;
	/** This is fine, since java can handle jagged arrays */
	public float[][] _splits;
	
	public final BinUpdater updater;

	/**
	 * Constructs a new buffer
	 * 
	 * @param splits
	 *            The data defining the data splits, usually a jagged array
	 * @param dataIndex
	 *            the index of the raw stats each split is on
	 * @param binTotal
	 *            total number of indexes for this buffer
	 */
	public StatBuffer(float[][] splits, int[] dataIndex, int binTotal, BinUpdater updater) {
		/* TODO: splits and dataIndex lengths don't equal throw exception */
		_binTotal = binTotal;
		_dataIndex = dataIndex;
		_splits = splits;

		int[] dimensions = new int[Math.max(1, splits.length)];
		dimensions[0] = 1;
		for (int i = 0; i < splits.length; ++i)
			dimensions[i] = splits[i].length + 1;

		_binSet = new VDA<float[]>(dimensions);

		Object[] obj = _binSet.getBackingArray();
		for (int i = 0; i < obj.length; ++i) {
			float[] bins = new float[binTotal];
			for (int j = 0; j < binTotal; ++j)
				bins[j] = 0;
			obj[i] = bins;
		}
		
		if (updater == null) {
			this.updater = new BinUpdater.DefaultUpdater();
		} else {
			this.updater = updater;
		}
	}

	/**
	 * Constructs a new stat buffer, using the lazy SplitSet
	 * 
	 * @param ss
	 *            SplitSet to use
	 * @param binTotal
	 *            total number of indexes for this buffer
	 */
	public StatBuffer(SplitSet ss, int binTotal, BinUpdater updater) {
		this(ss.getSplitSet(), ss.getIndexSet(), binTotal, updater);
	}

	/**
	 * This creates a flat, single dimension stat buffer.
	 * 
	 * @param binTotal
	 *            total number of indexes for this buffer
	 */
	public StatBuffer(int binTotal, BinUpdater updater) {
		this(new float[0][0], new int[0], binTotal, updater);
	}

	/**
	 * This method splits the raw data as defined by the data index and split
	 * information. It outputs a data array which can be fed to the VDA to get
	 * the desired bin set.
	 * 
	 * @param data
	 *            raw data
	 * @return data indexes
	 */
	public int[] convertData(StatData data) {
		int[] indexes = new int[Math.max(1, _dataIndex.length)];
		if (_splits.length == 0)
			return indexes;
		for (int i = 0; i < indexes.length; ++i) {
			int index = 0;
			for (; index < _splits[i].length; ++index) {
				if (data._data[_dataIndex[i]] < _splits[i][index])
					break;
			}
			indexes[i] = index;
		}
		return indexes;
	}

	/**
	 * Gets the individual stat buffer for the give data.
	 */
	public float[] getStatBin(StatData data) {
		return _binSet.get(convertData(data));
	}

	/**
	 * Updates this stat buffer with the given raw data
	 */
	public void update(StatData data, float weight) {
		float[] bins = getStatBin(data);
		double lindex = GFToIndex(data._lgf, _binTotal);
		double uindex = GFToIndex(data._ugf, _binTotal);
		updater.update(bins, lindex, uindex, weight);
	}

	/**
	 * Gets the best bin from the raw data
	 */
	public int getBestIndex(StatData data) {
		float[] bins = getStatBin(data);
		return getBestIndex(bins, scanWidth);
	}

	/**
	 * Gets the best guessfactor from the raw data
	 */
	public double getBestGF(StatData data) {
		return indexToGF(getBestIndex(data), _binTotal);
	}

	/**
	 * Static method used to find the best index. Width is number of extra bins
	 * to each side to test.
	 */
	public static final int getBestIndex(float[] bin, int width) {
		float[] score = getBinScores(bin, width);
		int bestIndex = 0; /* TODO: This could be problematic */
		for (int i = 0; i < bin.length; ++i)
			if (score[i] > score[bestIndex])
				bestIndex = i;
		return bestIndex;
	}

	/**
	 * Gets the scores for each bin in the given stat bin.
	 */
	protected static final float[] getBinScores(float[] bin, int width) {
		float[] scores = new float[bin.length];
		for (int i = 0; i < bin.length; ++i) {
			float score = 0;
			// It is a little complicated in there, but things will
			// never go out of bounds
			for (int j = -width; j < width + 1; ++j) {
				int abs = Math.abs(j) + 1;
				if (i + j < 0 || i + j >= bin.length) {
					if (i == 0 || i == bin.length - 1) {
						score += bin[i - j] / abs;
					} else {
						int offset = (j < 0) ? -3 : 3;
						score += bin[i - j + offset] / abs;
					}
				} else {
					score += bin[i + j] / abs;
				}
			}
			scores[i] = score;
		}
		return scores;
	}

	/**
	 * Converts a bin index to a guessfactor based on total number of bins.
	 */
	public static final double indexToGF(int index, int bins) {
		index = Math.min(Math.max(0, index), bins - 1);
		double half = (bins - 1.0) / 2.0;
		return (index / half) - 1.0;
	}

	/**
	 * Converts a guessfactor to a bin index, based on number of bins. This
	 * returns a double, incase there is any need to know how much it over hangs
	 * each bin. Cast to an int for the raw information.
	 */
	public static final double GFToIndex(double gf, int bins) {
		gf = Math.min(Math.max(-1.0, gf), 1.0) + 1.0;
		double half = (bins - 1.0) / 2.0;
		return gf * half;
	}
}