package nat.move;

import java.awt.*;
import java.awt.geom.*;
import java.util.*;
import robocode.*;
import robocode.util.*;
import chase.s2.stat.*;

/**
 * BHMove -- movement of BlackHole
 * 
 * @author Nat Pavasant
 * @license zlib
 */
public final class BHMove {
	private final AdvancedRobot robot;

	private static int BINS = MovementBuffer.MOVEMENT_BIN;
	private static int MIDDLE_BIN = BINS / 2;
	private Point2D.Double me;
	private Point2D.Double enemy;

	private int lastDirection = 1;
	private double lastVelocity = 0d;
	private int noWaveDirection = 1;

	// Stats
	private static StatBufferSet movementStats = MovementBuffer.getPrimaryMovementBuffer();
	private static StatBufferSet flattenerStats = MovementBuffer.getSecondaryMovementBuffer();
	private WeakHashMap<EnemyWave, StatData> dataMap = new WeakHashMap<EnemyWave, StatData>();
	private WeakHashMap<EnemyWave, float[]> statCache = new WeakHashMap<EnemyWave, float[]>();

	// Dynamic Factor
	private static double desiredDistance = 550;
	private static int bulletHit = 0, bulletFired = 0;
	private static double energyHit = 0, energyFired = 0;
	private static boolean flattener = false;
	private static boolean isRammer = false;
	private static double approchingVelocity = 0.0d;
	private static double count = 0d;

	// Surfing
	private ArrayList<EnemyWave> waves;
	private ArrayList<Integer> directionsHistory;
	private ArrayList<Double> bearingHistory;
	private ArrayList<double[]> dataHistory;
	private ArrayList<Point2D.Double> locationHistory;

	private static double enemyEnergy = 100.0;
	private static double distanceDangerFactor = 1.5;
	private static double turnLimit = .35d;

	// Wall Smoothing
	private RoundRectangle2D.Double movableField;
	private RoundRectangle2D.Double movableFieldInnerBound;
	private double WALL_STICK = 160;
	private double bfWidth, bfHeight;

	// Painting
	private ArrayList<Point2D.Double> paintingPredictedPositions = new ArrayList<Point2D.Double>();

	public BHMove(AdvancedRobot r) {
		robot = r;

		// Set the battlefield
		bfWidth = robot.getBattleFieldWidth();
		bfHeight = robot.getBattleFieldHeight();
		
		if (movableField == null) {
			movableField = new RoundRectangle2D.Double(20, 20, bfWidth - 40,
					bfHeight - 40, 100, 100);
			movableFieldInnerBound = new RoundRectangle2D.Double(25, 25,
					bfWidth - 50, bfHeight - 50, 90, 90);

		}
		waves = new ArrayList<EnemyWave>();
		directionsHistory = new ArrayList<Integer>();
		bearingHistory = new ArrayList<Double>();
		dataHistory = new ArrayList<double[]>();
		locationHistory = new ArrayList<Point2D.Double>();

		dataHistory.add(new double[] { 0, 0, 0, 0, 0 });
	}

	public final void onScannedRobot(ScannedRobotEvent e) {
		
		me = new Point2D.Double(robot.getX(), robot.getY());

		double lateralVelocity = robot.getVelocity()
				* Math.sin(e.getBearingRadians());
		double absBearing = e.getBearingRadians() + robot.getHeadingRadians();

		int direction = (int) Math.signum(lateralVelocity);
		if (direction == 0)
			direction = lastDirection;
		lastDirection = direction;

		directionsHistory.add(0, new Integer(direction));
		bearingHistory.add(0, new Double(absBearing + Math.PI));
		locationHistory.add(0, me);

		double bulletPower = enemyEnergy - e.getEnergy();
		if (bulletPower < 3.01 && bulletPower > 0.09
				&& directionsHistory.size() > 2) {
			EnemyWave ew = new EnemyWave();
			ew.fireTime = robot.getTime() - 1;
			ew.bulletVelocity = bulletVelocity(bulletPower);
			ew.distanceTraveled = bulletVelocity(bulletPower);
			ew.direction = directionsHistory.get(2).intValue();
			ew.directAngle = bearingHistory.get(2).doubleValue();
			ew.fireLocation = (Point2D.Double) enemy.clone();

			waves.add(ew);

			bulletFired++;
			energyFired += bulletPower;

			dataMap.put(ew, new StatData(getDataSet(e, direction, bulletPower),
					0, 0));
		}

		enemyEnergy = e.getEnergy();

		// update after EnemyWave detection, because that needs the previous
		// enemy location as the source of the wave
		enemy = project(me, absBearing, e.getDistance());
		
		// Rammer detection
		double appVel = e.getVelocity() * -Math.cos(absBearing - e.getHeadingRadians());
		approchingVelocity = (approchingVelocity*count + appVel) / (count + 1d);
		count++;
		
		if (approchingVelocity > 4 && !isRammer) {
			robot.out.println("Rammer detected -- robot is moving away at all cost!");
			isRammer = true;
			desiredDistance = 1000d;
			WALL_STICK = 140;
			distanceDangerFactor = 15d;
			turnLimit = 0.8d;
		}

		updateWaves();
		doSurfing();
	}

	public final void onBulletHit(BulletHitEvent e) {
		enemyEnergy -= Rules.getBulletDamage(e.getBullet().getPower());
	}

	public final void onHitRobot(HitRobotEvent e) {
		enemyEnergy -= 0.6;
	}

	public final void onHitByBullet(HitByBulletEvent e) {
		enemyEnergy += e.getBullet().getPower() * 3;
		logBulletHit(e.getBullet(), true);
	}

	public final void onBulletHitBullet(BulletHitBulletEvent e) {
		logBulletHit(e.getHitBullet(), false);
		robot.out.println(robot.getTime() + " BulletHitBullet");
	}

	private final double[] getDataSet(ScannedRobotEvent e, int direction,
			double bulletPower) {
		double[] oldData = dataHistory.get(0);

		int BFT = (int) Math.ceil(e.getDistance() / (20 - 3 * bulletPower));
		double MEA = Math.asin(8 / (20 - 3 * bulletPower));

		// Now we record all my robot information for surfing clustering
		// BFT, vel, latvel, advvel, accl, tsvc, tsdc, tss, tsdecel
		// walldistf, walldistr, dl10, dl20, dl30
		double[] data = new double[14];
		data[0] = BFT;
		data[1] = Math.abs(robot.getVelocity());
		data[2] = Math.abs(robot.getVelocity()
				* Math.sin(e.getBearingRadians()));
		data[3] = robot.getVelocity() * Math.cos(e.getBearingRadians());
		data[4] = robot.getVelocity() - lastVelocity;
		data[5] = (data[1] == oldData[0] ? oldData[1] + 1 : 0) / BFT;
		data[6] = (direction == lastDirection ? oldData[2] + 1 : 0) / BFT;
		data[7] = (data[1] != 0d ? oldData[3] + 1 : 0) / BFT;
		data[8] = (data[4] >= 0 ? oldData[4] + 1 : 0) / BFT;
		data[9] = wallDistance(me, e.getDistance(), absoluteBearing(enemy, me), 1)  / MEA;
		data[10] = wallDistance(me, e.getDistance(), absoluteBearing(enemy, me), -1) / MEA;
		data[11] = getDistanceInLastNTick(10) * 1.25d;
		data[12] = getDistanceInLastNTick(20) * 0.625d;
		data[13] = getDistanceInLastNTick(30) * 0.416d;

		// Stored Data Format:
		// Velocity, TSVC, TSDC, TSS, TSDECEL

		dataHistory.add(new double[] { data[1], data[5], data[6], data[7],
				data[8] });

		lastVelocity = robot.getVelocity();

		return data;
	}

	private final void logBulletHit(Bullet b, boolean realHit) {
		Point2D.Double hitBulletLocation = new Point2D.Double(b.getX(), b
				.getY());
		EnemyWave hitWave = findHitWave(hitBulletLocation, b.getVelocity());

		if (realHit) {
			bulletHit++;
			energyHit += b.getPower();
		}

		if (hitWave != null) {
			logHit(hitWave, hitBulletLocation);
			waves.remove(hitWave);
		} else {
			robot.out.println(robot.getTime()
					+ " Error! Bullet detected on non-existant wave!");
		}
	}

	private final EnemyWave findHitWave(Point2D.Double location, double velocity) {
		for (EnemyWave ew : waves) {
			if (Math.abs(ew.distanceTraveled
					- location.distance(ew.fireLocation)) < 50
					&& Utils.isNear(velocity, ew.bulletVelocity)) {
				return ew;
			}
		}
		return null;
	}

	private final void updateWaves() {
		for (int i = 0; i < waves.size(); i++) {
			EnemyWave ew = waves.get(i);

			ew.distanceTraveled = (robot.getTime() - ew.fireTime)
					* ew.bulletVelocity;
			if (ew.distanceTraveled > me.distance(ew.fireLocation)) {
				logVisit(ew, me);
				waves.remove(i);
				i--;
			}
		}
	}

	private final EnemyWave getFirstImpactSurfableWave() {
		double minTime = Double.POSITIVE_INFINITY;
		EnemyWave surfWave = null;

		for (int x = 0; x < waves.size(); x++) {
			EnemyWave ew = waves.get(x);
			double distance = me.distance(ew.fireLocation)
					- ew.distanceTraveled;
			double impactTime = distance / ew.bulletVelocity;
			if (distance > ew.bulletVelocity && impactTime < minTime) {
				surfWave = ew;
				minTime = impactTime;
			}
		}

		return surfWave;
	}

	private final static double getGFFromAbsoluteAngle(EnemyWave ew,
			double angle) {
		double offsetAngle = (angle - ew.directAngle);
		return Utils.normalRelativeAngle(offsetAngle)
				/ maxEscapeAngle(ew.bulletVelocity) * ew.direction;
	}

	private final static int getBinFromGF(double factor) {
		return (int) limit(0, (factor * MIDDLE_BIN) + MIDDLE_BIN, BINS - 1);
	}

	private final void updateStats() {
		statCache.clear();
	}

	private final float[] getWaveStat(EnemyWave w) {
		float[] stat = statCache.get(w);
		if (stat == null) {
			StatData data = dataMap.get(w);
			if (data == null)
				return null;
			stat = movementStats.getCombinedScoreData(data);
			if (flattener) {
				float[] flat = flattenerStats.getCombinedScoreData(data);
				for (int i = 0; i < flat.length; i++) {
					stat[i] = stat[i]/2f + flat[i]/2f;
				}
			}
			statCache.put(w, stat);
		}
		return stat;
	}

	private final void logHit(EnemyWave ew, Point2D.Double targetLocation) {

		// Rough BotWidth calculation
		// TODO Precise intersection
		double botWidth = Math.atan(22 / ew.fireLocation
				.distance(targetLocation));
		double angle = absoluteBearing(ew.fireLocation, targetLocation);

		double lowerGF = getGFFromAbsoluteAngle(ew, Utils
				.normalAbsoluteAngle(angle - botWidth));
		double upperGF = getGFFromAbsoluteAngle(ew, Utils
				.normalAbsoluteAngle(angle + botWidth));

		double maxGF = Math.max(lowerGF, upperGF);
		double minGF = Math.min(lowerGF, upperGF);

		StatData data = dataMap.get(ew);
		if (data == null) {
			robot.out.println("Error! No wave data!");
			return;
		}

		data._lgf = minGF;
		data._ugf = maxGF;

		// Record new data
		movementStats.update(data, 1f);

		// Clear the stats cache
		updateStats();
	}
	
	private final void logVisit(EnemyWave ew, Point2D.Double targetLocation) {

		// Rough BotWidth calculation
		// TODO Precise intersection
		double botWidth = Math.atan(22 / ew.fireLocation
				.distance(targetLocation));
		double angle = absoluteBearing(ew.fireLocation, targetLocation);

		double lowerGF = getGFFromAbsoluteAngle(ew, Utils
				.normalAbsoluteAngle(angle - botWidth));
		double upperGF = getGFFromAbsoluteAngle(ew, Utils
				.normalAbsoluteAngle(angle + botWidth));

		double maxGF = Math.max(lowerGF, upperGF);
		double minGF = Math.min(lowerGF, upperGF);

		StatData data = dataMap.get(ew);
		if (data == null) {
			robot.out.println("Error! No wave data!");
			return;
		}

		data._lgf = minGF;
		data._ugf = maxGF;

		// Record new data
		flattenerStats.update(data, 1f);

		// Clear the stats cache
		updateStats();
	}

	private final double getGoAngle(Point2D.Double center,
			Point2D.Double current, int direction) {
		double bearing = absoluteBearing(center, current);
		double distance = center.distance(current);
		double distanceFactor = limit(1d - turnLimit, distance / desiredDistance, 1d + turnLimit);
		bearing += (distanceFactor * direction * (Math.PI / 2));

		if (isRammer)
			bearing = absoluteBearing(enemy, current);
		return wallSmoothing(me, bearing, direction);
	}

	// CREDIT: mini sized predictor from Apollon, by rozu
	// http://robowiki.net?Apollon
	private final Point2D.Double predictPosition(EnemyWave surfWave,
			int direction) {
		Point2D.Double predictedPosition = (Point2D.Double) me.clone();
		double predictedVelocity = robot.getVelocity();
		double predictedHeading = robot.getHeadingRadians();
		double maxTurning, moveAngle, moveDir;

		int counter = 0; // number of ticks in the future
		boolean intercepted = false;

		do {
			moveAngle = getGoAngle(surfWave.fireLocation, predictedPosition,
					direction)
					- predictedHeading;
			moveDir = 1;

			if (Math.cos(moveAngle) < 0) {
				moveAngle += Math.PI;
				moveDir = -1;
			}

			moveAngle = Utils.normalRelativeAngle(moveAngle);

			// maxTurning is built in like this, you can't turn more then this
			// in one tick
			maxTurning = Math.PI / 720d
					* (40d - 3d * Math.abs(predictedVelocity));
			predictedHeading = Utils.normalRelativeAngle(predictedHeading
					+ limit(-maxTurning, moveAngle, maxTurning));

			// this one is nice ;). if predictedVelocity and moveDir have
			// different signs you want to break down
			// otherwise you want to accelerate (look at the factor "2")
			predictedVelocity += (predictedVelocity * moveDir < 0 ? 2 * moveDir
					: moveDir);
			predictedVelocity = limit(-8, predictedVelocity, 8);

			// calculate the new predicted position
			predictedPosition = project(predictedPosition, predictedHeading,
					predictedVelocity);

			counter++;

			if (predictedPosition.distance(surfWave.fireLocation) < surfWave.distanceTraveled
					+ (counter * surfWave.bulletVelocity)
					+ surfWave.bulletVelocity) {
				intercepted = true;
			}
			paintingPredictedPositions.add(predictedPosition);
		} while (!intercepted && counter < 500);

		return predictedPosition;
	}

	private final double checkDanger(EnemyWave surfWave, int direction) {
		Point2D.Double pos = predictPosition(surfWave, direction);

		// Note: botWidth here use factor '25' to cover a bit larger than
		// registering hit
		double botWidth = Math.atan(25 / surfWave.fireLocation.distance(pos));

		double angle = absoluteBearing(surfWave.fireLocation, pos);
		int lowerIndex = getBinFromGF(getGFFromAbsoluteAngle(surfWave, Utils
				.normalAbsoluteAngle(angle - botWidth)));
		int upperIndex = getBinFromGF(getGFFromAbsoluteAngle(surfWave, Utils
				.normalAbsoluteAngle(angle + botWidth)));

		int maxIndex = Math.max(lowerIndex, upperIndex);
		int minIndex = Math.min(lowerIndex, upperIndex);

		double danger = 0d;

		float[] _surfStats = getWaveStat(surfWave);
		for (int i = minIndex; i <= maxIndex; i++)
			danger += _surfStats[i];

		danger /= (maxIndex - minIndex + 1);

		if (Utils.isNear(danger, 0d)) {
			danger = 1e-8;
		}
		
		double distance = enemy.distanceSq(pos);
		
		if (isRammer) {
			return 1/distance;
		}
		
		return danger / Math.pow(distance, distanceDangerFactor);
	}

	private final void doSurfing() {
		paintingPredictedPositions.clear();
		
		if (robot.getRoundNum() >= 3)
			doDynamicFactor();

		EnemyWave surfWave = getFirstImpactSurfableWave();

		double goAngle;
		if (surfWave == null) {
			goAngle = getGoAngle(enemy, me, noWaveDirection);
			Point2D.Double pos = project(me, goAngle, WALL_STICK);
			if (enemy.distanceSq(pos) < WALL_STICK * WALL_STICK * 2) {
				goAngle = getGoAngle(enemy, me,
						noWaveDirection = -noWaveDirection);
			}
		} else {
			double dangerLeft = checkDanger(surfWave, -1);
			double dangerRight = checkDanger(surfWave, 1);

			if (dangerLeft < dangerRight) {
				goAngle = getGoAngle(surfWave.fireLocation, me, -1);
			} else {
				goAngle = getGoAngle(surfWave.fireLocation, me, 1);
			}
		}

		setBackAsFront(goAngle);
	}
	
	private final void doDynamicFactor() {
		int round = robot.getRoundNum();
		if (energyFired < 30*round) desiredDistance = 250;
		else if (energyFired < 50*round) desiredDistance = 350;
		else desiredDistance = limit(420d,420 + Math.pow(2,100d*(double)bulletHit/(bulletFired*2)),650d);
		
		double hitrate = (double)bulletHit/bulletFired;
		if (hitrate > 0.1 && !flattener && !isRammer) {
			flattener = true;
			robot.out.println("Curve Flattening Enabled");
		} else if (hitrate < 0.07 && flattener && !isRammer) {
			flattener = false;
			robot.out.println("Curve Flattening Disabled");
		}
		
		if (isRammer) {
			desiredDistance = 1000d;
		}
	}

	private static final class EnemyWave {
		Point2D.Double fireLocation;
		long fireTime;
		double bulletVelocity, directAngle, distanceTraveled;
		int direction;
	}

	// Credit: Voidious's wall smoothing
	private final double wallSmoothing(Point2D.Double botLocation,
			double startAngle, int orientation) {
		double x = botLocation.x;
		double y = botLocation.y;
		double angle = startAngle;

		angle += (4 * Math.PI);

		double testX = x + (Math.sin(angle) * WALL_STICK);
		double testY = y + (Math.cos(angle) * WALL_STICK);
		double wallDistanceX = Math.min(x - 25, bfWidth - x - 25);
		double wallDistanceY = Math.min(y - 25, bfHeight - y - 25);
		double testDistanceX = Math.min(testX - 25, bfWidth - testX - 25);
		double testDistanceY = Math.min(testY - 25, bfHeight - testY - 25);

		double adjacent = 0;
		int g = 0;

		while (!movableField.contains(testX, testY) && g++ < 25) {
			if (testDistanceY < 0 && testDistanceY < testDistanceX) {
				// wall smooth North or South wall
				angle = ((int) ((angle + (Math.PI / 2)) / Math.PI)) * Math.PI;
				adjacent = Math.abs(wallDistanceY);
			} else if (testDistanceX < 0 && testDistanceX <= testDistanceY) {
				// wall smooth East or West wall
				angle = (((int) (angle / Math.PI)) * Math.PI) + (Math.PI / 2);
				adjacent = Math.abs(wallDistanceX);
			}

			angle += orientation
					* (Math.abs(Math.acos(adjacent / WALL_STICK)) + 0.00000000005);

			testX = x + (Math.sin(angle) * WALL_STICK);
			testY = y + (Math.cos(angle) * WALL_STICK);
			testDistanceX = Math.min(testX - 25, bfWidth - testX - 25);
			testDistanceY = Math.min(testY - 25, bfHeight - testY - 25);
		}

		return Utils.normalAbsoluteAngle(angle);
	}

	private static final Point2D.Double project(Point2D.Double sourceLocation,
			double angle, double length) {
		return new Point2D.Double(sourceLocation.x + Math.sin(angle) * length,
				sourceLocation.y + Math.cos(angle) * length);
	}

	private static final double absoluteBearing(Point2D.Double source,
			Point2D.Double target) {
		return Math.atan2(target.x - source.x, target.y - source.y);
	}

	private static final double limit(double min, double value, double max) {
		return Math.max(min, Math.min(value, max));
	}

	private static final double bulletVelocity(double power) {
		return (20D - (3D * power));
	}

	private static final double maxEscapeAngle(double velocity) {
		return Math.asin(8.0 / velocity);
	}

	private final void setBackAsFront(double goAngle) {
		double angle = Utils.normalRelativeAngle(goAngle
				- robot.getHeadingRadians());
		if (Math.abs(angle) > (Math.PI / 2)) {
			if (angle < 0) {
				robot.setTurnRightRadians(Math.PI + angle);
			} else {
				robot.setTurnLeftRadians(Math.PI - angle);
			}
			robot.setBack(100);
		} else {
			if (angle < 0) {
				robot.setTurnLeftRadians(-1 * angle);
			} else {
				robot.setTurnRightRadians(angle);
			}
			robot.setAhead(100);
		}

		// Corner Avoider
		if (!movableFieldInnerBound.contains(me))
			robot.setMaxVelocity(7d);
		if (!movableField.contains(me))
			robot.setMaxVelocity(3d);
	}

	private final double getDistanceInLastNTick(int tick) {
		tick = Math.min(tick, locationHistory.size() - 1);
		return me.distance(locationHistory.get(tick));
	}

	// CREDIT: MORE STUFF BY SIMONTON =)
	private final double wallDistance(Point2D.Double sourceLocation, double eDist,
			double eAngle, int oDir) {
		return Math.min(Math.min(Math.min(distanceWest(bfHeight - 25
				- sourceLocation.getY(), eDist, eAngle - Math.PI / 2, oDir),
				distanceWest(bfHeight - 25 - sourceLocation.getX(), eDist,
						eAngle + Math.PI, oDir)), distanceWest(sourceLocation
				.getY() - 25, eDist, eAngle + Math.PI / 2, oDir)),
				distanceWest(sourceLocation.getX() - 25, eDist, eAngle, oDir));
	}

	private final double distanceWest(double toWall, double eDist, double eAngle, int oDir) {
		if (eDist <= toWall) {
			return Double.POSITIVE_INFINITY;
		}
		double wallAngle = Math.acos(-oDir * toWall / eDist) + oDir * Math.PI/2;
		return Utils.normalAbsoluteAngle(oDir * (wallAngle - eAngle));
	}

	public final void onPaint(java.awt.Graphics2D g) {
		g.setFont(new Font(Font.DIALOG, Font.PLAIN, 11));
		java.text.NumberFormat nf = java.text.NumberFormat.getNumberInstance();
		nf.setMaximumFractionDigits(2);
		nf.setMinimumFractionDigits(2);

		for (int i = 0; i < waves.size(); i++) {
			EnemyWave w = (waves.get(i));
			Point2D.Double center = w.fireLocation;
			int radius = (int) w.distanceTraveled;
			double MEA = maxEscapeAngle(w.bulletVelocity);

			float[] _surfStats = statCache.get(w);
			if (_surfStats != null) {
				float[] normalized = new float[_surfStats.length];
				float max = 0f;
				for (float a : _surfStats)
					max = Math.max(a, max);
				if (!Utils.isNear(0d, max)) {
					for (int j = 0; j < _surfStats.length; j++) {
						normalized[j] = _surfStats[j] / max;
					}
				} else {
					Arrays.fill(normalized, 0);
				}
				for (int j = 0; j < BINS; j++) {
					double thisDanger = limit(0, normalized[j], 1);
					Point2D.Double thisPoint = project(center, w.directAngle
							+ w.direction * (j - MIDDLE_BIN)
							/ (double) MIDDLE_BIN * MEA, radius + 10);
					int x = (int) Math.round(thisPoint.x);
					int y = (int) Math.round(thisPoint.y);

					// Color-based wave
					g.setColor(Color.getHSBColor(
							.66f - 0.66f * (float) thisDanger, 1f, 1f));
					g.fillOval(x - 2, y - 2, 4, 4);
				}
			}
			g.setColor(Color.darkGray);
			int x = (int) Math.round(center.x);
			int y = (int) Math.round(center.y);
			g.drawOval(x - radius, y - radius, radius * 2, radius * 2);

			Point2D.Double hand = project(center, w.directAngle, radius);
			g.drawLine((int) center.x, (int) center.y, (int) hand.x,
					(int) hand.y);
			g.drawOval((int) hand.x - 5, (int) hand.y - 5, 10, 10);
		}
		g.setColor(Color.pink);
		for (Point2D.Double p : paintingPredictedPositions) {
			int x = (int) Math.round(p.x);
			int y = (int) Math.round(p.y);
			g.fillOval(x - 2, y - 2, 4, 4);
		}

		g.setColor(Color.white);
		g.draw(movableField);
		g.draw(movableFieldInnerBound);
		try {
			g.setColor(Color.darkGray);
			g
					.drawRect((int) robot.getX() - 18, (int) robot.getY() - 18,
							36, 36);
			g.drawRect((int) enemy.x - 18, (int) enemy.y - 18, 36, 36);
			g.drawLine((int) robot.getX(), (int) robot.getY(), (int) enemy.x,
					(int) enemy.y);
			g.setColor(Color.white);
			int x = (int) Math.round((robot.getX() - enemy.x) / 2d + enemy.x) - 40;
			int y = (int) Math.round((robot.getY() - enemy.y) / 2d + enemy.y);
			g.drawString("Distance: " + nf.format(me.distance(enemy)), x, y);
			g.drawString("Desired: " + nf.format(desiredDistance), x, y + 11);
		} catch (NullPointerException ignored) {
		}
		g.setColor(Color.white);
		g.drawString(flattener ? "Curve Flattening Enabled" : "Curve Flattening Disabled", 10, 10);
		g.drawString("Enemy Hit rate: " + nf.format(100d*(double)bulletHit/bulletFired) + "% ("
				+ nf.format(100d*(double)energyHit/energyFired) + "%)", 145, 10);
	}
}
