package ag.battledata;

/**
 * Two-dimensional vector
 * @author Andree Grosse
 */
public class DataVector
{
    public double x, y;

    /**
     * Constructor
     */
    public DataVector() {		
		this.x = .0;
       	this.y = .0;
    }
    
    /**
     * Constructor
     * @param v DataVector whoose variables will be copied
     */
    public DataVector(DataVector v) {		
		this.x = v.x;
       	this.y = v.y;
    }
    
    /**
     * Constructor
     * @param x x-value
     * @param y y-value
     */
    public DataVector(double x, double y) {		
		this.x = x;
       	this.y = y;
    }

	/**
	* Constructs a (normalized) vector from a Robocode-angle.
	*/
	public DataVector(double angle) {
		x = Math.sin(angle);
		y = Math.cos(angle);
	}
		
	/**
	* Angle between two vectors with respect to the direction (RoboCode-
	* measurement, i.e. clockwise angle is positive). This vector and 
	* argument vector must be normalized.
	*/
	public double angleTo(DataVector vector) {
		double sign = Math.signum(y * vector.x - x * vector.y);
		// due to rounding error, abs(x * vector.x + y * vector.y) can be
		// > 1.0, so return then
		double temp = x * vector.x + y * vector.y;
		if(temp > 1.0) return 0;
		if(temp < -1.0) return Math.PI;
		if(sign == .0) sign = 1;
		return sign * Math.acos(temp);
	}
	
	// ------ Operations ----------
	
	/**
	 * @return the negative of this DataVector
	 */
	public DataVector negative() {  
         return new DataVector(-x, -y);
    }

	/**
	 * @param m 
	 * @return This DataVector multiplied by m
	 */
	public DataVector multiple(double m) {
		return new DataVector(m * x, m * y);
	}
	
	/**
	 * 
	 * @param v DataVector to add
	 * @return The sum of this DataVector and v
	 */
	public DataVector plus(DataVector v) {
		return new DataVector(x + v.x, y + v.y);
	}
	
	/**
	 * 
	 * @param v DataVector to substract
	 * @return The difference between this DataVector and v
	 */
	public DataVector minus(DataVector v) {
		return new DataVector(x - v.x, y - v.y);
	}
		
	/**
	 * Normalizes this DataVector
	 * @return The length of the DataVector
	 */
	public double normalize() {
		if(x == .0 && y == .0)
			return .0;
		double norm = Math.sqrt(x * x + y * y);	
		this.x /= norm;
       	this.y /= norm;
       	return norm;
	}
	
	/**
	 * Returns the length of this DataVector
	 * @return The length of the DataVector
	 */
	public double length() {
		if(x == .0 && y == .0)
			return .0;
		double norm = Math.sqrt(x * x + y * y);	
       	return norm;
	}
	
	/**
	* Clockwise rotation of the DataVector
	* @return The rotation of this DataVector.
	*/
	public DataVector rotate(double alpha) {
		DataVector v = new DataVector(this);
		v.x = x * Math.cos(alpha) + y * Math.sin(alpha);
		v.y = -x * Math.sin(alpha) + y * Math.cos(alpha);
		return v;
	}
	
	/**
	*  Gets the angle (between -PI and PI) to the Vecor. 
	*  Vector must be normalized.
	*  @return angle
	*/
	public double angle() {
		if(x == .0)
			return Math.acos(y);
		else
			return Math.signum(x) * Math.acos(y);
	}
	
	/**
	 * Prints the DataVector data.
	 */
	public void print() {
		System.out.println("X: " + x + ", Y: " + y);	
	}
	
	
	
}
