package ag.movement;

import ag.battledata.DataVector;


/**
 * A class representing a gravity line that emits force to both sides
 * @author Andree Grosse
 *
 */
public class GravLine extends DataVector implements GravObject{

	private double mForce, mExp, mMaxDist;
	private DataVector mForceDirection;
	
	/**
	 * Constructor
	 * @param position Position of the GravEllipse
	 * @param force GravForce of this point
	 * @param exp Decline exponent
	 * @param maxDist Maximum distance for force
	 * @param direction Direction of the force
	 */
	public GravLine(DataVector position, double force, double exp, 
			double maxDist, DataVector direction) {
		super(position);
		mForce = Math.signum(force) * Math.pow(Math.abs(force), exp);
		mExp = exp;
		mForceDirection = direction;
		mMaxDist = maxDist;
	}

	/**
	 * Sets the position of this line
	 * @param pos new position
	 */
	public void setPosition(DataVector pos){
		x = pos.x;
		y = pos.y;
	}
	
	/**
	 * Sets the direction of this GravLine
	 * @param dir new direction
	 */
	public void setDirection(DataVector dir){
		mForceDirection = dir;
	}
	
	/**
	 * Calculates the forces emitted on a point.
	 * @param point the point for which to calculate the force
	 * @return a DataVector representing the force
	 */
	public DataVector getForceOn(DataVector point){
		DataVector diff = point.minus(this);
		// Calculate the scalar product, i.e. the length
		// of the difference projected on mForceDirection
		double temp = diff.x * mForceDirection.x + diff.y * mForceDirection.y;

		if(temp < .0) {
			// negative force if behind the line
			return calcForce(mForceDirection, Math.abs(temp)).negative();
		}else{
			return calcForce(mForceDirection, temp);
		}		
	}	
	
	/**
	 * Applies force formula on a vector
	 * @param dir the simple difference vector
	 * @param len distance from force emitter to point
	 * @return a DataVector representing the force 
	 */
	private DataVector calcForce(DataVector dir, double len){
		if(len > mMaxDist){
			return new DataVector(.0, .0);
		}
		else if(len < 0.1)
			len = 0.1;
		len = Math.pow(len, mExp);				
		return dir.multiple(mForce / len);
	}

}
