package pl.Patton.Guns;

import robocode.*;

import java.awt.geom.Point2D;

import pl.Abstract.Gun;
import pl.Enemy.AdvancedEnemy;
import pl.Utilities.*;

/*******************************************************************************
 * A gun that utilizes Predictive Targeting - circular combined with linear.
 * 
 * Since: dev
 * 
 * Last changed: never
 * 
 * Created from Secrets from the Robocode masters: Circular targeting found at
 * http://www.ibm.com/developerworks/java/library/j-circular/
 ******************************************************************************/
public class PTGun extends Gun {

	public PTGun(String name, AdvancedRobot robot) {
		super(name, robot);
	}

	public void run(AdvancedEnemy enemy) {
		Point2D.Double myLoc = new Point2D.Double(robot.getX(), robot.getY());
		// If they're far away, don't fire as hard
		double firePower = enemy.velocity == 0 && enemy.pVelocity == 0 ? 3 : MiscUtils
				.limit(0.1, 3, 400 / enemy.distance);
		double bulletSpeed = BulletUtils.bulletVelocity(firePower);
		long time;
		Point2D.Double futPos = new Point2D.Double(enemy.x, enemy.y);
		// With each loop, the time and location gets more and more
		// accurate
		for (int i = 0; i < 32; i++) {
			time = robot.getTime() + (long) (myLoc.distance(futPos) / bulletSpeed);
			futPos = getFuturePos(time, enemy);
		}
		// Limit the value because it's absolutely ridiculous for an opponent to
		// be outside the battlefield
		futPos.x = MiscUtils.limit(18, robot.getBattleFieldWidth() - 18, futPos.x);
		futPos.y = MiscUtils.limit(18, robot.getBattleFieldHeight() - 18, futPos.y);
		double absDeg = AngleUtils.absoluteBearing(myLoc, futPos);
		// Face off towards him
		robot.setTurnGunRightRadians(AngleUtils.normalizeBearing(absDeg
				- robot.getGunHeadingRadians()));
		// If I'm facing the right direction, fire
		if (robot.getGunHeat() == 0 && Math.abs(robot.getGunTurnRemaining()) < 4)
			robot.setFire(firePower);
	}

	/**
	 * Calculates the probable future position of the opponent at a given time
	 * based on past information. Useful for both linear and circular targeting.
	 * 
	 * CREDIT: Alisdair Owen
	 * 
	 * @param when the time to calculate the point at
	 * @param enemy the enemy to calculate the point of
	 * @return the future position
	 */
	public Point2D.Double getFuturePos(long when, AdvancedEnemy enemy) {
		double futX = enemy.x, futY = enemy.y;
		double timeChange = when - enemy.lastUpdated;
		double change = enemy.heading - enemy.pHeading;
		// If they're not moving in a straight line, use circular targeting
		if (Math.abs(change) > 0.00001) {
			// How much they move
			double radius = enemy.velocity / change;
			// How much their heading will have changed overall by the time I
			// shoot at them
			double totalHeadingChange = timeChange * change;
			// And run the calculations
			futX = enemy.x + (Math.cos(enemy.heading) * radius)
					- (Math.cos(enemy.heading + totalHeadingChange) * radius);
			futY = enemy.y + (Math.sin(enemy.heading + totalHeadingChange) * radius)
					- (Math.sin(enemy.heading) * radius);
		}
		// If they are, use linear.
		else {
			futX = enemy.x + Math.sin(enemy.heading) * enemy.velocity * timeChange;
			futY = enemy.y + Math.cos(enemy.heading) * enemy.velocity * timeChange;
		}
		return new Point2D.Double(futX, futY);
	}
}