/*
 * Created on 31/10/2003
 */
package mnt.utils;

import robocode.*;
import java.awt.geom.*;
import robocode.util.*;

/**
 * Clase que contiene m�todos auxiliares para el c�lculo de �ngulos, distancias,
 * etc.
 * 
 * @author Lolo
 * @version 1.0
 */
public final class RoboMath {
    public static final int MAX_VELOCITY = 8;

    public static final int TAMAO_ROBOT = 36;

    /**
     * Calcula la coordenada x de un robot enemigo en base a nuestra posici�n,
     * la distancia hasta nosotros y su �ngulo.
     * 
     * @param x
     *            nuestra coordenada x
     * @param distancia
     *            distancia desde nuestro robot al enemigo.
     * @param ang
     *            �ngulo del robot enemigo.
     * @return la coordenada x del robot enemigo.
     */
    public static double calculaX(double x, double distancia, double ang) {
        return x + Math.sin(Math.toRadians(ang)) * distancia;
    }

    /**
     * Calcula la coordenada y de un robot enemigo en base a nuestra posici�n,
     * la distancia hasta nosotros y su �ngulo.
     * 
     * @param y
     *            nuestra coordenada y
     * @param distancia
     *            distancia desde nuestro robot al enemigo.
     * @param ang
     *            �ngulo del robot enemigo.
     * @return la coordenada y del robot enemigo.
     */
    public static double calculaY(double y, double distancia, double ang) {
        return y + Math.cos(Math.toRadians(ang)) * distancia;
    }

    /**
     * Devuelve el �ngulo relativo a nuestro robot de las coordenadas x e y. Es
     * el �ngulo que nuestro robot debe girar para encarar las coordenadas x e
     * y. El �ngulo devuelto est� comprendido entre 180 y -180.
     * 
     * @param miRobot
     *            referencia a nuestro Robot.
     * @param x
     *            coordenada x.
     * @param y
     *            coordenada y
     * @return el �ngulo relativo a nuestro robot.
     */
    public static double getBearing(Robot miRobot, double x, double y) {
        double bearing = Math.toDegrees(Math.atan((x - miRobot.getX())
                / (y - miRobot.getY())))
                - miRobot.getHeading();
        if (y > miRobot.getY())
            return normalRelativeAngle(bearing);
        else if (x > miRobot.getX())
            return normalRelativeAngle(180 + bearing);
        else
            return normalRelativeAngle(bearing - 180);
    }

    /**
     * Devuelve el �ngulo relativo al ca�on de nuestro robot de las coordenadas
     * x e y. Es el �ngulo que nuestro robot debe girar para encarar las
     * coordenadas x e y. El �ngulo devuelto est� comprendido entre 180 y -180.
     * 
     * @param miRobot
     *            referencia a nuestro Robot.
     * @param x
     *            coordenada x.
     * @param y
     *            coordenada y
     * @return el �ngulo relativo al ca�on de nuestro robot.
     */
    public static double getBearingFromGun(Robot miRobot, double x, double y) {
        double bearing = Math.toDegrees(Math.atan2((x - miRobot.getX()),
                (y - miRobot.getY())))
                - miRobot.getGunHeading();
        if (y > miRobot.getY())
            return normalRelativeAngle(bearing);
        else if (x > miRobot.getX())
            return normalRelativeAngle(180 + bearing);
        else
            return normalRelativeAngle(bearing - 180);
    }

    /**
     * Devuelve el �ngulo relativo a nuestro robot de las coordenadas x e y. Es
     * el �ngulo que nuestro robot debe girar para encarar las coordenadas x e
     * y. El �ngulo devuelto est� comprendido entre 180 y -180.
     * 
     * @param rx
     *            coordenada x de nuestro Robot.
     * @param ry
     *            coordenada y de nuestro Robot.
     * @param rang
     *            �ngulo de nuestro robot.
     * @param x
     *            coordenada x.
     * @param y
     *            coordenada y
     * @return el �ngulo relativo a nuestro robot.
     */
    public static double getBearing(double rx, double ry, double rang,
            double x, double y) {
        double bearing = Math.toDegrees(Math.atan((x - rx) / (y - ry))) - rang;
        if (y > ry)
            return normalRelativeAngle(bearing);
        else if (x > rx)
            return normalRelativeAngle(180 + bearing);
        else
            return normalRelativeAngle(bearing - 180);
    }

    /**
     * Devuelve el �ngulo pasado como p�rametro como un �ngulo comprendido entre
     * -180 y 180.
     * 
     * @param angle
     *            �ngulo a transformar.
     * @return el �ngulo transformado.
     */
    public static double normalRelativeAngle(double angle) {
        return Math.toDegrees(Utils.normalRelativeAngle(Math.toRadians(angle)));
    }

    /**
     * dado un angulo cualquier devolvemos el mismo angulo pero expresado entre
     * -180 grados y 180... es decir r=270 grados seria -90 grados (se usan
     * angulos en radianes)
     * 
     * @param r
     *            angulo a normalizar
     * @return angulo normalizado
     */
    public static double normalRelativeAngleR(double r) {
        if (r > -Math.PI && r <= Math.PI)
            return r;
        while (r <= -Math.PI)
            r += 2 * Math.PI;
        while (r > Math.PI)
            r -= 2 * Math.PI;
        return r;
    }

    /**
     * Devuelve el �ngulo pasado como p�rametro como un �ngulo comprendido entre
     * 0 y 360.
     * 
     * @param angle
     *            �ngulo a transformar.
     * @return el �ngulo transformado.
     */
    public static double normalAbsoluteAngle(double ang) {
        if (ang < 0)
            return 360 + (ang % 360);
        else if (ang >= 360)
            return ang % 360;
        else
            return ang;
    }

    /**
     * M�todo que sirve para estimar la coordenada x de la posici�n donde
     * encontrar� un enemigo dentro de tiempo unidades de tiempo.
     * 
     * @param e
     *            Enemigo del que se quiere obtener su coordenada x aproximada.
     * @param tiempo
     * @return coordenada x estimada.
     */
    public static double estimaX(Enemigo e, double tiempo) {
        return e.getX() + Math.sin(Math.toRadians(e.getHeading()))
                * e.getVelocity() * tiempo;
    }

    public static double estimaX(double x, double angulo, double velocidad,
            double tiempo) {
        return x + Math.sin(Math.toRadians(angulo)) * velocidad * tiempo;
    }

    /**
     * M�todo que sirve para estimar la coordenada x de la posici�n donde
     * encontrar� un enemigo dentro de tiempo unidades de tiempo.
     * 
     * @param e
     *            Enemigo del que se quiere obtener su coordenada x aproximada.
     * @param tiempo
     * @param difAng
     *            velocidad angular.
     * @return coordenada x estimada.
     */
    public static double estimaX(Enemigo e, double tiempo, double difAng) {
        double x;
        if (Math.abs(difAng) > 0.00001) {
            double radius = e.getVelocity() / difAng;
            double tothead = tiempo * Math.toRadians(difAng);
            x = e.getX()
                    + (Math.cos(Math.toRadians(e.getHeading())) * radius)
                    - (Math.cos(Math.toRadians(e.getHeading()) + tothead) * radius);
        } else {
            x = estimaX(e, tiempo);
        }
        return x;
    }

    /**
     * M�todo que sirve para estimar la coordenada y de la posici�n donde
     * encontrar� un enemigo dentro de tiempo unidades de tiempo.
     * 
     * @param e
     *            Enemigo del que se quiere obtener su coordenada y aproximada.
     * @param tiempo
     * @return coordenada y estimada.
     */
    public static double estimaY(Enemigo e, double tiempo) {
        return e.getY() + Math.cos(Math.toRadians(e.getHeading()))
                * e.getVelocity() * tiempo;
    }

    public static double estimaY(double y, double angulo, double velocidad,
            double tiempo) {
        return y + Math.cos(Math.toRadians(angulo)) * velocidad * tiempo;
    }

    /**
     * M�todo que sirve para estimar la coordenada y de la posici�n donde
     * encontrar� un enemigo dentro de tiempo unidades de tiempo.
     * 
     * @param e
     *            Enemigo del que se quiere obtener su coordenada y aproximada.
     * @param tiempo
     * @param difAng
     *            velocidad angular.
     * @return coordenada y estimada.
     */
    public static double estimaY(Enemigo e, double tiempo, double difAng) {
        double y;
        if (Math.abs(difAng) > 0.00001) {
            double radio = e.getVelocity() / difAng;
            double tothead = tiempo * Math.toRadians(difAng);
            y = e.getY()
                    + (Math.cos(Math.toRadians(e.getHeading())) * radio)
                    - (Math.cos(Math.toRadians(e.getHeading()) + tothead) * radio);
        } else {
            y = estimaY(e, tiempo);
        }
        return y;
    }

    /**
     * Devuelve el tiempo que le queda a la bala para llegar a destino. Se
     * utiliza como m�todo auxiliar para el m�todo tiempoDeImpacto.
     * 
     * @param miRobot
     *            referencia a nuestro robot.
     * @param e
     *            Enemigo al que se quiere disparar.
     * @param tiempo
     *            tiempo para el que se quiere estimar la posici�n.
     * @param potencia
     *            potencia de la bala.
     * @return tiempo que le queda a la bala para llegar a destino.
     */
    private static double aux(Robot miRobot, Enemigo e, Enemigo e2,
            double tiempo, double potencia) {
        double difang = Math.abs(e.getHeading() - e2.getHeading());
        return distancia(miRobot, estimaX(e, tiempo, difang), estimaY(e,
                tiempo, difang))
                - velocidadBala(potencia) * tiempo;
    }

    private static double aux(Robot miRobot, Enemigo e, double tiempo,
            double potencia) {

        return distancia(miRobot, estimaX(e, tiempo), estimaY(e, tiempo))
                - velocidadBala(potencia) * tiempo;
    }

    private static double aux(Robot miRobot, Enemigo e, double vel,
            double tiempo, double potencia) {

        return distancia(miRobot,
                estimaX(e.getX(), e.getHeading(), vel, tiempo), estimaY(e
                        .getY(), e.getHeading(), vel, tiempo))
                - velocidadBala(potencia) * tiempo;
    }

    /**
     * Devuelve la velocidad de una bala.
     * 
     * @param potencia
     *            potencia de la bala.
     * @return la velocidad de la bala.
     */
    public static double velocidadBala(double potencia) {
        return 20 - 3 * potencia;
    }

    /**
     * Devuelve el tiempo en el que se estima impactar� la bala con el robot
     * enemigo.
     * 
     * @param miRobot
     *            referencia a nuestro robot.
     * @param e
     *            enemigo al que se quiere disparar.
     * @param potencia
     *            potencia de la bala que se quiere disparar.
     * @param precision
     *            presici�n con la que se quiere obtener el tiempo de impacto.
     * @return tiempo de impacto.
     */
    public static double tiempoDeImpacto(Robot miRobot, Enemigo e,
            double potencia, double precision) {

        double t = 20;
        double ultT = 10;
        double auxT;
        double auxUltT = aux(miRobot, e, ultT, potencia);
        int i = 15;

        while (Math.abs(t - ultT) > precision && (i != 0)) {
            i--;
            auxT = aux(miRobot, e, t, potencia);
            if ((auxT - auxUltT) == 0.0)
                break;
            double proxT = t - auxT * (t - ultT) / (auxT - auxUltT);
            ultT = t;
            t = proxT;
            auxUltT = auxT;
        }

        return t;
    }

    public static double tiempoDeImpacto(Robot miRobot, Enemigo e, double vel,
            double potencia, double precision) {

        double t = 20;
        double ultT = 10;
        double auxT;
        double auxUltT = aux(miRobot, e, vel, ultT, potencia);
        int i = 15;

        while (Math.abs(t - ultT) > precision && (i != 0)) {
            i--;
            auxT = aux(miRobot, e, vel, t, potencia);
            if ((auxT - auxUltT) == 0.0)
                break;
            double proxT = t - auxT * (t - ultT) / (auxT - auxUltT);
            ultT = t;
            t = proxT;
            auxUltT = auxT;
        }

        return t;
    }

    public static double tiempoDeImpacto(Robot miRobot, Enemigo e, Enemigo e2,
            double potencia, double precision) {

        double t = 20;
        double ultT = 10;
        double auxT;
        double auxUltT = aux(miRobot, e, e2, ultT, potencia);
        int i = 15;

        while (Math.abs(t - ultT) > precision && (i != 0)) {
            i--;
            auxT = aux(miRobot, e, e2, t, potencia);
            if ((auxT - auxUltT) == 0.0)
                break;
            double proxT = t - auxT * (t - ultT) / (auxT - auxUltT);
            ultT = t;
            t = proxT;
            auxUltT = auxT;
        }

        return t;
    }

    /**
     * M�todo que se usa para saber el da�o que har�a una bala de potencia
     * potencia.
     * 
     * @param potencia
     *            potencia de la bala.
     * @return el da�o que causar�a la bala.
     */
    public static double daoPorBala(double potencia) {
        if (potencia > 1)
            return 4 * potencia + 2 * (potencia - 1);
        else if (potencia > 3)
            return 16;
        else
            return 4 * potencia;
    }

    public static int sign(double num) {
        return num > 0 ? 1 : -1;
    }

    /**
     * M�todo que se usa para saber la vida que se recobrar�a de impactar una
     * bala de potencia potencia en un robot enemigo.
     * 
     * @param potencia
     *            potencia de la bala.
     * @return la vida que se recobrar�a.
     */
    public static double vidaRecobrada(double potencia) {
        return 3 * potencia;
    }

    /**
     * Devuelve la distancia entre nuestro robot y uno enemigo.
     * 
     * @param miRobot
     *            nuestro robot.
     * @param e
     *            robot enemigo.
     * @return la distancia entre ambos.
     */
    public static double distancia(Robot miRobot, Enemigo e) {
        return Math.sqrt((miRobot.getX() - e.getX())
                * (miRobot.getX() - e.getX()) + (miRobot.getY() - e.getY())
                * (miRobot.getY() - e.getY()));
    }

    /**
     * Devuelve la distancia entre nuestro robot y unas coordenadas.
     * 
     * @param miRobot
     *            nuestro robot.
     * @param x
     *            coordenada x.
     * @param y
     *            coordenada y.
     * @return la distancia entre ambos.
     */
    public static double distancia(Robot miRobot, double x, double y) {
        return Math.sqrt((miRobot.getX() - x) * (miRobot.getX() - x)
                + (miRobot.getY() - y) * (miRobot.getY() - y));
    }

    /**
     * Devuelve la distancia entre dos coordenadas.
     * 
     * @param miRobot
     *            nuestro robot.
     * @param x1
     *            coordenada x.
     * @param y1
     *            coordenada y.
     * @param x2
     *            coordenada x.
     * @param y2
     *            coordenada y.
     * @return la distancia entre ambos.
     */
    public static double distancia(double x1, double y1, double x2, double y2) {
        return Math.sqrt((x1 - x2) * (x1 - x2) + (y1 - y2) * (y1 - y2));
    }

    public static long tiempoDeLLegadaDeBala(Robot miRobot, Point2D pos,
            double bp) {
        return (long) (distancia(miRobot, pos.getX(), pos.getY()) / velocidadBala(bp));
    }

    public static double getAbsBearing(Robot miRobot, Point2D ePos) {
        return Math.toDegrees(Math.atan2((ePos.getX() - miRobot.getX()), (ePos
                .getY() - miRobot.getY())));
    }

    public static double getAbsBearingRadians(Robot miRobot, Point2D ePos) {
        return Math.atan2((ePos.getX() - miRobot.getX()),
                (ePos.getY() - miRobot.getY()));
    }

    public static double absoluteBearing(Point2D source, Point2D target) {
        return Math.atan2(target.getX() - source.getX(), target.getY()
                - source.getY());
    }

    public static double ajustar(double v, double min, double max) {
        return Math.max(min, Math.min(max, v));
    }
    
    public static int ajustar(int v, int min, int max){
        return Math.max(min, Math.min(max, v));
    }

    public static double maxAnguloDeHuida(double bulletVelocity) {
        return Math.asin(MAX_VELOCITY / bulletVelocity);
    }
    
	public static Point2D.Double calculaPosicion(Point2D.Double pos, double heading, double distancia){
		
		return new Point2D.Double(pos.x + distancia*Math.sin(heading), pos.y + distancia*Math.cos(heading));			
	}
}