package ags.rougedc.waves;

import static robocode.util.Utils.normalRelativeAngle;
import static robocode.util.Utils.normalAbsoluteAngle;

import java.util.ArrayList;
import java.util.List;

import robocode.Rules;
import ags.utils.points.*;
import ags.utils.*;
import ags.rougedc.robots.VirtualRobot;

/**
 * A generic class for waves, handles intersection, and GuessFactor calculations
 * 
 * @author Alexander Schultz
 */
public abstract class Wave implements Cloneable {
    private final AbsolutePoint origin;
    private final double power;
    private final double speed;
    private double radius;
    private Range hitrange;
    private Range newhitrange;
    private Range lasthitrange;
    abstract public VirtualRobot getStaticTarget();
    
    public Wave(AbsolutePoint origin, double power) {
        this(origin, power, 0);
    }
    
    public Wave(AbsolutePoint origin, double power, double radius) {
        this.origin = (AbsolutePoint)origin.clone();
        this.power = power;
        this.speed = Rules.getBulletSpeed(power);
        this.radius = radius;
    }
    
    // Cloning constructor
    protected Wave(Wave source) {
        origin = source.origin;
        power = source.power;
        speed = source.speed;
        radius = source.radius;
        if (source.hitrange != null) hitrange = source.hitrange.clone();
        if (source.newhitrange != null) newhitrange = source.newhitrange.clone();
        if (source.lasthitrange != null) lasthitrange = source.lasthitrange.clone();
    }
    
    public AbsolutePoint getOrigin() { return origin;  }
    public double getPower() { return power; }
    public double getSpeed() { return speed; }
    public double getRadius() { return radius; }

    // Moves the wave one tick 
    public void move() { radius += speed; }
    
    // Checks if there's a hit
    public void checkHit(VirtualRobot r) {
        lasthitrange = hitrange;
        newhitrange = this.getGuessRange(r.getLocation());
        if (hitrange == null)
            hitrange = newhitrange;
        else if (newhitrange != null)
            hitrange = hitrange.grow(newhitrange);
    }
    // Return the range of Guessfactors that hit
    public Range getHitRange() { return hitrange; }
    public Range getNewHitRange() { return newhitrange; }
    public Range getLastHitRange() { return lasthitrange; }
    
    public boolean expired(VirtualRobot r) {
        // Check if it's too far away for any intersection
        if (origin.distance(r.getLocation())+Math.sqrt(2*18*18) < (radius-speed))
            return true;
        return false;
    }
    
    public boolean isNear(AbsolutePoint p, double margin) {
        double d = origin.distance(p);
        return (d <= radius+margin && d >= radius-margin);
    }
    public boolean isNearOld(AbsolutePoint p, double margin) {
        double d = origin.distance(p);
        return (d <= (radius-speed)+margin && d >= (radius-speed)-margin);
    }
    
    // Check if the wave hit a point in it's last movement
    public boolean intersects(AbsolutePoint p) {
        double d = origin.distance(p);
        return (d <= radius && d >= (radius-speed));
    }
    public boolean intersects(RelativePoint p) {
        return (p.magnitude <= radius && p.magnitude >= (radius-speed));
    }
    
    // Get the absolute max escape angle
    public double getMaxEscapeAngle() {
        return Math.asin(Rules.MAX_VELOCITY / speed);
    }
    
    public double getAngleTo(AbsolutePoint p) {
        return normalAbsoluteAngle(Math.atan2(p.x-origin.x, p.y-origin.y));
    }
    public double getHOTAngle() {
        AbsolutePoint p = getStaticTarget().getLocation();
        return normalAbsoluteAngle(Math.atan2(p.x-origin.x, p.y-origin.y));
    }
    private double getTargetLateralDirection() {
        RelativePoint v = getStaticTarget().getVelocity();

        double lateralVelocity = (v.magnitude * Math.sin(v.getDirection() - getHOTAngle()));
        if (lateralVelocity >= 0)
            return 1;
        else
            return -1;
    }
    
    // Get the guessfactor of an angle or point
    public double getGuessFactor(double angle) {
        return getTargetLateralDirection()*normalRelativeAngle(angle-getHOTAngle())/getMaxEscapeAngle();
    }
    public double getGuessFactor(RelativePoint p) {
        return getGuessFactor(p.getDirection());
    }
    public double getGuessFactor(AbsolutePoint p) {
        return getGuessFactor(RelativePoint.fromPP(origin, p));
    }
    
    // Get an angle from a guessfactor
    public double getAngle(double guessfactor) {
        return normalAbsoluteAngle((guessfactor*getMaxEscapeAngle()/getTargetLateralDirection())+getHOTAngle());
    }
    
    // Get the intersecting guessfactor range of a bot centered at a given point
    public Range getGuessRange(AbsolutePoint p) {
        // Check if it's too far away for any intersection
        if (Math.abs(origin.distance(p) - radius) > Math.sqrt(2*18*18)+speed)
            return null;
                
        // Store a list of possible extreme points 
        List<AbsolutePoint> possiblepoints = new ArrayList<AbsolutePoint>();
        
        // Get the sides of the bot
        double bottom = p.y - 18;
        double top = p.y + 18;
        double left = p.x - 18;
        double right = p.x + 18;
        
        // Check if there is actually intersections with any sides
        double lastradius = radius-speed;
        possiblepoints.addAll(XCircleIntersect(origin, lastradius, left,   bottom, top));
        possiblepoints.addAll(XCircleIntersect(origin, radius,     left,   bottom, top));
        possiblepoints.addAll(XCircleIntersect(origin, lastradius, right,  bottom, top));
        possiblepoints.addAll(XCircleIntersect(origin, radius,     right,  bottom, top));
        
        possiblepoints.addAll(YCircleIntersect(origin, lastradius, bottom, left,   right));
        possiblepoints.addAll(YCircleIntersect(origin, radius,     bottom, left,   right));
        possiblepoints.addAll(YCircleIntersect(origin, lastradius, top,    left,   right));
        possiblepoints.addAll(YCircleIntersect(origin, radius,     top,    left,   right));
        
        // Get the corners of the bot
        AbsolutePoint c1 = AbsolutePoint.fromXY(left,  bottom);
        AbsolutePoint c2 = AbsolutePoint.fromXY(left,  top);
        AbsolutePoint c3 = AbsolutePoint.fromXY(right, top);
        AbsolutePoint c4 = AbsolutePoint.fromXY(right, bottom);
        
        // Add corners too if they're within what we want
        if (intersects(c1)) possiblepoints.add(c1);
        if (intersects(c2)) possiblepoints.add(c2);
        if (intersects(c3)) possiblepoints.add(c3);
        if (intersects(c4)) possiblepoints.add(c4);
        
        // Find the lowest and highest guessfactors
        Range r = null;
        for (AbsolutePoint point : possiblepoints) {
            double gf = getGuessFactor(point);
            if (r != null)
                r = r.grow(gf);
            else
                r = new Range(gf);
        }
        
        return r;
    }
    
    private static List<AbsolutePoint> XCircleIntersect(AbsolutePoint origin, double r, double x, double miny, double maxy) {
        List<AbsolutePoint> intersects = new ArrayList<AbsolutePoint>();
        double dx = x - origin.x;
        double D = r*r-dx*dx;
        
        // Negative discriminant, no intersection
        if (D < 0)
            return intersects;
        
        // Get an intersect
        double d = Math.sqrt(D);
        double y1 = origin.y + d;
        if (y1 >= miny && y1 <= maxy)
            intersects.add(AbsolutePoint.fromXY(x, y1));
        
        // Zero discriminant, no need for more
        if (D == 0)
            return intersects;
        
        // Get another intersect
        double y2 = origin.y - d;
        if (y2 >= miny && y2 <= maxy)
            intersects.add(AbsolutePoint.fromXY(x, y2));
        
        return intersects;
    }

    private static List<AbsolutePoint> YCircleIntersect(AbsolutePoint origin, double r, double y, double minx, double maxx) {
        List<AbsolutePoint> intersects = new ArrayList<AbsolutePoint>();
        double dy = y - origin.y;
        double D = r*r-dy*dy;
        
        // Negative discriminant, no intersection
        if (D < 0)
            return intersects;
        
        // Get an intersect
        double d = Math.sqrt(D);
        double x1 = origin.x + d;
        if (x1 >= minx && x1 <= maxx)
            intersects.add(AbsolutePoint.fromXY(x1, y));
        
        // Zero discriminant, no need for more
        if (D == 0)
            return intersects;
        
        // Get another intersect
        double x2 = origin.x - d;
        if (x2 >= minx && x2 <= maxx)
            intersects.add(AbsolutePoint.fromXY(x2, y));
        
        return intersects;
    }
}
