package djc;
import robocode.*;
import java.util.Hashtable;
import java.util.Enumeration;

/**
 * AntiGravTacticalStrategy
 *   My stab at the trendy AntiGravity movement algorithm.
 *
 * All code created by Dan Cieslak, 2002.
 *
 * All code that is mine is available for anyone to use for
 * any purpose, so long as I am listed as a source.
 *
 * the antiGravMove() method is courtesy Alisdair Owens AntiGravityBot.
 *
 */
public class AntiGravTacticalStrategy extends DefaultTacticalStrategy implements AntiGravInterface
{
    /* ********************************************************************************** */
    /*                                   CONSTANTS                                        */
    /* ********************************************************************************** */
    public static double TOO_CLOSE = 75;
    public static int MAX_TRACER_TIME = 30;
    public static int MAX_WALL_TIME = 15;
    public static double CORNER_THRESHOLD = 150;

    /* ********************************************************************************** */
    /*                                MEMBER VARIABLES                                    */
    /* ********************************************************************************** */
    protected int iteration = 0;
    protected Hashtable transientPoints = new Hashtable();

    protected GravPoint gpCenter;
    protected GravPoint gpRandom;
    protected GravPoint gpC[] = new GravPoint[4];
    protected GravPoint gpWall1;
    protected GravPoint gpWall2;
    protected int iWall1Time=-1;
    protected int iWall2Time=-1;
    protected int iRandomTime=-1;

    // Constructor; set up the AntiGravTacticalStrategy class.
    public AntiGravTacticalStrategy (StrategyBot theRobot, TacticalStrategyManager theStrategyManager)
    {
	super(theRobot, theStrategyManager);
	name = TacticalStrategyManager.ANTIGRAV;
    }

    /**
     * setMovement - Use AntiGravity algorithm to move.
     */
    public void setMovement()
    {
	setupTransientPoints();
	antiGravMove();
	clearTransientPoints();
    }

    /**
     * Setting up other GravPoints to consider
     */
    public void clearTransientPoints()
    {
	transientPoints.clear();
    }

    /**
     * Setting up other GravPoints to consider
     */
    public void setupTransientPoints()
    {
	double centerWidth = self.getBattleFieldWidth() / 2.0;
	double centerHeight = self.getBattleFieldHeight() / 2.0;
	
	if(gpCenter == null || self.getOthers() == 1) {
	    initCenterPoint();
	}

	if(gpRandom == null || self.getOthers() == 1) {
	    initRandomPoint();
	}

	iteration = (iteration + 1) % 20;
	if (iteration == 0 || (Math.abs(gpCenter.strength) < .1 && self.getOthers() > 1)) {
	    updateCenterPoint(); 
	}

	if(self.myPos.distanceFrom(gpCenter) < TOO_CLOSE && self.getOthers() > 1) {
	   gpCenter.strength *= 0;
	}

	// Wall avoidance
	// Add an attractive point somewhere to add some motion (gpTracer)
	if(self.myPos.x < TOO_CLOSE) {
	    gpWall1 = new GravPoint(0, self.myPos.y, GravPoint.DEFAULT_WALL_STRENGTH, 
				    GravPoint.DEFAULT_WALL_EXP);
	    if(iWall1Time < 0) iWall1Time = MAX_WALL_TIME;
	}
	if(self.myPos.x + TOO_CLOSE > 2 * centerWidth) {
	    gpWall1 = new GravPoint(2 * centerWidth, self.myPos.y, 
				    GravPoint.DEFAULT_WALL_STRENGTH, GravPoint.DEFAULT_WALL_EXP);
	    if(iWall1Time < 0) iWall1Time = MAX_WALL_TIME;
	}
	if(self.myPos.y < TOO_CLOSE) { 
	    gpWall2 = new GravPoint(self.myPos.x, 0, GravPoint.DEFAULT_WALL_STRENGTH, 
				    GravPoint.DEFAULT_WALL_EXP);
	    if(iWall2Time < 0) iWall2Time = MAX_WALL_TIME;
	}
	if(self.myPos.y + TOO_CLOSE > 2 * centerHeight) {
	    gpWall2 = new GravPoint(self.myPos.x, 2 * centerHeight, 
				    GravPoint.DEFAULT_WALL_STRENGTH, GravPoint.DEFAULT_WALL_EXP);
	    if(iWall2Time < 0) iWall2Time = MAX_WALL_TIME;
	}

	transientPoints.put("CENTERPOINT", gpCenter);
	
	if(iWall1Time >= 0 && gpWall1 != null) {
	    transientPoints.put("WALL1", gpWall1);
	    iWall1Time--;
	}
	if(iWall2Time >= 0 && gpWall2 != null) {
	    transientPoints.put("WALL2", gpWall2);
	    iWall2Time--;
	}

        // Corner avoidence for 1v1
        if(self.getOthers() == 1) {
	    avoidCorners();
	}
	/*
	if(self.getOthers() == 1) {
	    updateRandomPoint();
	    transientPoints.put("RANDOM", gpRandom);
	}
	*/
	// Avoid pointing directly at enemy in 1v1
	if(self.getOthers() == 1) {
	    avoidFacingCurrentEnemy();
	}

    }

    /** Initializes the Center Point */
    private void initCenterPoint() 
    {
	double centerStrength = (self.random.nextDouble() * GravPoint.DEFAULT_CENTER_STRENGTH_RANGE - 
				 GravPoint.DEFAULT_CENTER_STRENGTH);
	if(gpCenter == null) {
	    gpCenter = new GravPoint( self.getBattleFieldWidth() / 2, 
				      self.getBattleFieldHeight() / 2, 
				      centerStrength,
				      GravPoint.CENTER_EXPONENT);
	}
	gpCenter.strength = centerStrength;

	if(self.getOthers() == 1) {
	    gpCenter.strength = GravPoint.ONE_V_ONE_CENTER_STRENGTH;
	    gpCenter.exponent = GravPoint.ONE_V_ONE_CENTER_EXPONENT;
	    gpCenter.bStrongerAtDistance = true;
	}
    }

    /** Initializes the Random Point */
    private void initRandomPoint() 
    {
	if(gpRandom == null) {
	    gpRandom = new GravPoint( 0, 0,
				      -GravPoint.DEFAULT_TARGET_STRENGTH / 3,
				      GravPoint.DEFAULT_TARGET_POSITION);
				      
	}
	gpRandom.bStrongerAtDistance = true;
    }

    /** Updates the Center Point */
    private void updateCenterPoint() 
    {
	if(self.getOthers() == 1) { // 1v1 stay in the middle of the field
	    gpCenter.strength = GravPoint.ONE_V_ONE_CENTER_STRENGTH;
	    gpCenter.exponent = GravPoint.ONE_V_ONE_CENTER_EXPONENT;
	    gpCenter.bStrongerAtDistance = true;
	} else {
	    gpCenter.strength = (self.random.nextDouble() * GravPoint.DEFAULT_CENTER_STRENGTH_RANGE 
				 - GravPoint.DEFAULT_CENTER_STRENGTH);
	}
    }

    /** Initializes the Random Point */
    private void updateRandomPoint() 
    {

    }

    /** Sets up points in corners to avoid */
    private void avoidCorners()
    {
	if(gpC[0] == null) {
	    gpC[0] = new GravPoint(0 , 0, -GravPoint.DEFAULT_CORNER_STRENGTH_RANGE,
				   GravPoint.CORNER_EXPONENT);
	}
	if(gpC[1] == null) {
	    gpC[1] = new GravPoint(self.getBattleFieldWidth() , 
				   self.getBattleFieldHeight() , 
				   -GravPoint.DEFAULT_CORNER_STRENGTH_RANGE,
				   GravPoint.CORNER_EXPONENT);
	}
	if(gpC[2] == null) {
	    gpC[2] = new GravPoint(0 , 
				   self.getBattleFieldHeight() , 
				   -GravPoint.DEFAULT_CORNER_STRENGTH_RANGE,
				   GravPoint.CORNER_EXPONENT);
	}
	if(gpC[3] == null) {
	    gpC[3] = new GravPoint(self.getBattleFieldWidth() , 
				   0 , -GravPoint.DEFAULT_CORNER_STRENGTH_RANGE,
				   GravPoint.CORNER_EXPONENT);
	}
	for(int i=0;i<4;i++) {
	    if(self.myPos.distanceFrom(gpC[i]) < CORNER_THRESHOLD) {
		transientPoints.put("C" + i, gpC[i]);
	    }
	}
    }

    /** Sets up points in corners to avoid */
    private void avoidFacingCurrentEnemy()
    {
	if(self.currentTarget != null) {
	    double myHeading = self.getHeadingRadians();
	    double headingToEnemy = self.myPos.headingTo(self.currentTarget.position);
	    
	    GravPoint gpTmp;
	    double dx = 2 * MathHelper.BOT_RADIUS * Math.sin(headingToEnemy + Math.PI / 4.0);
	    double dy = 2 * MathHelper.BOT_RADIUS * Math.cos(headingToEnemy + Math.PI / 4.0);
	    
	    gpTmp = new GravPoint(self.myPos.x + dx, self.myPos.y + dy, 3.0 /* Exponent */ ,
				  3000 /* Strength */);
	    transientPoints.put("HEADING_CHANGER", gpTmp);
	}
    }

    /**
     * Wrapper to the AntiGravity movement algorithm with
     * wall avoidance correction (which is not working, so far)
     */
    public void antiGravMove()
    {
	Enumeration enumTargetList = self.targetList.elements();
	GravPoint gp = new GravPoint();
	gp.self = self;
	Coordinate force = gp.computeAntiGravForces(self.myPos,
						    enumTargetList,
						    transientPoints.elements(),
						    self.inboundBulletList.elements());
	
	Coordinate targetPosition = new Coordinate(self.myPos.x - force.x, self.myPos.y - force.y);

	self.goTo(targetPosition);
    }

}
