package djc;
import java.util.Enumeration;

/**
 * GravPoint class - based on GravPoint class
 *   from Alisdair Owens (AntiGravityBot)
 */
public class GravPoint extends Coordinate
{
    /** Default */
    public static final double DEFAULT_TARGET_POSITION = 2.0;
    /** Default target strength setting */
    public static final double DEFAULT_TARGET_STRENGTH = -7000.0;
    /** Default incoming bullet strength setting */
    public static final double DEFAULT_INCOMING_BULLET_STRENGTH = -1000.0;
    /** Default strength setting for center point */
    public static final double DEFAULT_CENTER_STRENGTH = 1000.0;
    /** Default strength setting for corner points */
    public static final double DEFAULT_CORNER_STRENGTH_RANGE = 2000.0;
    /** 1v1 strength setting for center point */
    //public static final double ONE_V_ONE_CENTER_STRENGTH = 1.0 / 6.0;
    public static final double ONE_V_ONE_CENTER_STRENGTH = 40;
    /** Default strength setting for center point */
    public static final double DEFAULT_CENTER_STRENGTH_RANGE = 2000.0;
    /** Default strength setting for tracer point */
    public static final double TRACER_STRENGTH = 1000.0;
    /** Default strength setting for walls */
    public static final double DEFAULT_WALL_STRENGTH = -8000.0;
    /** Default exponent setting for walls */
    public static final double DEFAULT_WALL_EXP = 3.0;
    /** Default exponent setting for transients */
    public static final double DEFAULT_TRANSIENT_EXPONENT = 2.0;
    /** Default exponent setting for center point */
    public static final double CENTER_EXPONENT = 1.75;
    /** Default exponent setting for center point */
    public static final double CORNER_EXPONENT = 2.0;
    /** 1v1 exponent setting for center point */
    public static final double ONE_V_ONE_CENTER_EXPONENT = 0;
    /** Default exponent setting for tracer point */
    public static final double TRACER_EXPONENT = 1.5;
    /** Default exponent setting for incoming bullet */
    public static final double INCOMING_BULLET_EXPONENT = 1.7;
    /** Default strength setting for walls */
    public static final double MAX_FORCE = 150;

    public double strength = 0;
    public double exponent = 3.0;
    public boolean bStrongerAtDistance = false;
    public double distanceFactor = 20;
    public StrategyBot self = null;

    public GravPoint(double pX, double pY, double pStrength, double pExp)
    {
	x = pX;
	y = pY;
	strength = pStrength;
	exponent = pExp;
    }

    public GravPoint() {
	super();
    }

    /**
     * Basic computation for force.  Throttled to + or - MAX_FORCE.
     *
     * Added option for inverting the distance relationship through
     * the bStrongerAtDistance variable.
     *
     * @param origin - Current Position of Bot
     */
    public double computeForce(Coordinate origin)
    {
	double force;
	if(bStrongerAtDistance)
	    force = strength * Math.pow( distanceFrom(origin) / distanceFactor, exponent );
	else
	    force = strength / Math.pow( distanceFrom(origin) / distanceFactor, exponent );

	if (force < 0)
	    return Math.max(force, -MAX_FORCE);
	else
	    return Math.min(force, MAX_FORCE);
    }

    /**
     * Populates the strength field for a normal bot.
     *
     * @param trgtEnergy - Enemy energy
     * @param myEnegery  - My energy
     */
    public static double strengthForStandardTarget(StrategyBot _s, Target _t)
    {
	if(((_t.energy < 15) && (_s.getEnergy() > 3 * _t.energy) || _t.energy == 0.0) &&
	   _s.inboundBulletList.size() == 0)
	    return -DEFAULT_TARGET_STRENGTH; // Attractive force
	else
	    return DEFAULT_TARGET_STRENGTH;
	
	//return DEFAULT_TARGET_STRENGTH;
    }

    /**
     * Populates the strength field for an incoming bullet
     *
     * @param trgtEnergy - Enemy energy
     * @param myEnegery  - My energy
     */
    public static double strengthForIncomingBullet()
    {
	return DEFAULT_INCOMING_BULLET_STRENGTH;
    }

    /**
     * Iterate through two Enumerations that have GravPoints
     *
     * Using these enumerations is not the right way to do this;
     * I can not think of a good way to convert the self.targetList
     * into an Enumeration of GravPoints without dead bots.
     *
     * @param origin - current position
     * @param e1 - First list of Targets
     * @param e2 - Second list of GravPoints
     *
     * @return a Coordinate where .x is xForce and .y is yForce
     */
    public Coordinate computeAntiGravForces(Coordinate origin,
					    Enumeration e1,
					    Enumeration e2)
    {
	double force;
	double ang;
	double xforce = 0;
	double yforce = 0;
	Target t;
	GravPoint p;
	
	while(e2.hasMoreElements()) {
	    p = (GravPoint)e2.nextElement();
	    force = p.computeForce(origin);
	    ang = p.headingTo(origin);
	    xforce += force * Math.sin(ang);
	    yforce += force * Math.cos(ang);
	    //self.out.println("x=" + (int)xforce + " y=" + (int)yforce + " from force " + (int)force + " at " + 
	    //		     (int)Math.toDegrees(ang) + " str = " + (int)p.strength + "d = " +
	    //		     (int)self.myPos.distanceFrom(p) + " exp was " + p.exponent + " from " + (int)p.x +
	    //		     "," + (int)p.y + " bStrngerDist=" + p.bStrongerAtDistance);
	}

	while(e1.hasMoreElements()) {
	    t = (Target)e1.nextElement();
	    if(t.isAlive) {  // ignore dead people
		force = t.gp.computeForce(origin);
		ang = t.position.headingTo(origin);
		xforce += force * Math.sin(ang);
		yforce += force * Math.cos(ang);
		//self.out.println("x=" + (int)xforce + " y=" + (int)yforce + " from force " + (int)force + " at " + 
		//		 (int)Math.toDegrees(ang) + " str = " + (int)t.gp.strength + " exp was " + t.gp.exponent);
	    }
	}

	//self.out.println("Sum of forces: " + xforce + "," + yforce);

	//self.out.println("Sum of forces after wall avoidance: " + xforce + "," + yforce);

	return new Coordinate(xforce, yforce);
    }

    /**
     * Iterate through two Enumerations that have GravPoints
     *
     * Using these enumerations is not the right way to do this;
     * I can not think of a good way to convert the self.targetList
     * into an Enumeration of GravPoints without dead bots.
     *
     * @param origin - current position
     * @param e1 - First list of Targets
     * @param e2 - Second list of GravPoints
     * @param e3 - Third list of GravPoints
     *
     * @return a Coordinate where .x is xForce and .y is yForce
     */
    public Coordinate computeAntiGravForces(Coordinate origin,
					    Enumeration e1,
					    Enumeration e2,
					    Enumeration e3)
    {
	double force;
	double ang;
	double xforce = 0;
	double yforce = 0;
	GravPoint p;
	Coordinate c;

	c = computeAntiGravForces(origin, e1, e2);
	xforce = c.x;
	yforce = c.y;

	while(e3.hasMoreElements()) {
	    p = (GravPoint)e3.nextElement();
	    force = p.computeForce(origin);
	    ang = p.headingTo(origin);
	    xforce += force * Math.sin(ang);
	    yforce += force * Math.cos(ang);
	    //self.out.println("Bullet x=" + (int)xforce + " y=" + (int)yforce + " from force " + (int)force + " at " + 
	    //		     (int)Math.toDegrees(ang) + " exp was " + p.exponent);
	}

	//self.out.println("Sum of forces: " + xforce + "," + yforce);

	//self.out.println("Sum of forces after wall avoidance: " + xforce + "," + yforce);

	return new Coordinate(xforce, yforce);
    }
}
