package tzu.strategy;

import robocode.*;
import tzu.intel.*;
import tzu.event.*;
import tzu.util.*;
import tzu.movement.*;
import tzu.strategy.*;

/**
 * Provides a movement strategy for an advanced robot.
 */
public class MovementStrategy extends AbstractStrategy {

    AntiGravityManager      antiGravityManager;
    MoveManager             moveManager;
    TargetStrategyInterface targets;
    long                    lastRam;
    Bot                     prevRammer;


    /**
     * Create a movement strategy object.
     * @param ar    your AdvancedRobot.
     * @param emi   object implementing EnemyManagerInterface
     * @param tsi   object implementing TargetStrategyInterface.
     */
    public MovementStrategy(
            AdvancedRobot ar,
            EnemyManagerInterface emi,
            TargetStrategyInterface tsi,
            AntiGravityManager agm) {

        super(ar, emi);
        moveManager         = new MoveManager(ar);
        targets             = tsi;
        antiGravityManager  = agm;
        lastRam             = 0;
        prevRammer          = null;
    }


    /**
     * Reinitialization for second and subsequent rounds of battle.
     */
    public void reinitialize() {
        moveManager.reinitialize();
        lastRam             = 0;
        prevRammer          = null;
    }


    /**
     * Choose the best movement pattern for the current scenario and
     * execute it.
     */
    public void takeTurn() {

        double heading1;
        double heading2;
        double realHeading = moveManager.getRealHeading();

        if (myRobot.getTime() - lastRam <  5 &&  prevRammer != null) {
            moveManager.evadeRam(prevRammer);
            return;
        }

        Bot b = targets.getTarget();
        AntiGravity ag = antiGravityManager.calcAntiGravity();

        if (ag.getForce() == 0.0 && myRobot.getOthers() == 1) {

            if (b != null) {

                heading1 = BotMath.zeroTo360(b.getBearing() + A_90);
                heading2 = BotMath.zeroTo360(b.getBearing() - A_90);

                if (BotMath.abs(BotMath.plusMinus180(realHeading, heading1)) <
                    BotMath.abs(BotMath.plusMinus180(realHeading, heading2))) {

                    moveManager.turnTo(heading1);
                } else {
                    moveManager.turnTo(heading2);

                }
            }
            moveManager.setMove(HOP);
            moveManager.setSpeed(SLOW_SPEED);

        } else {

            /*
             * Anti-gravity forces will sometimes cause the robot to shake
             * back and forth rapidly, which is not good.  To prevent this
             * from occuring, only allow anti-gravity to reverse our direction
             * of travel if:
             *
             * 1) There is only one opponent left.
             * 2) We are about to run into a wall.
             * 3) We aren't currently moving or are moving very slow.
             * 4) We are currently being rammed or were just rammed.
             * 5) We have moved more than 40 pixels from our last position.
             */

            if (
                myRobot.getOthers() == 1                ||
                moveManager.headedForWall()             ||
                moveManager.getMove() == 0.0            ||
                moveManager.getSpeed() <= SLOW_SPEED    ||
                myRobot.getTime() - lastRam < RAMMING_ATTACK_MAX_AGE ||
                BotMath.abs(moveManager.shortestTurn(ag.getDirection())) < 90 ||
                moveManager.moveTracker.distanceFromLastStop() > 40) {

                heading1 = ag.getDirection() + moveManager.wiggle()/4;

                /*
                 * Don't allow anti-gravity to take us on a shallow angle
                 * of attack towards the enemy.  This is akin to committing
                 * suicide.
                 */
                if (myRobot.getOthers() == 1 && b != null) {
                    heading2 = BotMath.plusMinus180(
                            b.getBearing(),
                            ag.getDirection());

                    if (Math.abs(heading2) < 20) {
                        if (heading2 < 0) {
                            heading1 = BotMath.zeroTo360(b.getBearing() - 20);
                        } else {
                            heading1 = BotMath.zeroTo360(b.getBearing() + 20);
                        }
                    }
                }

                moveManager.turnTo(heading1);
                moveManager.setMove(ag.getForce());
                moveManager.setSpeed(MAX_SPEED);
            }
        }
    }

    public void onHitRobot(HitRobotEvent e) {
        lastRam = e.getTime();
        Bot rammer = enemyManager.get(e.getName());
        if (rammer == prevRammer) {
            moveManager.evadeRam(rammer);
        } else {
            prevRammer = rammer;
        }
    }
}
