/**
 * 
 */
package banshee.mini;

 import static java.lang.Math.PI;
import static robocode.util.Utils.normalAbsoluteAngle;
import static robocode.util.Utils.normalRelativeAngle;

import java.awt.geom.Point2D;
import java.util.Random;
import java.util.Vector;

import robocode.AdvancedRobot;
import robocode.Condition;
import robocode.Event;
import robocode.RobotDeathEvent;
import robocode.Rules;
import robocode.ScannedRobotEvent;
import banshee.Detractor;
import banshee.Detractors;
import banshee.Enemies;
import banshee.Enemy;

/**
 * @author banshee
 */
public class Nexus6 extends AdvancedRobot {
	
	private final String DET_X_LEFT = "DET_X_LEFT";
	private final String DET_X_RIGHT = "DET_X_RIGHT";
	private final String DET_Y_DOWN = "DET_Y_DOWN";
	private final String DET_Y_UP = "DET_Y_UP";
	
	private Enemies enemies = new Enemies();
	
	private Detractors detractors = new Detractors();
	
	private static double BATT_WIDTH;
	private static double BATT_HEIGHT;
	private static double BATT_DIAG;
	
	// current turn properties
	/** our heading in current turn (radians) */
	private double currHead;
	private double currX;
	private double currY;
	private long currTurn;

	@Override
	public void run() {

		// gather all events we get every turn
		addCustomEvent(new Condition("every_turn", 99) {
			@Override
			public boolean test() {
				// this will be called *before* main loop body
				setTurnProperties();
				processEvents(getAllEvents());
				return false;
			}
		});
		
		// set some general properties
		BATT_WIDTH = getBattleFieldWidth();
		BATT_HEIGHT = getBattleFieldHeight();
		BATT_DIAG = Point2D.distance(0, 0, BATT_WIDTH, BATT_HEIGHT);
		
		// our every_turn condition won't be called until turn1, so for turn0 we have to init properties manually
		setTurnProperties();
		
		// set radar & gun independent from movement
		setAdjustRadarForRobotTurn(true);
		setAdjustRadarForGunTurn(true);
		setAdjustGunForRobotTurn(true);
		
		// main loop
		while (true) {

			// now logic *after* processing events			
			Enemy currTarget = enemies.getCurrentTarget();
			
			if (currTarget != null) {
				// lock radar on target, how to make the arc wider?
				setTurnRadarRightRadians(normalRelativeAngle(currTarget.getAbsAngle() - getRadarHeadingRadians()));
			} else {
				// scan continuously
				setTurnRadarRightRadians(Rules.RADAR_TURN_RATE_RADIANS);
			}
			
			updateDetractors();
			
			// now find new direction to move, counting weighted average from detractors
			double newRelX = 0;
			double newRelY = 0;

			for (Detractor d : detractors.getDetractors()) {
				double diffX = d.getX() - currX;
				double diffY = d.getY() - currY;
				
				double power = d.getPower();
				
				newRelX += power * (-diffX);
				newRelY += power * (-diffY);
			}

			// if we were facing north how much we'd have to turn to face this detractor
			double gotoAngleAbs = Math.atan2(newRelX - currX, newRelY - currY);
			
			// correct with curr heading and find relative, so we don't have to turn more than 180
			double gotoAngleRel = normalRelativeAngle(gotoAngleAbs - currHead);

			// a bit of move randomization
			if (currTurn % 5 == 0) {
				Random rand = new Random();
				int right = (rand.nextFloat() > 0.5F) ? 1 : -1;
				gotoAngleRel = right * Math.toRadians(rand.nextInt((int) Rules.MAX_TURN_RATE * 2));
			}
			setTurnRightRadians(gotoAngleRel);

			// if about to turn sharp (> 120 degrees), stop
			if (Math.abs(gotoAngleRel) > 2 * PI / 3) {
				setAhead(0);
			} else {
				setAhead(160); // some large number...
			}
			
			// now setting gun
			if (currTarget != null) {

				double dist = Point2D.distance(currX, currY, currTarget.getX(), currTarget.getY());
				double power = 1;
				if (dist < 250) power++;
				if (dist < 60) power++;

				// linear targeting, using law of sines
				double tAbsAngle = currTarget.getAbsAngle();

				// we need to find angle A,
				// from law of sines: sin A = sin B * a / b
				double a = currTarget.getVelocity(); // enemy velocity
				double b = Rules.getBulletSpeed(power); // bullet velocity
				double angleB = currTarget.getHeading() - tAbsAngle; // OK?
				// so:
				double angleA = Math.asin(Math.sin(angleB) * a / b);
				
				// correct with current angles, OK?
				double gunTurn = normalRelativeAngle(tAbsAngle - getGunHeadingRadians() + angleA);
				setTurnGunRightRadians(gunTurn);

				setFire(power);
			}
			execute();
		}
	}
	

	private void setTurnProperties() {
		// setting this turn properties
		this.currHead = getHeadingRadians();
		this.currX = getX();
		this.currY = getY();
		this.currTurn = getTime();
	}
	
	private void updateDetractors() {
		// updating wall detractors (enemy dets should've been updated while processing events)
		detractors.update(DET_X_LEFT, 0, currY, detPower(currX));
		detractors.update(DET_X_RIGHT, BATT_WIDTH, currY, detPower(BATT_WIDTH - currX));
		detractors.update(DET_Y_DOWN, currX, 0, detPower(currY));
		detractors.update(DET_Y_UP, currX, BATT_HEIGHT, detPower(BATT_HEIGHT - currY));
	}
	
	
	/**
	 * @param events Everything our robot knows about the outer world at the beginning of each turn 
	 */
	private void processEvents(Vector<Event> events) {
		for (Event e : events) {
			if (e instanceof ScannedRobotEvent) {
				processScannedRobotEvent((ScannedRobotEvent) e);
			}
			if (e instanceof RobotDeathEvent) {
				processRobotDeathEvent((RobotDeathEvent) e);
			}
		}
	}

	private void processRobotDeathEvent(RobotDeathEvent e) {
		detractors.remove(e.getName());
		enemies.remove(e.getName());
	}


	private void processScannedRobotEvent(ScannedRobotEvent e) {
		double bear = e.getBearingRadians();
		double dist = e.getDistance();
		String name = e.getName();

		// find point where are enemy is
		
		// angle between North and line [me - target]
		double relTargetAngle = normalRelativeAngle(currHead + bear);
		
		double tanRel = Math.tan(relTargetAngle);
		
//		double diffY = dist / Math.sqrt(1 + Math.pow(tanRel, 2));
		double diffY = dist / Math.hypot(1, tanRel);
		double diffX = diffY * tanRel;
		
		if (Math.abs(relTargetAngle) > PI / 2) { // reverse for 2nd, 3rd quarter
			diffX *= -1;
			diffY *= -1;
		}
		
		double enemyY = currY + diffY;
		double enemyX = currX + diffX;
		
		// gun angles
		double tAbsAngle = normalAbsoluteAngle(bear + currHead);
		
		// update enemy info
		enemies.update(name, enemyX, enemyY, tAbsAngle, e.getVelocity(), bear, e.getHeadingRadians());
		enemies.setCurrentTarget(name);
		
		// update this enemy detractor info
		double power = detPower(dist);
		detractors.update(name, enemyX, enemyY, power);
	}

	/**
	 * @param dist
	 * @return
	 */
	private double detPower(double dist) {
		// let detracting power be inversely proportional to distance 
		double power = Math.pow(1 / (dist / BATT_DIAG), 2);
		return power;
	}

}
