package suh.movement;

import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import robocode.*;
import robocode.util.Utils;
import suh.util.Utility;

public class WaveSurfing extends Movement {
    public static final int BINS = 51;
    public static final int VELOCITY_INDEX = 5;
    public static final int DISTANCE_INDEX = 5;
    public static final double WALL_STICK = 160;
    
    public Point2D.Double myLocation;
    public Point2D.Double enemyLocation;
    
    public static double[][][] stats = new double[VELOCITY_INDEX][DISTANCE_INDEX][BINS];
    public ArrayList<EnemyWave> enemyWaves = new ArrayList<EnemyWave>();
    public ArrayList<Integer> surfDirections = new ArrayList<Integer>();
    public ArrayList<Double> surfAbsBearings = new ArrayList<Double>();
    
    public Rectangle2D.Double field;
    public double enemyEnergy = 100.0;
    public double lastVelocity;
    public double lastDistance;
    
    public WaveSurfing(AdvancedRobot self) {
	super(self);
    }

    @Override
    public void init() {
	field = new Rectangle2D.Double(18, 18, self.getBattleFieldWidth()-36, self.getBattleFieldHeight()-36);
	enemyWaves.clear();
	surfDirections.clear();
	surfAbsBearings.clear();
    }
    
    @Override
    public void onScannedRobot(ScannedRobotEvent e) {
	double bearing, power;
	
	myLocation = new Point2D.Double(self.getX(), self.getY());
	
	surfDirections.add(0, self.getVelocity() * Math.sin(e.getBearingRadians()) >= 0 ? 1 : -1);
	surfAbsBearings.add(0, (bearing = self.getHeadingRadians() + e.getBearingRadians()) + Math.PI);
	
	if(0.1 <= (power = enemyEnergy - (enemyEnergy = e.getEnergy())) && power <= 3.0 && surfDirections.size() > 2) {
	    EnemyWave ew = new EnemyWave();
	    ew.fireTime = self.getTime() - 1;
	    ew.velocity = Rules.getBulletSpeed(power);
	    ew.distance = ew.velocity;
	    ew.direction = surfDirections.get(2);
	    ew.directAngle = surfAbsBearings.get(2);
	    ew.origin = (Point2D.Double)enemyLocation.clone();
	    ew.stats = stats[(int)(Math.abs(lastVelocity)/2.0)][(int)Math.min(lastDistance/200.0, 4)];
	    enemyWaves.add(ew);
	}
	
	enemyLocation = Utility.project(myLocation, bearing, e.getDistance());
	
	lastVelocity = self.getVelocity();
	lastDistance = e.getDistance();
	
	updateWaves();
	doSurfing();
    }
    
    public void onHitByBullet(HitByBulletEvent e) {
        if (!enemyWaves.isEmpty()) {
            Point2D.Double hitBulletLocation = new Point2D.Double(e.getBullet().getX(), e.getBullet().getY());
            EnemyWave hitWave = null;

            for (int x = 0; x < enemyWaves.size(); x++) {
                EnemyWave ew = (EnemyWave)enemyWaves.get(x);

                if(Math.abs(ew.distance - myLocation.distance(ew.origin)) < 50
                    && Math.abs(Rules.getBulletSpeed(e.getBullet().getPower()) - ew.velocity) < 0.001) {
                    hitWave = ew;
                    break;
                }
            }

            if (hitWave != null) {
                logHit(hitWave, hitBulletLocation);
	
                enemyWaves.remove(enemyWaves.lastIndexOf(hitWave));
            }
        }
    }
    
    public void updateWaves() {
        for (int x = 0; x < enemyWaves.size(); x++) {
            EnemyWave ew = (EnemyWave)enemyWaves.get(x);

            ew.distance = (self.getTime() - ew.fireTime) * ew.velocity;
            if (ew.distance > myLocation.distance(ew.origin) + 50) {
                enemyWaves.remove(x);
                x--;
            }
        }
    }

    public EnemyWave getClosestSurfableWave() {
        double closestDistance = 50000;
        EnemyWave surfWave = null;

        for (int x = 0; x < enemyWaves.size(); x++) {
            EnemyWave ew = (EnemyWave)enemyWaves.get(x);
            double distance = myLocation.distance(ew.origin) - ew.distance;

            if (distance > ew.velocity && distance < closestDistance) {
                surfWave = ew;
                closestDistance = distance;
            }
        }

        return surfWave;
    }

    public int getFactorIndex(EnemyWave ew, Point2D.Double targetLocation) {
        double offsetAngle = (Utility.absoluteBearing(ew.origin, targetLocation) - ew.directAngle);
        double factor = Utils.normalRelativeAngle(offsetAngle) / Utility.maxEscapeAngle(ew.velocity) * ew.direction;
        return (int)Utility.limit(0, (factor * ((BINS - 1) / 2)) + ((BINS - 1) / 2), BINS - 1);
    }

    public void logHit(EnemyWave ew, Point2D.Double targetLocation) {
        int index = getFactorIndex(ew, targetLocation);

        for (int x = 0; x < BINS; x++) {
            ew.stats[x] += 1.0 / (Math.pow(index - x, 2) + 1);
        }
    }

    public Point2D.Double predictPosition(EnemyWave surfWave, int direction) {
    	Point2D.Double predictedPosition = (Point2D.Double)myLocation.clone();
    	double predictedVelocity = self.getVelocity();
    	double predictedHeading = self.getHeadingRadians();
    	double maxTurning, moveAngle, moveDir;

        int counter = 0;
        boolean intercepted = false;

    	do {
    		moveAngle =
                wallSmoothing(predictedPosition, Utility.absoluteBearing(surfWave.origin,
                predictedPosition) + (direction * (Math.PI/2)), direction)
                - predictedHeading;
    		moveDir = 1;

    		if(Math.cos(moveAngle) < 0) {
    			moveAngle += Math.PI;
    			moveDir = -1;
    		}

    		moveAngle = Utils.normalRelativeAngle(moveAngle);

    		maxTurning = Math.PI/720d*(40d - 3d*Math.abs(predictedVelocity));
    		predictedHeading = Utils.normalRelativeAngle(predictedHeading
                + Utility.limit(-maxTurning, moveAngle, maxTurning));

    		predictedVelocity += (predictedVelocity * moveDir < 0 ? 2*moveDir : moveDir);
    		predictedVelocity = Utility.limit(-8, predictedVelocity, 8);

    		predictedPosition = Utility.project(predictedPosition, predictedHeading, predictedVelocity);

		counter++;

            if (predictedPosition.distance(surfWave.origin) < surfWave.distance + (counter * surfWave.velocity) + surfWave.velocity) {
                intercepted = true;
            }
    	} while(!intercepted && counter < 500);

    	return predictedPosition;
    }

    public double checkDanger(EnemyWave surfWave, int direction) {
        int index = getFactorIndex(surfWave, predictPosition(surfWave, direction));
        return surfWave.stats[index];
    }

    public void doSurfing() {
        EnemyWave surfWave = getClosestSurfableWave();

        if (surfWave == null) { return; }

        double dangerLeft = checkDanger(surfWave, -1);
        double dangerRight = checkDanger(surfWave, 1);

        double goAngle = Utility.absoluteBearing(surfWave.origin, myLocation);
        if (dangerLeft < dangerRight) {
            goAngle = wallSmoothing(myLocation, goAngle - (Math.PI/2), -1);
        } else {
            goAngle = wallSmoothing(myLocation, goAngle + (Math.PI/2), 1);
        }

        Utility.setBackAsFront(self, goAngle);
    }
    
    public double wallSmoothing(Point2D.Double botLocation, double angle, int orientation) {
        while (!field.contains(Utility.project(botLocation, angle, WALL_STICK))) {
            angle += orientation * 0.05;
        }
        return angle;
    }
}
