package davidalves.net.data;

import robocode.*;
import davidalves.net.*;
import davidalves.net.math.*;
import davidalves.net.targeting.*;
import davidalves.net.targeting.strategies.*;
import davidalves.net.util.*;
import davidalves.net.movement.*;
import davidalves.net.movement.strategies.*;
import davidalves.net.movement.strategies.duel.*;

import java.util.*;
import java.io.*;

/**
 * @author David Alves
 *
 */

/*
 * 
 * NOTE: Unlike many other classes, this one does not store references to our bot and environment,
 * because if we did that, we'd have to save and load every single damn instance in memory for EACH
 * file we wrote a Strategy Database to. :-P
 * 
 * 
 */
 
public class StrategyDatabase implements Serializable{
	
	//Targeting
	TargetingStrategyInterface[][] targetingStrategy;
	MovingAverageInterface[][] targetingSuccessRate;
	HashSet virtualBullets;
	final int numTargetingStrategies = 22;
	final int numRanges = 6;
	
	//Movement
	MovementInterface[] movement;
	MovingAverageInterface[] movementWinRate;
	final int numMovementStrategies = 2;
	
	public StrategyDatabase(AbstractAdvancedBot me, EnvironmentInterface environment){	
		
		//Targeting
		targetingStrategy = new TargetingStrategyInterface[numRanges][numTargetingStrategies];
		targetingSuccessRate = new MovingAverageInterface[numRanges][numTargetingStrategies];
		virtualBullets = new HashSet();
		
		for (int i = 0; i < numRanges; i++){
			//Put the circular strategy first so that it's the default
			targetingStrategy[i][0] = new InstantaneousCircularStrategy();
			//targetingStrategy[i][0] = new StatisticalLinearStrategy(me, environment, .8);
			
			for (int ii = 0; ii < numTargetingStrategies; ii++){
				targetingSuccessRate[i][ii] = new FiniteMovingAverage(100);
				//successRate[i][ii] = new InfiniteMovingAverage();
			}
			for (int ii = 1; ii < numTargetingStrategies; ii++){
				
				targetingStrategy[i][ii] = new StatisticalLinearStrategy(-1.0 + ((double)(ii-1)) / 10);
			}
		}
		
		//Movement
		movementWinRate = new MovingAverageInterface[numMovementStrategies];
		for(int i = 0; i < numMovementStrategies; i++){
			movementWinRate[i] = new InfiniteMovingAverage();
		}
		/**/
		movement = new MovementInterface[numMovementStrategies];
		
		//movement[0]  = new WaypointStrategy(me, 5, 10, 5, 10);
		movement[0]  = new WaypointStrategy(me, 1, 1, 1, 5);
		
		movement[1]  = new WaypointStrategy(me, 1, 1, 1, 5);
		
		//movement[2] = new WaypointStrategy(me, 5, 10, 5, 10);
		
		//movement[3] = new WaypointStrategy(me, 5, 10, 5, 10);
		//*/
	}


/////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////// Targeting Functions /////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////

	public void updateVirtualBullets(AbstractAdvancedBot me, EnvironmentInterface environment){
		VirtualBullet virtualBullet;
		Iterator vbi = virtualBullets.iterator();
		//me.out.println("There are " + virtualBullets.size() + " virtual bullets");
		while(vbi.hasNext()){
			virtualBullet = (VirtualBullet) vbi.next();
			if (virtualBullet.getPosition((double)me.getTime()).distanceTo(environment.getRobotByName(virtualBullet.getTarget()).getLocation()) < 30){
				targetingSuccessRate[virtualBullet.getRangeClass()][virtualBullet.getFiringStrategy()].addValue(100);
				/*
				me.out.println("HIT! by strategy " + targetingStrategy[virtualBullet.getRangeClass()][virtualBullet.getFiringStrategy()] + " at time " + me.getTime());
				me.out.println("Bullet at " + virtualBullet.getPosition(me.getTime()));
				me.out.println("Enemy at " + environment.getRobotByName(virtualBullet.getTarget()).getLocation());
				//*/
				vbi.remove();
			} else if (environment.isOutOfBounds(virtualBullet.getPosition((double)me.getTime()))){
				targetingSuccessRate[virtualBullet.getRangeClass()][virtualBullet.getFiringStrategy()].addValue(0);
				/*
				me.out.println("MISS! by strategy " + targetingStrategy[virtualBullet.getRangeClass()][virtualBullet.getFiringStrategy()] + " at time " + me.getTime());
				me.out.println("Bullet at " + virtualBullet.getPosition(me.getTime()));
				me.out.println("Enemy at " + environment.getRobotByName(virtualBullet.getTarget()).getLocation());
				//*/
				vbi.remove();
			}
		}	
	}
	
	public void fireVirtualBullets(AbstractAdvancedBot me, EnvironmentInterface environment, RobocodeRobot target, double bulletPower){
		VirtualBullet virtualBullet;
		int rangeClass = DaveMath.getRangeClass(target.getDistance(), bulletPower);
		//Fire new VirtualBullets
		for(int i = 0; i < numTargetingStrategies; i++){
			virtualBullet = new VirtualBullet(		
				target.getName(), //Target
				rangeClass, //Range
				i, //Strategy
				me.getAbsoluteAngleTo(targetingStrategy[rangeClass][i].predictedIntercept(me, environment, bulletPower)), //Angle fired at
				me.getLocation(), //Origin
				(double) me.getTime(), //Time fired
				bulletPower //Power
				);
			/*
			me.out.print("Firing virtual bullet from " + me.getLocation() + " at angle " + Math.round(me.getAbsoluteAngleTo(targetingStrategy[rangeClass][i].predictedIntercept(bulletPower))));
			me.out.println(" at time " + me.getTime());
			//*/
			virtualBullets.add(virtualBullet);
		}	
		
	}
	
	public TargetingStrategyInterface getIdealTargetingStrategy(RobocodeRobot target, double bulletPower){
		int bestStrategy = 0;
		VirtualBullet virtualBullet;
		int rangeClass = DaveMath.getRangeClass(target.getDistance(), bulletPower);
		for(int i = 0; i < numTargetingStrategies; i++){
			//Select the best strategy
			if(targetingSuccessRate[rangeClass][i].getAverage() > targetingSuccessRate[rangeClass][bestStrategy].getAverage()){
				bestStrategy = i;
			}
		}
		return targetingStrategy[rangeClass][bestStrategy];
	}
	
	public String toString(){
		String returnValue = "\n\nTargeting Strategy Database Debugging Information\n";
		returnValue += "-------------------------------------------------\n";
		for(int i = 0; i < numRanges; i++){
			returnValue += "\nValues for range " + i + "\n";
			for(int ii = 0; ii < numTargetingStrategies; ii++){
				returnValue += "   " + targetingStrategy[i][ii] + ": " + targetingSuccessRate[i][ii] + "\n";
			}
		}
		return returnValue;
	}
	
	public String hitPercentageForRangeClass(int rangeClassQuery){
		String returnValue = "\nHit percentages for range class " + rangeClassQuery;
		returnValue += "\n---------------------------------\n";
		for(int ii = 0; ii < numTargetingStrategies; ii++){
			returnValue += "   " + targetingStrategy[rangeClassQuery][ii] + ": " + targetingSuccessRate[rangeClassQuery][ii] + "\n";
		}
		return returnValue;
	}
	
	public double probabilityOfHit(RobocodeRobot target, double bulletPower){
		int bestStrategy = 0;
		VirtualBullet virtualBullet;
		int rangeClass = DaveMath.getRangeClass(target.getDistance(), bulletPower);
		for(int i = 0; i < numTargetingStrategies; i++){
			if(targetingSuccessRate[rangeClass][i].getAverage() > targetingSuccessRate[rangeClass][bestStrategy].getAverage()){
				bestStrategy = i;
			}
		}
		return Math.max(0,targetingSuccessRate[rangeClass][bestStrategy].getAverage());
	}
	
	public double confidenceInProbabilityToHit(RobocodeRobot target, double bulletPower){
		int bestStrategy = 0;
		VirtualBullet virtualBullet;
		int rangeClass = DaveMath.getRangeClass(target.getDistance(), bulletPower);
		for(int i = 0; i < numTargetingStrategies; i++){
			if(targetingSuccessRate[rangeClass][i].getAverage() > targetingSuccessRate[rangeClass][bestStrategy].getAverage()){
				bestStrategy = i;
			}
		}
		return Math.min(Math.max(0,100 - 100 / targetingSuccessRate[rangeClass][bestStrategy].getSize()), 99);
	}
	
	public void resetAfterFileLoad(){
		virtualBullets = new HashSet();
	}
	
/////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////// Movement Functions /////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
	/*public int getIdealMovementStrategy(){
		int bestStrategy = 0;
		for (int i = 0; i < numMovementStrategies; i++){
			if (movementWinRate[i].getAverage() > movementWinRate[bestStrategy].getAverage()){
				bestStrategy = i;
			}
		}
		return bestStrategy;
	}
	
	public int getLeastUsedMovementStrategy(){
		int bestStrategy = 0;
		for (int i = 0; i < numMovementStrategies; i++){
			if (movementWinRate[i].getSize() < movementWinRate[bestStrategy].getSize()){
				bestStrategy = i;
			}
		}
		return bestStrategy;
	}*/
	
	public int getMovementStrategy(AbstractAdvancedBot me){
		int bestStrategy = 0;
		//if (me.getTime() % 2 == 1){
			for (int i = 0; i < numMovementStrategies; i++){
				//If we don't have enough data to determine which of the two is better, use the least used one
				double tTest = DaveMath.tTest(movementWinRate[i].getAverage(), movementWinRate[bestStrategy].getAverage(), movementWinRate[i].getSize(), movementWinRate[bestStrategy].getSize());
				//me.out.println("tTest... Result: " + DaveString.formatDouble(tTest) + " Percent1: " + DaveString.formatDouble(movementWinRate[i].getAverage()) + " Percent2: "  + DaveString.formatDouble(movementWinRate[bestStrategy].getAverage()) + " Size: " + (movementWinRate[i].getSize() + movementWinRate[bestStrategy].getSize()));
				if (tTest > Math.random()){
					if (movementWinRate[i].getSize() < movementWinRate[bestStrategy].getSize()){
						//me.out.println("Switching to less used movement strategy to gather data... Old: " + movement[bestStrategy] + " New: " + movement[i]);
						//me.out.println("> tTest results... Result: " + DaveString.formatDouble(tTest) + " Percent1: " + DaveString.formatDouble(movementWinRate[i].getAverage()) + " Percent2: "  + DaveString.formatDouble(movementWinRate[bestStrategy].getAverage()) + " Size: " + (movementWinRate[i].getSize() + movementWinRate[bestStrategy].getSize()));
						bestStrategy = i;
					}
				//if we know which one is better, use the better one
				} else {
					if (movementWinRate[i].getAverage() > movementWinRate[bestStrategy].getAverage()){
						//me.out.println("Switching to better movement strategy... Old: " + movement[bestStrategy] + " New: " + movement[i]);
						//me.out.println("> tTest results... Result: " + DaveString.formatDouble(tTest) + " Percent1: " + DaveString.formatDouble(movementWinRate[i].getAverage()) + " Percent2: "  + DaveString.formatDouble(movementWinRate[bestStrategy].getAverage()) + " Size: " + (movementWinRate[i].getSize() + movementWinRate[bestStrategy].getSize()));
						bestStrategy = i;
					}
				}
			}
		//} else {
		/*	for (int i = numMovementStrategies - 1; i > -1; i--){
				//If we don't have enough data to determine which of the two is better, use the least used one
				double tTest = DaveMath.tTest(movementWinRate[i].getAverage(), movementWinRate[bestStrategy].getAverage(), movementWinRate[i].getSize(), movementWinRate[bestStrategy].getSize());
				//me.out.println("tTest... Result: " + DaveString.formatDouble(tTest) + " Percent1: " + DaveString.formatDouble(movementWinRate[i].getAverage()) + " Percent2: "  + DaveString.formatDouble(movementWinRate[bestStrategy].getAverage()) + " Size: " + (movementWinRate[i].getSize() + movementWinRate[bestStrategy].getSize()));
				if (tTest > Math.random()){
					if (movementWinRate[i].getSize() < movementWinRate[bestStrategy].getSize()){
						me.out.println("Switching to less used movement strategy to gather data... Old: " + movement[bestStrategy] + " New: " + movement[i]);
						me.out.println("> tTest results... Result: " + DaveString.formatDouble(tTest) + " Percent1: " + DaveString.formatDouble(movementWinRate[i].getAverage()) + " Percent2: "  + DaveString.formatDouble(movementWinRate[bestStrategy].getAverage()) + " Size: " + (movementWinRate[i].getSize() + movementWinRate[bestStrategy].getSize()));
						bestStrategy = i;
					}
				//if we know which one is better, use the better one
				} else {
					if (movementWinRate[i].getAverage() > movementWinRate[bestStrategy].getAverage()){
						me.out.println("Switching to better movement strategy... Old: " + movement[bestStrategy] + " New: " + movement[i]);
						me.out.println("> tTest results... Result: " + DaveString.formatDouble(tTest) + " Percent1: " + DaveString.formatDouble(movementWinRate[i].getAverage()) + " Percent2: "  + DaveString.formatDouble(movementWinRate[bestStrategy].getAverage()) + " Size: " + (movementWinRate[i].getSize() + movementWinRate[bestStrategy].getSize()));
						bestStrategy = i;
					}
				}
			}
		}	*/
			
		return bestStrategy;
	}
	
	public void addLoss(int strategy){
		movementWinRate[strategy].addValue(0);
	}
	
	public void addWin(int strategy){
		movementWinRate[strategy].addValue(100);
	}
	
	public MovementInterface getMovementInterface(int i){
		return movement[i];
	}
	
	public MovingAverageInterface getMovementStrategyDamageRate(int i){
		return movementWinRate[i];
	}
	
	public String winRatesTable(){
		String returnValue = "";
		returnValue += "Movement Strategy Win Rates\n-=-=-=-=-=-=-=-=-=-=-=-=-=-";
		for(int i = 0; i < numMovementStrategies; i++){
			returnValue += "\n> " + getMovementInterface(i) + " : " + getMovementStrategyDamageRate(i);
		}
		returnValue += "\n";
		return returnValue;
	}
}
