package davidalves.net.movement.strategies.duel;

import robocode.*;
import davidalves.net.*;
import davidalves.net.data.*;
import davidalves.net.movement.MovementInterface;
import davidalves.net.math.*;
import davidalves.net.util.*;
import java.util.*;
import java.io.*;

//My highly tweaked anti-gravity movement method
public class WaypointStrategy implements MovementInterface, Constants, Serializable { 
	
	Point destination = new Point(0,0);
	double destinationGoodness = -100;
	double movementTime = 0;
	double stopTime = 0;
	boolean moving = false;
	double movementDirection = 1; //1 = Driving forwards, -1 = driving backwards
	
	final double MAX_SEARCH_AREA_SIZE = 800;
	
	//From zero to one, zero being true anti-gravity and 1 being random movement
	//MAX is not final so that we can set it upon instantiation
	final double MAX_RANDOMNESS = .00001;
	final double MIN_RANDOMNESS = 0;
	
	final int MIN_SEARCH_POINTS = 10;
	final int MAX_SEARCH_POINTS = 15;
	
	final double MIN_TRAVEL_DISTANCE_FACTOR = 0;
	final double MAX_TRAVEL_DISTANCE_FACTOR = 50;
	
	final double WALL_AVOIDANCE_FACTOR = 40000;		//How much walls should be avoided, relative to other things
	final double WALL_AVOIDANCE_FALLOFF = 2;			//How quickly the wall avoidance effect falls off from the wall
	
	final double CORNER_AVOIDANCE_FACTOR = 90000;	//How much corners should be avoided relative to other things
	final double CORNER_AVOIDANCE_FALLOFF = 3;		//How quickly the corner avoidance falls off as you get farther and farther from the corner
	
	final double ANGULAR_RESISTANCE_FACTOR = 100;			//How much we should avoid pointing straight toward/away from them
	final double ANGULAR_RESISTANCE_FALLOFF = 3.5;
	
	final double ENEMY_AVOIDANCE_FACTOR = 40000;		//See above
	final double ENEMY_AVOIDANCE_FALLOFF = 2;			//Ok ok you get the idea
	
	/*double BULLET_AVOIDANCE_FACTOR = 0;
	final double BULLET_AVOIDANCE_FALLOFF = 2;
	final long BULLET_AVOIDANCE_FRAME_OFFSET = 5;*/
	
	//final double ENEMY_ATTRACTION_FACTOR = 0;		//pushing us towards the enemy, 
	//final double ENEMY_ATTRACTION_FALLOFF = 3;
	
	double MIN_BASE_MOVEMENT_TIME = 5;
	double MAX_BASE_MOVEMENT_TIME = 10;
	double MIN_MOVEMENT_TIME_RANDOMNESS = 5;
	double MAX_MOVEMENT_TIME_RANDOMNESS = 10;
	
	final double MIN_STOP_TIME = 10;
	final double MAX_STOP_TIME = 30;
	final double MIN_STOP_PROBABILITY = .5;
	final double MAX_STOP_PROBABILITY = 1;
	/*
	final double MIN_STOP_TIME = 5;
	final double MAX_STOP_TIME = 30;
	final double MIN_STOP_PROBABILITY = .3;
	final double MAX_STOP_PROBABILITY = .6;
	//*/
	final double GOODNESS_SCALE = 10000;
	
	HashSet prospectiveDestinations;
	
	public WaypointStrategy(AbstractAdvancedBot me, double minBaseMovementTime, double maxBaseMovementTime, double minMovementTimeRandomness, double maxMovementTimeRandomness){
		prospectiveDestinations = new HashSet();
		
		MIN_BASE_MOVEMENT_TIME = minBaseMovementTime;
		MAX_BASE_MOVEMENT_TIME = maxBaseMovementTime;
		MIN_MOVEMENT_TIME_RANDOMNESS = minMovementTimeRandomness;
		MAX_MOVEMENT_TIME_RANDOMNESS = maxMovementTimeRandomness;
	}

/////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////// Movement Management //////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
	
	public void driveTank(AbstractAdvancedBot me, EnvironmentInterface environment){
		
		if(environment.targetLocked()){
			strafeTarget(me, environment);
		} else {
			seekOutTarget(me, environment);
		}
	}
	
	void strafeTarget(AbstractAdvancedBot me, EnvironmentInterface environment){
		
		if((me.getLocation().equals(destination) || movementTime <= 0) && moving){
			if(me.getLocation().equals(destination)){
				//me.out.println("Arrived at destination, stopping...");
			} else {
				//me.out.println("Movement timed out without reaching destination, stopping...");
			}
			movementTime = Math.min(movementTime, 3);
			moving = false;
			stopTime = pickStopTime(me, environment);
		}
		
		//me.out.println("Movement Time: " + Math.round(movementTime) + " Stop Time: " + Math.round(stopTime) + " Moving: " + moving);
		
		if (!moving){
			if(stopTime > 0){
				//me.out.println("Stopped, will start moving in " + Math.round(stopTime) + " turns");
				stopTime--;
				if (stopTime < 5){
					addProspectiveDestinations(me, environment);
				}
			} else {
				if(prospectiveDestinations.size() == 0){
					addProspectiveDestinations(me, environment);
				}
				
				//destinationGoodness = getGoodness(me, environment, destination);
				destinationGoodness = Double.MIN_VALUE;
				
				Iterator i = prospectiveDestinations.iterator();
				while(i.hasNext()){
					Point prospectiveDestination = (Point) i.next();
					double prospectiveDestinationGoodness = getGoodness(me, environment, prospectiveDestination);
					
					if (prospectiveDestinationGoodness > destinationGoodness){
						//me.out.println("Changing destinations on turn: " + me.getTime());
						movementTime = pickMovementTime(me, environment);
						moving = true;
						if(environment.targetLocked()){
							destination = prospectiveDestination;
							destinationGoodness = prospectiveDestinationGoodness;
							
						}
					}
					i.remove();
				}
				//me.out.println("Enemy is at range: " + Math.round(getEnemyDistance(me, environment)));
				//me.out.println("Next movement decision in " + Math.round(Math.ceil(movementTime)) + " turns...");
			}
		} else { //We're moving
			goTo(destination, me, environment);
			
			//me.out.println("Moving (Distance to destination: " + Math.round(me.getLocation().distanceTo(destination))+ "), will stop or change direction in " + Math.round(movementTime) + " turns");
			movementTime--;
			
			if (movementTime < 5){
				//me.out.println("Stopping soon, adding to list of prospective destinations...");
				addProspectiveDestinations(me, environment);
			}
		}
	}
	
	/*void seekOutTarget(AbstractAdvancedBot me, EnvironmentInterface environment){
		if((me.getLocation().equals(destination) || movementTime <= 0) && moving){
			movementTime = Math.min(movementTime, 3);
			moving = false;
			stopTime = pickStopTime(me, environment);
		}
		if (!moving){
			if(stopTime > 0){
				stopTime--;
				if (stopTime < 8){
					addProspectiveDestinations(me, environment);
				}
			} else {
				if(prospectiveDestinations.size() == 0){
					addProspectiveDestinations(me, environment);
				}
				destinationGoodness = Double.MIN_VALUE;
				Iterator i = prospectiveDestinations.iterator();
				while(i.hasNext()){
					Point prospectiveDestination = (Point) i.next();
					double prospectiveDestinationGoodness = getGoodness(me, environment, prospectiveDestination);
					
					if (prospectiveDestinationGoodness > destinationGoodness){
						movementTime = pickMovementTime(me, environment);
						moving = true;
						if((directionTo(me.getAbsoluteAngleTo(prospectiveDestination), me, environment) == directionTo(me.getAbsoluteAngleTo(destination), me, environment) || (Math.random() < .4))
								 && DaveMath.angularDifferenceBetween(me.getAbsoluteAngleTo(prospectiveDestination), me.getAbsoluteAngleTo(destination)) > 25
								 && DaveMath.angularDifferenceBetween(me.getAbsoluteAngleTo(prospectiveDestination), me.getAbsoluteAngleTo(destination)) < 50){
									destination = prospectiveDestination.nearestRobotLocation(environment.getArenaSize(),60);;
									destinationGoodness = prospectiveDestinationGoodness;
						}
					}
					i.remove();
				}
			}
		} else { //We're moving
			goTo(destination, me, environment);
			movementTime--;
			if (movementTime < 8){
				addProspectiveDestinations(me, environment);
			}
		}
	}*/
	void seekOutTarget(AbstractAdvancedBot me, EnvironmentInterface environment){
		if((me.getLocation().equals(destination) || movementTime <= 0)){
			movementTime = Math.min(movementTime, 0);
			moving = false;
			stopTime = pickStopTime(me, environment);
		}
		if (!moving){ //We're stopped
			destination = me.getLocation().plus(getRandomPointWithinAcceptableArea(me, environment));
			movementTime = pickMovementTime(me, environment);
			moving = true;
		} else { //We're moving
			goTo(destination, me, environment);
			movementTime--;
		}
	}
	
/////////////////////////////////////////////////////////////////////////////////////////
////////////////////////// Destination Point Utility Functions //////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////

	void addProspectiveDestinations(AbstractAdvancedBot me, EnvironmentInterface environment){
		double enemyDistance = getEnemyDistance(me, environment);
		
		int pointsToTest = (int) DaveMath.scaledValue(MIN_SEARCH_POINTS, MAX_SEARCH_POINTS, 0, 1200, enemyDistance);
		int destinations = prospectiveDestinations.size();
		
		destinationGoodness = getGoodness(me, environment, destination);
		
		//me.out.println("Searching for potential destinations...");

		for(int i = 0; i < pointsToTest; i++){
			Point offset = getRandomPointWithinAcceptableArea(me, environment);
			Point prospectiveDestination = me.getLocation().plus(offset).nearestPossibleRobotLocation(environment.getArenaSize());
			double prospectiveDestinationGoodness = getGoodness(me, environment, prospectiveDestination);
			if (prospectiveDestinationGoodness > destinationGoodness || Math.random() > .9){
				prospectiveDestinations.add(prospectiveDestination);
			}
		}
		
		//me.out.println("Found " + (prospectiveDestinations.size() - destinations) + " new prospective destinations for a total of " + prospectiveDestinations.size());
	}
	
	double getGoodness(AbstractAdvancedBot me, EnvironmentInterface environment, Point p){
		double enemyDistance = getEnemyDistance(me, environment);
		double randomness = DaveMath.scaledValue(MIN_RANDOMNESS, MAX_RANDOMNESS, 0, 1200, enemyDistance);
		return DaveMath.random(GOODNESS_SCALE - randomness * GOODNESS_SCALE, GOODNESS_SCALE) / (getForceAtPoint(p, enemyDistance, me, environment));
	}
	
	double getEnemyDistance(AbstractAdvancedBot me, EnvironmentInterface environment){
		return environment.targetLocked()?(environment.getClosestTarget().getDistance()):0;
	}
	
	double getEnemyBulletTravelTime(AbstractAdvancedBot me, EnvironmentInterface environment){
		if(environment.targetLocked()){
			return environment.getClosestTarget().getLastBulletTravelTime();
		} else {
			//Return some random crap so we go all over the place looking for a target
			return 10 + Math.random() * 20;
		}
	}
	
Point getRandomPointWithinAcceptableArea(AbstractAdvancedBot me, EnvironmentInterface environment){
		double enemyDistance = getEnemyDistance(me, environment);
		if(!environment.targetLocked())
			enemyDistance = 200;
		Point myLocation = me.getLocation();
		double x = myLocation.getX();
		double y = myLocation.getY();
		double borderSize = 50;
		double minX = borderSize;
		double minY = borderSize;
		double maxX = environment.getArenaSize().getX() - borderSize;
		double maxY = environment.getArenaSize().getY() - borderSize;
		double searchAreaSize = Math.min(Math.max(50, enemyDistance) , MAX_SEARCH_AREA_SIZE);
		Point p;
		
		p = new Point(
		DaveMath.random(
			//Minimum X
			(x - searchAreaSize < minX)?(minX - x):(-searchAreaSize),
			//Maximum X
			(x + searchAreaSize > maxX)?(maxX - x):(searchAreaSize)),
		DaveMath.random(
			//Minimum Y
			(y - searchAreaSize < minY)?(minY - y):(-searchAreaSize),
			//Maximum Y
			(y + searchAreaSize > maxY)?(maxY - y):(searchAreaSize)));
		
		return p;
	}
	
	
	double getForceAtPoint(Point position, double closestBotDistance, AbstractAdvancedBot me, EnvironmentInterface environment) {
		Hashtable forces = new Hashtable(30);
		double force = 0, temp;
		long time = me.getTime();
		double attackAngle = 0;
		
		Point p;					//A temporary point
		RobocodeRobot en = new RobocodeRobot();	//An enemy Robot
		EnemyBullet b;				//A bullet
		Iterator i = environment.getEnemyBulletIterator();
		Point arenaCenter = environment.getArenaCenter();
		
		//Factor enemies into our movement strategy
		/*if (environment.targetLocked()){
			force += Math.pow(environment.getTarget().getDistance() * ENEMY_ATTRACTION_FACTOR, ENEMY_ATTRACTION_FALLOFF);
			me.out.println("Attraction force: " + force);
		}*/
		
		
		
		
		//Factor RobocodeRobot bullets into our movement strategy... or not :-P
		while (i.hasNext()){
			b = (EnemyBullet)i.next();
			if (!environment.isOutOfBounds(b.getPosition(time))) {
			} else {
				i.remove();
			}
		}
		//*/
		
		//Avoid walls - Not really needed, our point selection method should prevent us from crashing...
		/*
		forceFromWalls += WALL_AVOIDANCE_FACTOR / Math.pow(me.getBattleFieldWidth() - me.getX(), WALL_AVOIDANCE_FALLOFF);
		forceFromWalls += WALL_AVOIDANCE_FACTOR / Math.pow(me.getX(), WALL_AVOIDANCE_FALLOFF);
		forceFromWalls += WALL_AVOIDANCE_FACTOR / Math.pow(me.getBattleFieldHeight() - me.getY(), WALL_AVOIDANCE_FALLOFF);
		forceFromWalls += WALL_AVOIDANCE_FACTOR / Math.pow(me.getY(), WALL_AVOIDANCE_FALLOFF);
		//*/
		if(environment.targetLocked()){
			RobocodeRobot target = environment.getTarget();
			attackAngle = Math.abs(DaveMath.angularDifferenceBetween(target.getBearing() + 90, me.getLocation().absoluteAngleTo(position)));
			if(attackAngle > 90)
				attackAngle = 180 - attackAngle;
			
			temp = ANGULAR_RESISTANCE_FACTOR * Math.pow(attackAngle / 90, ANGULAR_RESISTANCE_FALLOFF);
			forces.put("Angle to Enemy", new Double(temp));
		}
		
		//So we're encouraged to travel further
		temp = DaveMath.random(MIN_TRAVEL_DISTANCE_FACTOR, MAX_TRAVEL_DISTANCE_FACTOR) / closestBotDistance;
		forces.put("Travel Distance", new Double(temp));
		
		//Avoid getting TOO close to corners - corners are good, but not if you're stuck
		double maxDistanceFromCenter = (new Point(0,0)).distanceTo(arenaCenter);
		double currentDistanceFromCenter = position.distanceTo(arenaCenter);
		temp = CORNER_AVOIDANCE_FACTOR / Math.pow(maxDistanceFromCenter - currentDistanceFromCenter, CORNER_AVOIDANCE_FALLOFF);
		forces.put("Center Attraction", new Double(temp));

		Enumeration e = forces.elements();
		while (e.hasMoreElements()){
			force += ((Double) e.nextElement()).doubleValue();
		}
		
		//Debugging messages
		/*
		me.out.println("Summary of forces:");
		e = forces.keys();
		while (e.hasMoreElements()){
			String key = (String) e.nextElement();
			double absoluteForce = ((Double) forces.get(key)).doubleValue();
			double forcePercent = 100 * absoluteForce / force;
			me.out.println(" " + key + " Force: " + DaveString.formatDouble(absoluteForce) + " (" + DaveString.formatDouble(forcePercent) + "%)");
		}
		//*/
		
		return force;
	}

/////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////// Movement Timing ////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////

	double pickMovementTime(AbstractAdvancedBot me, EnvironmentInterface environment){
		double closestBotDistance = getEnemyDistance(me, environment);
		double enemyBulletTravelTime = getEnemyBulletTravelTime(me, environment);
		double baseMovementTime = MIN_BASE_MOVEMENT_TIME + (closestBotDistance / 1200) * (MAX_BASE_MOVEMENT_TIME - MIN_BASE_MOVEMENT_TIME);
		
		if(me.getOthers() != 0){
			return Math.max(1, baseMovementTime + Math.random() * enemyBulletTravelTime);
		} else {
			return 10 + Math.random() * 15;
		}
	}
	
	double pickStopTime(AbstractAdvancedBot me, EnvironmentInterface environment){
		double closestBotDistance = getEnemyDistance(me, environment);
		double enemyBulletTravelTime = getEnemyBulletTravelTime(me, environment);
		double baseMovementTime = MIN_BASE_MOVEMENT_TIME + (closestBotDistance / 1200) * (MAX_BASE_MOVEMENT_TIME - MIN_BASE_MOVEMENT_TIME);
		double stopProbability = DaveMath.scaledValue(MIN_STOP_PROBABILITY, MAX_STOP_PROBABILITY, 0, 1200, closestBotDistance);
		
		if(Math.random() < stopProbability){
			return DaveMath.random(0, DaveMath.scaledValue(MIN_STOP_TIME, MAX_STOP_TIME, 0, 1200, closestBotDistance));
		} else {
			return 0;
		}
	}
	
/////////////////////////////////////////////////////////////////////////////////////////
/////////////////// Destination Point to Movement Command Translation ///////////////////
/////////////////////////////////////////////////////////////////////////////////////////

	void goTo(Point p, AbstractAdvancedBot me, EnvironmentInterface environment) {
		double angleInDegrees = me.getLocation().absoluteAngleTo(p);
		if(!me.getLocation().equals(p)){
			double newDirection = turnTo(angleInDegrees, me, environment);
			double distanceToTravel = me.getLocation().distanceTo(p);
			me.setAhead(newDirection * distanceToTravel);
		}
	}
	
	double turnTo(double angleInDegrees, AbstractAdvancedBot me, EnvironmentInterface environment) {
		double ang;
		double dir;
		ang = DaveMath.angularDifferenceBetween(me.getHeading(), angleInDegrees);
		if (ang > DaveMath.QUARTERCIRCLE) {
			ang -= DaveMath.HALFCIRCLE;
			dir = -1;
		} else if (ang < -DaveMath.QUARTERCIRCLE) {
			ang += DaveMath.HALFCIRCLE;
			dir = -1;
		} else {
			dir = 1;
		}
		me.setTurnRight(ang);
		return dir;
	}
	
	double directionTo(double angleInDegrees, AbstractAdvancedBot me, EnvironmentInterface environment) {
		double ang;
		double dir;
		ang = DaveMath.angularDifferenceBetween(me.getHeading(), angleInDegrees);
		if (ang > DaveMath.QUARTERCIRCLE) {
			ang -= DaveMath.HALFCIRCLE;
			dir = -1;
		} else if (ang < -DaveMath.QUARTERCIRCLE) {
			ang += DaveMath.HALFCIRCLE;
			dir = -1;
		} else {
			dir = 1;
		}
		return dir;
	}
	
	public boolean isValid(AbstractAdvancedBot me, EnvironmentInterface environment){
		return me.getOthers() <= 1;
	}
	
	public String toString(){
		return "HybridDuel";
		
	}
}