package xander.core;

import java.awt.geom.Path2D;
import java.util.ArrayList;
import java.util.List;

import xander.core.gun.AutoFireCondition;
import xander.core.gun.power.FixedPowerSelector;
import xander.core.gun.power.PowerSelector;
import xander.core.math.RCPhysics;
import xander.gfws.BalancedFactorIndexer;
import xander.gfws.FactorIndexer;

/**
 * Container for framework configuration parameters.
 * 
 * @author Scott Arnold
 */
public class Configuration {

	private int snapshotHistorySize                      = 80;    // number of snapshots to store for each opponent in the history
	private boolean logEnemyBulletFiredStats             = false; // whether or not to log enemy bullet stats to standard out at the end of each round
	private boolean autoFireOnDisabledOpponents          = true;  // whether or not framework should automatically use a head-on gun to fire on disabled opponents
	private List<AutoFireCondition> autoFireConditions   = null;
	private int cpuUtilizationDataPoints                 = 200;   // number of ticks to remember for drawing CPU utilization graph
	private Double cpuConstantMS                         = null;  // CPU constant in milliseconds (used by CPU utilization graph)
	private boolean logComponentRunTimes                 = false; // whether or not to log CPU usage for each gun, drive, and radar to standard out at the end of each round
	private boolean logDriveTimes                        = false; // whether or not to log drive usage data to standard out at the end of each round
	private int rollingVirtualHitRatioDepth              = 40;    // number of data points to remember for the virtual hit ratios
	private PowerSelector disabledOpponentPowerSelector  = new FixedPowerSelector(RCPhysics.MIN_FIRE_POWER); // power selector to use for the head-on gun when auto firing on disabled opponents
	private boolean clearBattleStatsOnNewVersion         = true;  // whether or not to clear out the saved battle stats when a new version of this robot run 
	private boolean processNonFiringWaves                = false; 
	private PowerSelector nonFiringPowerSelector         = null;
	private boolean usePreciseMEAForMyWaves              = false; // whether or not to store opponents Precise MEA in waves generated by this robot's gun 
	private boolean usePreciseMEAForOpponentWaves        = false; // whether or not to store this robot's Precise MEA in waves generated by the opponent's gun
	private Path2D.Double myPreciseMEADriveBounds        = null;  // drive bounds for this robot used to calculate this robot's Precise MEA
	private Path2D.Double opponentPreciseMEADriveBounds  = null;  // drive bounds for opponent used to calculate opponent's Precise MEA
	private FactorIndexer opponentFactorIndexer          = new BalancedFactorIndexer(); // factor indexer for converting between factor angles and factor indexes for opponents
	private FactorIndexer myFactorIndexer                = new BalancedFactorIndexer(); // factor indexer for converting between factor angles and factor indexes for myself 

	public int getSnapshotHistorySize() {
		return snapshotHistorySize;
	}

	public void setSnapshotHistorySize(int snapshotHistorySize) {
		this.snapshotHistorySize = snapshotHistorySize;
	}

	public boolean isLogEnemyBulletFiredStats() {
		return logEnemyBulletFiredStats;
	}

	public void setLogEnemyBulletFiredStats(boolean logEnemyBulletFiredStats) {
		this.logEnemyBulletFiredStats = logEnemyBulletFiredStats;
	}

	public boolean isAutoFireOnDisabledOpponents() {
		return autoFireOnDisabledOpponents;
	}

	public void setAutoFireOnDisabledOpponents(boolean autoFireOnDisabledOpponents) {
		this.autoFireOnDisabledOpponents = autoFireOnDisabledOpponents;
	}

	public void setAutoFireConditions(List<AutoFireCondition> autoFireConditions) {
		this.autoFireConditions = autoFireConditions;
	}
	
	public void addAutoFireCondition(AutoFireCondition autoFireCondition) {
		if (autoFireConditions == null) {
			autoFireConditions = new ArrayList<AutoFireCondition>();
		}
		autoFireConditions.add(autoFireCondition);
	}
	
	public List<AutoFireCondition> getAutoFireConditions() {
		return autoFireConditions;
	}
	
	public boolean isUsePreciseMEAForMyWaves() {
		return usePreciseMEAForMyWaves;
	}

	public Path2D.Double getMyPreciseMEADriveBounds() {
		return myPreciseMEADriveBounds;
	}
	
	/**
	 * Sets whether or not to store precise MEA for self with the waves in
	 * the wave history.  Setting this to true causes additional processing,
	 * so only set this to true if you are going to use it.
	 * 
	 * @param usePreciseMEAForMyWaves
	 * @param preciseMEADriveBounds
	 */
	public void setUsePreciseMEAForMyWaves(boolean usePreciseMEAForMyWaves, Path2D.Double oppPreciseMEADriveBounds) {
		this.usePreciseMEAForMyWaves = usePreciseMEAForMyWaves;
		this.opponentPreciseMEADriveBounds = oppPreciseMEADriveBounds;
	}
	
	public boolean isUsePreciseMEAForOpponentWaves() {
		return usePreciseMEAForOpponentWaves;
	}

	public Path2D.Double getOpponentPreciseMEADriveBounds() {
		return opponentPreciseMEADriveBounds;
	}
	
	/**
	 * Sets whether or not to store precise MEA of opponent with the waves in
	 * the wave history.  Setting this to true causes additional processing,
	 * so only set this to true if you are going to use it.
	 * 
	 * @param usePreciseMEAForOpponentWaves
	 * @param opponentPreciseMEADriveBounds
	 */
	public void setUsePreciseMEAForOpponentWaves(boolean usePreciseMEAForOpponentWaves, Path2D.Double myPreciseMEADriveBounds) {
		this.usePreciseMEAForOpponentWaves = usePreciseMEAForOpponentWaves;
		this.myPreciseMEADriveBounds = myPreciseMEADriveBounds;
	}
	
	public PowerSelector getDisabledOpponentPowerSelector() {
		return disabledOpponentPowerSelector;
	}

	public void setDisabledOpponentPowerSelector(
			PowerSelector disabledOpponentPowerSelector) {
		this.disabledOpponentPowerSelector = disabledOpponentPowerSelector;
	}

	public int getRollingVirtualHitRatioDepth() {
		return rollingVirtualHitRatioDepth;
	}

	public void setRollingVirtualHitRatioDepth(int rollingVirtualHitRatioDepth) {
		this.rollingVirtualHitRatioDepth = rollingVirtualHitRatioDepth;
	}

	public boolean isLogComponentRunTimes() {
		return logComponentRunTimes;
	}

	public void setLogComponentRunTimes(boolean logComponentRunTimes) {
		this.logComponentRunTimes = logComponentRunTimes;
	}

	public boolean isLogDriveTimes() {
		return logDriveTimes;
	}

	public void setLogDriveTimes(boolean logDriveTimes) {
		this.logDriveTimes = logDriveTimes;
	}

	public int getCPUUtilizationDataPoints() {
		return cpuUtilizationDataPoints;
	}

	public void setCPUUtilizationDataPoints(int cpuUtilizationDataPoints) {
		this.cpuUtilizationDataPoints = cpuUtilizationDataPoints;
	}

	public Double getCpuConstantMS() {
		return cpuConstantMS;
	}

	public void setCpuConstantMS(Double cpuConstantMS) {
		this.cpuConstantMS = cpuConstantMS;
	}

	public boolean isClearBattleStatsOnNewVersion() {
		return clearBattleStatsOnNewVersion;
	}

	public void setClearBattleStatsOnNewVersion(boolean clearBattleStatsOnNewVersion) {
		this.clearBattleStatsOnNewVersion = clearBattleStatsOnNewVersion;
	}

	public boolean isProcessNonFiringWaves() {
		return processNonFiringWaves;
	}

	public void setProcessNonFiringWaves(boolean processNonFiringWaves) {
		this.processNonFiringWaves = processNonFiringWaves;
	}

	public PowerSelector getNonFiringPowerSelector() {
		return nonFiringPowerSelector;
	}

	public void setNonFiringPowerSelector(PowerSelector nonFiringPowerSelector) {
		this.nonFiringPowerSelector = nonFiringPowerSelector;
	}

	public FactorIndexer getMyFactorIndexer() {
		return myFactorIndexer;
	}

	public void setMyFactorIndexer(FactorIndexer factorIndexer) {
		this.myFactorIndexer = factorIndexer;
	}
	
	public FactorIndexer getOpponentFactorIndexer() {
		return opponentFactorIndexer;
	}

	public void setOpponentFactorIndexer(FactorIndexer factorIndexer) {
		this.opponentFactorIndexer = factorIndexer;
	}
}
