package jwst.DAD.Targeting;

import jwst.DAD.*;
import jwst.DAD.utils.*;
import jwst.Enemies.*;
import jwst.DAD.Targeting.WaveBullet;

import java.util.*;
import java.awt.geom.*;
import java.awt.*;

import robocode.util.*;
import robocode.*;

/**********************************************************************************************
 * 
 * Shooting: Guess Factor Targeting - it creates an array of data that tracks where the enemy
 * has moved in a set of movements and fires towards the area in which the enemy is most 
 * likely to move. It records the information in what could be called a virtual graph, creating
 * a set of data that eventually has a high point. That peak in the graph tells us where the
 * enemy is most likely to move and we fire in that location.
 * 
 * @author Jared Wong
 * @author Shawn Thomas
 * @period Period 2
 * @date June 10th, 2009
 * 
 * Credit: http://testwiki.roborumble.org/w/index.php?title=GuessFactor_Targeting_Tutorial
 * 
 * ********************************************************************************************/
public class GuessFactorTargeting
{
    /* **********************Guess-Factor Stuff**************************** */
    /**
     * a public variable to keep track of the number of hits I have against the specified enemy
     */
    public static double hits = 0;
    /**
     * a public variable to keep track of the number of times I have fired at a specified enemy
     */
    public static double totalFired = 0;
    // to hold the waves. nothing special.
    Vector<WaveBullet> waves = new Vector<WaveBullet>();
    Vector<WaveBullet> fastWaves = new Vector<WaveBullet>();
    
    // 47 bins are used because the max escape angle possible. possible.
    // is calculated by the arcsine( 8 / ( 20 - 3 * 3 ) )
    // it is the biggest angle possible, using the maximum bullet power possible. 3.
    // the angle. in degrees. is 46.658241772778. i wanted to use 1 bin per 2 angles. 
    private final static int BINS = 47;
    private final static int MIDDLE_BIN = (BINS - 1) / 2;

    // initiate the direction variable
    private static int direction = 1;
    
    // maximum distance i want to account for. thinking that the enemy will probably not
    // account for a different strategy any further and will then be the same
    // this distance segmentation is mainly for the purpose of different patterns/strategies
    // at different distances relative to me
    private final static int MAX_DISTANCE =  400;
    private final static int MAIN_DISTANCE_INDEXES = 4;
    private final static int DISTANCE_INDEXES = MAIN_DISTANCE_INDEXES + 1;
    
    // there are 9 absolute velocities: 0 1 2 3 4 5 6 7 8
    private final static double ABSOLUTE_VELOCITIES = 9;
    // to change how many absolute velocities per index change this
    private final static double MAX_ABSOLUTE_VELOCITIES_PER_INDEX = 2;
    // to find the number of velocity indexes needed take the least integer greater than
    // the number of absolute velocities divided by the number of absolute velocities per
    // index
    private final static int VELOCITY_INDEXES = 
        (int)Math.ceil(ABSOLUTE_VELOCITIES / MAX_ABSOLUTE_VELOCITIES_PER_INDEX);
    
    // for ease just make them the same
    private final static int PREVIOUS_VELOCITY_INDEXES = VELOCITY_INDEXES;
    
    // to keep track of the enemies velocity
    private static int enemyVelocityIndex;
    
    // either too close or in the middle
    private final static int WALL_DISTANCE = 60;
    private final static int MAIN_WALL_DISTANCE_INDEXES = 3;
    private final static int WALL_INDEXES = MAIN_WALL_DISTANCE_INDEXES + 1;
    
    private final static int numSegmentations = 4;
    private static int[] segmentationIndexes = new int[numSegmentations];
    
    private static double[][][][][] stats = new double[DISTANCE_INDEXES][VELOCITY_INDEXES][PREVIOUS_VELOCITY_INDEXES][WALL_INDEXES][BINS];
    private static double[] fastStats = new double[BINS];
    
    /* **************************Gun Variables***************************** */
    
    // 1.9 seems to be a popular bullet power among the top bots
    // ....those guys know what they're doing... I'll trust them.. :]
    // good number because we don't want to run out of energy too fast,
    // but still want to do some good damage
    private final static double DEFAULT_BULLET_POWER = 1.9;
    
    // as per robocode physics
    private final static double MAX_BULLET_POWER = 3.0;
    
    /* ************************Control Thy Self**************************** */
    
    // an instance of ourself, our robot
    private AdvancedRobot robot;

    /**
     * A constructor for a GFT gun
     * @param robot - An AdvancedEnemy to control
     */
    public GuessFactorTargeting( AdvancedRobot robot )//constructor
    {
        this.robot = robot;
    }

    /**
     * Heart of the GFT gun
     * @param e - AdvancedEnemy to fire at
     */
    public void fire( AdvancedEnemy e )
    {
        // loop through the waves and check a hit
        // if it did hit the enemy, remove it
        for ( int i = 0; i < waves.size(); i++ )
        {
            WaveBullet currentWave = waves.get( i );
            if ( currentWave.checkHit( e.getLocation(), robot.getTime() ) )
            {
                waves.remove( currentWave );
                i--;
            }
        }
        
        // basically the bearing from north(0) to the enemy
        double absBearing = robot.getHeadingRadians() + e.getBearingRadians();
        
        // finds the power of the bullet based on: 1) your current energy 
        //                                         2) your distance from the enemy robot
        double power = Math.min(e.getEnergy() / 4,(e.getDistance() < 100 ? 
                                                   MAX_BULLET_POWER  : DEFAULT_BULLET_POWER));
        
        //if our hit percentage is greater thean 55 percent and the
        //number of bullets fired is greater than 50, start using
        // max bullet power because we are hitting this robot
        //very often
        if(hits/totalFired > .55 && totalFired > 50)
            power = MAX_BULLET_POWER;
        
        // figure out the direction relative to you the enemy robot is traveling
        if ( e.getVelocity() != 0 )
        {
            direction = Math.sin( e.getHeadingRadians() - absBearing )
                * e.getVelocity() < 0 ? -1 : 1;
        }

        // find the distance index where the enemy robot is currently at when the wave hits
        int primaryDistanceIndex = (int)(e.getDistance() / MAX_DISTANCE / MAIN_DISTANCE_INDEXES );
        // for the implementation where I only care about different indexes until a certain distance
        int distanceIndex = primaryDistanceIndex < MAIN_DISTANCE_INDEXES ? 
                            primaryDistanceIndex : MAIN_DISTANCE_INDEXES;
        
        segmentationIndexes[0] = distanceIndex;
        
        // assign velocity indexes
        int previousVelocityIndex = enemyVelocityIndex;
        int velocityIndex = enemyVelocityIndex = 
            (int) Math.abs(e.getVelocity() / MAX_ABSOLUTE_VELOCITIES_PER_INDEX);
        
        segmentationIndexes[1] = velocityIndex;
        segmentationIndexes[2] = previousVelocityIndex;
        
        // figure out the distance away from the closest wall the enemy bot is
        double disFromWall = DADUtils.distanceFromWall( e.getLocation() );
        int primaryWallIndex = (int)(disFromWall / WALL_DISTANCE / MAIN_WALL_DISTANCE_INDEXES);
        int wallIndex = primaryWallIndex < MAIN_WALL_DISTANCE_INDEXES ?
                        primaryWallIndex : MAIN_WALL_DISTANCE_INDEXES;
        
        segmentationIndexes[3] = wallIndex;
        
        // set up the segmentations
        double[] currentStats = stats[distanceIndex][velocityIndex][previousVelocityIndex][wallIndex];
        
        // make the new wave based on the current standings of you and the enemy robot
        WaveBullet newWave = new WaveBullet( new Point2D.Double( robot.getX(), robot.getY() ),
                                             absBearing, power, direction, robot.getTime(),
                                             currentStats, stats, segmentationIndexes );
        
        WaveBullet fastWave = new WaveBullet( new Point2D.Double( robot.getX(), robot.getY() ),
                                             absBearing, power, direction, robot.getTime(),
                                             fastStats, stats, segmentationIndexes );
        
        int bestindex = MIDDLE_BIN; // initialize it to be in the middle,
                                    // guess factor 0.
        
        // find the best index to fire at
        if(!isEmpty(currentStats))
        {
            for ( int i = 0; i < BINS; i++ )
                if ( currentStats[bestindex] < currentStats[i] )
                    bestindex = i;
            System.out.println("using REGULAR stats for gun");
        }
        else
        {
            for ( int i = 0; i < BINS; i++ )
                if ( fastStats[bestindex] < fastStats[i] )
                    bestindex = i;
            System.out.println("using FAST stats for gun");
        }
        
        
        // this calculates the width of each bin
        double BIN_WIDTH = DADUtils.maxEscapeAngle(DADUtils.bulletVelocity(newWave.getPower()))/*newWave.maxEscapeAngle()*/ / MIDDLE_BIN;
            
        robot.setTurnGunRightRadians(Utils.normalRelativeAngle(
            // Face the gun towards enemy robot
            absBearing - robot.getGunHeadingRadians() +
            // then offset it to the bin angle by multiplying the width of each bin
            // by the number of bins over you should shoot
            (direction * BIN_WIDTH) * (bestindex - MIDDLE_BIN)));
        
        if ( robot.getGunHeat() == 0
            && Math.abs( robot.getGunTurnRemaining() ) < 5 )
        {
            newWave.setHasBullet( true );
            robot.setFireBullet( power );
            waves.add( newWave );
            totalFired++;
            fastWave.setHasBullet( true );
            fastWaves.add( fastWave );
        }
        else
        {
            newWave.setHasBullet( false );
            waves.add( newWave );
            fastWave.setHasBullet( false );
            fastWaves.add( fastWave );
        }
    }
    
    /**
     * Loops through a segment to see if it is empty
     * 
     * @param segment segment to check
     * @return true if it's empty, false if it's not
     */
    private boolean isEmpty(double[] segment) {
        for (int i = 0; i < segment.length; i++)
            if (segment[i] != 0D)
                return false;
        return true;
    }
    
    /**
     * paint method to be called for debugging and looking cool purposes
     * 
     * @param g - Graphics2D
     */
    public void onPaint(Graphics2D g)
    {
        g.setColor(Color.green);
        for(double i=1; i <= MAIN_DISTANCE_INDEXES; i++)
        {
            double radius = (i * MAX_DISTANCE / (MAIN_DISTANCE_INDEXES));
            int bottomX = (int)(robot.getX() - radius);
            int bottomY = (int)(robot.getY() - radius);
            int side = (int)(radius * 2);
            g.drawOval(bottomX, bottomY, side, side);
        }
        
        g.setColor( Color.yellow );
        for(double h = 1; h <= MAIN_WALL_DISTANCE_INDEXES; h++)
        {
            double distance = (h * WALL_DISTANCE / MAIN_WALL_DISTANCE_INDEXES);
            g.drawRect( (int)distance, (int)distance, 
                (int)(robot.getBattleFieldWidth() - distance * 2), 
                (int)(robot.getBattleFieldHeight() - distance * 2));
        }
        
        g.setColor(Color.blue);
        for(int j = 0; j < waves.size(); j++)
        {
            WaveBullet wave = waves.get( j );
            if(wave.getHasBullet())
            {
                double distance = ( robot.getTime() - wave.getFireTime() )* DADUtils.bulletVelocity(wave.getPower());
                g.drawOval( (int)(wave.getMyLocation().getX() - distance),
                            (int)(wave.getMyLocation().getY() - distance),
                            (int)(distance * 2), 
                            (int)(distance * 2 ));
            }
        }
        
        g.setColor( Color.red );
        for(int j = 0; j < waves.size(); j++)
        {
            WaveBullet wave = waves.get( j );
            if(!wave.getHasBullet())
            {
                double distance = ( robot.getTime() - wave.getFireTime() )* DADUtils.bulletVelocity(wave.getPower());
                g.drawOval( (int)(wave.getMyLocation().getX() - distance),
                            (int)(wave.getMyLocation().getY() - distance),
                            (int)(distance * 2), 
                            (int)(distance * 2 ));
            }
        }
    }

}