package sch;

/**
 * A library containing all math/statistic/trigonometic functions used by all bot's calculations
 */
public final class BotMath implements Constants {
	
	/**
	 * Returns the distance between two coordinates
	 * @param	p1		first point coordinate
	 * @param	p2		second point coordinate
	 * @return			distance
	 */
	public static double getRange (Point p1, Point p2){
		return getRange(p1.x, p1.y, p2.x, p2.y);
	}
	
	/**
	 * Returns the distance between two coordinates
	 * @param	x1		x coordinate of first point
	 * @param	y1		y coordinate of first point
	 * @param	x2		x coordinate of second point
	 * @param	y2		y coordinate of second point
	 * @return			distance
	 */
	public static double getRange (double x1, double y1, double x2, double y2){
	
		double dx=x2-x1;
		double dy=y2-y1;
		
		double h=Math.sqrt(dx*dx + dy*dy);
		return h;
	}
	
	/**
	 * Returns the absolute bearing between the two coordinates
	 * @param	p1		first point coordinate
	 * @param	p2		second point coordinate
	 * @return			angle in radians
	 */
	
	public static double absBearing (Point p1, Point p2){
		return absBearing(p1.x, p1.y, p2.x, p2.y);
	}
	
	/**
	 * Returns the absolute bearing between the two coordinates
	 * @param	x1		x coordinate of first point
	 * @param	y1		y coordinate of first point
	 * @param	x2		x coordinate of second point
	 * @param	y2		y coordinate of second point
	 * @return			angle in radians
	 */
	public static double absBearing(double x1,double y1, double x2,double y2 )
	{
		double dx=x2-x1;
		double dy=y2-y1;
		double h = getRange( x1,y1, x2,y2 );
		if( dx > 0 && dy > 0 )
		{
			return Math.asin( dx / h );
		}
		if( dx > 0 && dy < 0 )
		{
			return PI - Math.asin( dx / h );
		}
		if( dx < 0 && dy < 0 )
		{
			return PI + Math.asin( -dx / h );
		}
		if( dx < 0 && dy > 0 )
		{
			return 2.0*PI - Math.asin( -dx / h );
		}
		return 0;
	}
	
	/**
	 * Converts any angle (in radians) so that it is within the -PI to PI range
	 * @param	ang		angle in radians (any angle)
	 * @return			angle in radians (in -PI to PI)
	 */
	public static double normaliseBearing(double ang) {
		if (ang > PI)
			ang -= 2*PI;
		if (ang < -PI)
			ang += 2*PI;
		return ang;
	}
	
	/**
	 * Converts any angle (in radians) so that it is within the 0 to 2PI range
	 * @param	ang		angle in radians (any angle)
	 * @return			angle in radians (in 0 to 2PI)
	 */
	public static double normaliseHeading(double ang) {
		if (ang > 2*PI)
			ang -= 2*PI;
		if (ang < 0)
			ang += 2*PI;
		return ang;
	}
	
    /**
     * Convert a point in polar coordinate (range, bearing) to a cartesian coordinate (x,y)
	 * @param	range		distance of the point from the origin of axes
	 * @param	angle_rad	bearing of the point
	 * @return				(x,y) point position
     */	
	public static Point polar2Cartesian (double range, double angle_rad) {
		double x;
		double y;
		
		x=range*Math.sin(angle_rad);
		y=range*Math.cos(angle_rad);
		
		return new Point(x,y);
	}
	

    /**
     * Rounds a double to 2 decimal points
	 * @param	num		any double number
	 * @return			rounded number
     */
    public static double round2(double num) {
        return (double)Math.round(num * 100) / 100;
    }
    
	/**
	 * Returns the bullet velocity for a given firepower
	 * @param	firepower		power of the bullet
	 * @return					velocity of the bullet
	 */
	static double getBulletVelocity (double firePower) {
		return 20 - (3*firePower);
	}
	
	/**
	 * Calculate a percent ratio of a certain event
	 * @param	numEvents	number of events of the interesting events
	 * @param	totEvents	total number of events
	 * @return				ratio
	 */
	static double calcRatio (double numEvents, double totEvents) {
		return round2((numEvents/totEvents)*100);
	}
	
	
	static double sign(double value) {
		if (value<0)
			return -1.0;
		else
			return 1.0;
	}

	static double getBulletDamage(double bulletPower){
		double damege=4.0*bulletPower;
		if (bulletPower>1)
			damege+=2.0*(bulletPower-1.0);
		return damege;
	}
	
	static double getNeededFirePower(double neededDamage) {
	if 	(neededDamage<=4.0)
		return neededDamage/4.0;
	else
		return (neededDamage+2.0)/6.0;
	}
	
} 