package tzu;

import robocode.*;
import tzu.intel.*;
import tzu.strategy.*;
import tzu.util.*;
import tzu.radar.*;
import tzu.gun.*;
import tzu.movement.*;

/**
 * Provides an abstract advanced robot with many useful features.<p>
 * Copyright:   Copyright (c) 2001-2002.<br>
 * Company:     Rideau Code Works Inc.<br>
 *
 * You are given permission to copy and modify any code in the tzu package
 * provided:
 * <ul>
 *      <li> you distribute your robot as open source and...</li>
 *      <li> you identify which parts have been copied and...</li>
 *      <li> you include similar requirements with your source and...</li>
 *      <li> if you copy the majority of the code and only make small
 *           changes (effectively producing a variant) you indicate
 *           it as such by keeping the package name the same and...</li>
 *      <li> send me an email!  I like to hear from other
 *           robocode enthusiasts!</li>
 * </ul>
 *
 * @author      Ray Vermette
 * @version     0.1
 */
public abstract class AbstractAdvancedRobot
        extends AdvancedRobot implements Constants {

    static BattleField              battleField             = null;
    static RadarManager             radarManager            = null;
    static GunManager               gunManager              = null;
    static EnemyManager             enemyManager            = null;
    static TargetStrategy           targetStrategy          = null;
    static MovementStrategy         movementStrategy        = null;
    static EnemyBulletManager       enemyBulletManager      = null;
    static AntiGravityManager       antiGravityManager      = null;

    /** Main method called in every round of battle. */
    public void run() {

        initialize();

        while (true) {

            enemyManager.takeTurn();
            targetStrategy.takeTurn();
            radarManager.takeTurn();
            enemyBulletManager.takeTurn();
            antiGravityManager.takeTurn();
            movementStrategy.takeTurn();
            gunManager.takeTurn();
        }
    }

    /** Initialize this robot. */
    void initialize() {

        if (getRoundNum() == 0) {
            initializeBattle();
        } else {
            reinitialize();
        }

        setAdjustGunForRobotTurn(true);
        setAdjustRadarForGunTurn(true);
        setPriorities();
    }


    /** Initialization done only once per battle. */
    void initializeBattle() {

        battleField = new BattleField(this);
        initializeEnemyManager();
        initializeTargetStrategy();
        initializeRadarManager();
        initializeGunManager();
        initializeEnemyBulletManager();
        initializeAntiGravityManager();
        initializeMovementStrategy();
    }

    /** Initialize the enemy manager. */
    void initializeEnemyManager() {
        enemyManager = new EnemyManager(this);
    }


    /** Initialize the target strategy. */
    void initializeTargetStrategy() {
        targetStrategy = new TargetStrategy(this, enemyManager);
    }

    /** Initialize the radar manager. */
    void initializeRadarManager() {
        radarManager = new RadarManager(this, targetStrategy);
    }

    /** Initialize the movement strategy. */
    void initializeMovementStrategy() {
        movementStrategy = new MovementStrategy(
                this,
                enemyManager,
                targetStrategy,
                antiGravityManager);
    }

    /** Initialize the gun manager. */
    void initializeGunManager() {
        gunManager = new GunManager(this, targetStrategy);
    }


    void initializeEnemyBulletManager() {
        enemyBulletManager  = new EnemyBulletManager(this, targetStrategy);
    }


    void initializeAntiGravityManager() {
        antiGravityManager  = new AntiGravityManager(
                this, enemyManager, enemyBulletManager, targetStrategy);
    }


    /** Initialization done in second and subsequent rounds. */
    void reinitialize() {
        enemyManager.reinitialize();
        targetStrategy.reinitialize();
        radarManager.reinitialize();
        movementStrategy.reinitialize();
        gunManager.reinitialize();
        enemyBulletManager.reinitialize();
        antiGravityManager.reinitialize();
    }

    /** Set event priorities. */
    void setPriorities() {

        /* We don't care about these events. */
        setEventPriority("BulletMissedEvent", 0);
        setEventPriority("HitWallEvent", 0);

        /* This one is not as important. */
        setEventPriority("BulletHitBulletEvent", 1);
    }

    /** Pass the event to various managers for further processing. */
    public void onScannedRobot(ScannedRobotEvent e) {


        Bot b = enemyManager.get(e.getName());
        if (b != null) {

            double points = b.getEnergy() - e.getEnergy();


            /*
             * Take into account wall damage.
             */
            if (getOthers() < 2) {

                double distance = e.getDistance();
                double speed    = e.getVelocity();
                double damage   = BotMath.calcWallDamage(b.getSpeed());
                double heading  = BotMath.zeroTo360(getHeading() +  e.getBearing());
                double x        = BotMath.calcX(getX(), heading, distance);
                double y        = BotMath.calcY(getY(), heading, distance);

                if (speed == 0.0                    &&
                    b.getSpeed() > 2.0              &&
                    points - damage >= 0            &&
                   (x < BattleField.getMinX() + 1   ||
                    y < BattleField.getMinY() + 1   ||
                    x > BattleField.getMaxX() - 1   ||
                    y > BattleField.getMaxY() - 1)) {

                    points = points - BotMath.calcWallDamage(b.getSpeed());
                }
            }


            /*
             * A point drop in the enemy might indicate a bullet being fired.
             * Add bullets to the enemyBulletManager.
             */
            if (points > 0.0                        &&
                points <= MAX_BULLET_POWER          &&
               (points != b.getPrevHitDamage()      ||
                e.getTime() != b.getPrevHitTime())) {

                enemyBulletManager.addBullets(b,
                        BotMath.min(
                        points, MAX_BULLET_POWER));
            }
        }

        enemyManager.onScannedRobot(e);
        radarManager.onScannedRobot(e);
    }

    /** Pass the event to various managers for further processing. */
    public void onRobotDeath(RobotDeathEvent e) {
        enemyManager.onRobotDeath(e);
    }

    /** Pass the event to various managers for further processing. */
    public void onBulletHit(BulletHitEvent e) {
        enemyManager.onBulletHit(e);
    }

    /** Pass the event to various managers for further processing. */
    public void onHitByBullet(HitByBulletEvent e) {
        enemyManager.onHitByBullet(e);
    }

    public void onBulletHitBullet(BulletHitBulletEvent e) {
        enemyBulletManager.removeBullet(e.getHitBullet());
    }

    /** Pass the event to various managers for further processing. */
    public void onHitRobot(HitRobotEvent e) {
        enemyManager.onHitRobot(e);
        movementStrategy.onHitRobot(e);
    }

    /** Pass the event to various managers for further processing. */
    public void onCustomEvent(CustomEvent e) {
    }

//    /** Print a debugging message to the console. */
//    public void onSkippedTurn(SkippedTurnEvent e) {
//        out.println("Missed turn!");
//    }

    /** Pass the event to various managers for further processing. */
    public void onDeath(DeathEvent e) {
        enemyManager.onDeath(e);
    }

    /** Pass the event to various managers for further processing. */
    public void onWin(WinEvent e) {
        enemyManager.onWin(e);
    }
}
