/*
 * LinearEnemyBullet - utility class
 * Copyright (C) 2002  Joachim Hofer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * You can contact the author via email (qohnil@johoop.de) or write to
 * Joachim Hofer, Feldstr. 12, D-91052 Erlangen, Germany.
 */

package qohnil.blot;

import robocode.AdvancedRobot;

import java.io.Serializable;

import qohnil.util.Coords;
import qohnil.util.BotMath;
import qohnil.util.Polar;

public class LinearEnemyBullet extends VirtualBullet implements Serializable {

    static double linearFactor = 1.0;
    private static double reliability = 1.0;

    double directAngle = 0.0;
    double linearAngle = 0.0;

    public LinearEnemyBullet(double angle, double linearAngle, double firePower, Coords coords,
                              long time, Coords targetCoords) {

        super(angle, firePower, coords, time);

        Polar p = BotMath.coordsToPolar(targetCoords, coords);
        directAngle = p.getBearing();
        this.linearAngle = linearAngle;
        hitTime = time + Math.round(BotMath.getBulletFlightTime(firePower, p.getDistance()));
    }

    /**
     * Should be called when a bullet is fired at us...
     */
    public LinearEnemyBullet(double firePower, Coords coords, long time, Coords targetCoords,
                              double heading, double velocity) {

        this(getLinearPredictionAngle(
                    targetCoords, coords, heading, velocity, firePower, linearFactor),
             getLinearPredictionAngle(
                    targetCoords, coords, heading, velocity, firePower, 1.0),
                firePower, coords, time, targetCoords);
    }


    public double getLinearFactor() { return linearFactor; }

    /**
     * Should be called when a bullet hits us...
     */
    private void adaptLinearFactor(double realAngle) {

        // if linear and direct predictions are too near to each other,
        // don't adapt
        if (Math.abs(BotMath.normalizeRelativeAngle(linearAngle - directAngle)) < 0.001) {
            System.out.println("linear = direct angle, don't adapt");

            return;
        }

        // approximation for small angles: angle proportional to linearFactor
        double factor = BotMath.normalizeRelativeAngle(realAngle - directAngle)
                / BotMath.normalizeRelativeAngle(linearAngle - directAngle);

        // adapt previous linear factor
        linearFactor = linearFactor * 0.67 + factor * 0.33;

        System.out.println("dodging adapted:");
        System.out.println("pred. angle: " + heading + ", real angle: " + realAngle);
        System.out.println("direct angle: " + directAngle + ", linear angle: " + linearAngle);
        System.out.println("=> linearFactor = " + linearFactor);
    }

    /**
     * Should be called when a bullet hits us...
     */
    public void adaptReliability(double realAngle) {
        super.adaptReliability(realAngle);
        adaptLinearFactor(realAngle);
    }


    public static double getLinearPredictionAngle(Coords target, Coords src,
                                                  double heading, double velocity,
                                                  double firePower, double factor) {
        Coords predictedCoords = target;
        velocity *= factor;
        for (int i = 0; i < 3; i++) {
            long predictedTime = Math.round(BotMath.getBulletFlightTime(firePower, predictedCoords.distance(src)));
            predictedCoords = MovableData.getLinearPredictedCoords(predictedTime, predictedCoords, heading, velocity);
        }
        return BotMath.coordsToPolar(predictedCoords, src).getBearing();
    }

    public void setReliability(double reliability) {
        LinearEnemyBullet.reliability = reliability;
    }

    public double getReliability() {
        return LinearEnemyBullet.reliability;
    }


}
