package fromHell.movement;

import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Iterator;

import fromHell.utils.FHUtils;

import static fromHell.utils.FHUtils.*;

import robocode.*;
import robocode.util.Utils;


public class WaveSurfing {
    public static final double WALLSTICK = 160;
    static ArrayList<Segmentations> _hitSegmentations = new ArrayList<Segmentations>();
    public static Point2D.Double _myLocation;
    public static Point2D.Double _enemyLocation;
    public static ArrayList<EnemyWave> _enemyWaves;
    public static ArrayList<EnemyWave> _pastWaves;
    public static EnemyWave _surfWave;
    public static double _enemyEnergy = 100.0;
    public static Rectangle2D _battlefieldRectangle;
    static double _myLastAbsoluteLateralVelocity;
    static AdvancedRobot _robot;

    public WaveSurfing(AdvancedRobot robot) {
	_robot = robot;
	_enemyWaves = new ArrayList<EnemyWave>();
	_pastWaves = new ArrayList<EnemyWave>();
	_battlefieldRectangle = FHUtils.getBattlefieldRectangle(robot);
    }

    public void onScannedRobot(ScannedRobotEvent scannedRobotEvent) {
	EnemyWave enemyWave = new EnemyWave();
	double enemyBearing = scannedRobotEvent.getBearingRadians();
	double myHeading = _robot.getHeadingRadians();
	double absoluteBearing = enemyBearing + myHeading;
	double directAngle = absoluteBearing + Math.PI;
	double enemyDistance = scannedRobotEvent.getDistance();
	double myVelocity = _robot.getVelocity();
	double myLateralVelocity = myVelocity * Math.sin(enemyBearing);
	double myAbsoluteLateralVelocity = Math.abs(myLateralVelocity);
	int myDirection = sign(myLateralVelocity);
	double myAdvancingVelocity = myVelocity * Math.cos(enemyBearing);
	double enemyEnergy = scannedRobotEvent.getEnergy();
	double enemyDeltaEnergy = _enemyEnergy - enemyEnergy;
	_myLocation = new Point2D.Double(_robot.getX(), _robot.getY());
	_enemyLocation = project(_myLocation, absoluteBearing, enemyDistance);
	enemyWave.direction = myDirection;
	enemyWave.directAngle = directAngle;
	enemyWave.setSegmentation(enemyDistance, myAdvancingVelocity, _myLastAbsoluteLateralVelocity, myAbsoluteLateralVelocity);
	_pastWaves.add(0, enemyWave);
	addWave(enemyDeltaEnergy);
	updateWaves();
	if (checkDanger(-myDirection, myVelocity, myHeading) < checkDanger(myDirection, myVelocity, myHeading)) {
	    myDirection = -myDirection;
	}
	double goAngle = wallSmoothing(_myLocation, directAngle + myDirection * offset(enemyDistance), myDirection) - myHeading;
	_robot.setAhead(Math.cos(goAngle) * Double.POSITIVE_INFINITY);
	_robot.setTurnRightRadians(Math.tan(goAngle));
	_enemyEnergy = enemyEnergy;
	_myLastAbsoluteLateralVelocity = myAbsoluteLateralVelocity;
    }

    public void onBulletHitBullet(BulletHitBulletEvent bulletHitBulletEvent) {
	logEnemyBullet(bulletHitBulletEvent.getHitBullet());
    }

    public void onHitByBullet(HitByBulletEvent hitByBulletEvent) {
	logEnemyBullet(hitByBulletEvent.getBullet());
	_enemyEnergy += Rules.getBulletHitBonus(hitByBulletEvent.getBullet().getPower());
    }

    public void onBulletHit(BulletHitEvent bulletHitEvent) {
	_enemyEnergy -= Rules.getBulletDamage(bulletHitEvent.getBullet().getPower());
    }
    
    public void onWin(WinEvent winEvent) {
	_robot.setAhead(0);
	for (int i = 0; i < 50; i++) {
	    _robot.turnRight(30);
	    _robot.turnLeft(30);
	}
    }

    public static void addWave(double deltaEnergy) {
	if (_pastWaves.size() > 2 && deltaEnergy <= Rules.MAX_BULLET_POWER && deltaEnergy >= Rules.MIN_BULLET_POWER) {
	    EnemyWave enemyWave = _pastWaves.get(2);
	    enemyWave.bulletVelocity = FHUtils.bulletVelocity(deltaEnergy);
	    enemyWave.fireLocation = _enemyLocation;
	    _enemyWaves.add(enemyWave);
	}
    }

    public static void updateWaves() {
	double minimumDistance = Double.NEGATIVE_INFINITY;
	_surfWave = null;
	Iterator<EnemyWave> iterator = _enemyWaves.iterator();
	while (iterator.hasNext()) {
	    EnemyWave enemyWave = iterator.next();
	    enemyWave.distanceTraveled += enemyWave.bulletVelocity;
	    double distance = enemyWave.distanceTraveled - _myLocation.distance(enemyWave.fireLocation);
	    if (distance > 50) {
		iterator.remove();
		continue;
	    }
	    if (distance > minimumDistance && distance < -WALL_MARGIN) {
		_surfWave = enemyWave;
		minimumDistance = distance;
	    }
	}
    }

    public static double getFactor(EnemyWave enemyWave, Point2D.Double targetLocation) {
	return Utils.normalRelativeAngle(absoluteBearing(enemyWave.fireLocation, targetLocation) - enemyWave.directAngle) * enemyWave.direction / Math.asin(Rules.MAX_VELOCITY / enemyWave.bulletVelocity);
    }

    public static void logEnemyBullet(Bullet bullet) {
	Point2D.Double hitLocation = new Point2D.Double(bullet.getX(), bullet.getY());
	Iterator<EnemyWave> iterator = _enemyWaves.iterator();
	while (iterator.hasNext()) {
	    EnemyWave enemyWave = iterator.next();
	    if (Math.abs(enemyWave.distanceTraveled - hitLocation.distance(enemyWave.fireLocation)) <= 40) {
		enemyWave.segmentations.GuessFactor = getFactor(enemyWave, hitLocation);
		_hitSegmentations.add(enemyWave.segmentations);
		iterator.remove();
	    }
	}
    }

    public double checkDanger(int direction, double myVelocity, double myHeading) {
	Point2D.Double predictedPosition = _myLocation;
	double predictedVelocity = myVelocity;
	double predictedHeading = myHeading;
	double moveAngle;
	double moveDirection;
	double danger = 1;
	int counter = 0;
	do {
	    moveAngle = wallSmoothing(predictedPosition, absoluteBearing(_enemyLocation, predictedPosition) + (direction * (offset(predictedPosition.distance(_enemyLocation)))), direction) - predictedHeading;
	    moveDirection = 1;
	    if (Math.cos(moveAngle) < 0) {
		moveAngle += Math.PI;
		moveDirection = -1;
	    }
	    predictedHeading = Utils.normalRelativeAngle(predictedHeading + minMax(Utils.normalRelativeAngle(moveAngle), Rules.getTurnRateRadians(Math.abs(predictedVelocity))));
	    if (predictedVelocity * moveDirection < 0) {
		predictedVelocity += DECELERATION_RATE * moveDirection;
	    } else {
		predictedVelocity += ACCELERATION_RATE * moveDirection;
	    }
	    predictedVelocity = minMax(predictedVelocity, Rules.MAX_VELOCITY);
	    predictedPosition = project(predictedPosition, predictedHeading, predictedVelocity);
	} while (_surfWave != null && predictedPosition.distance(_surfWave.fireLocation) - WALL_MARGIN > _surfWave.distanceTraveled + ((++counter) * _surfWave.bulletVelocity));
	if (_surfWave != null) {
	    danger = getDanger(getFactor(_surfWave, predictedPosition), _surfWave.segmentations);
	}
	return danger / predictedPosition.distanceSq(_enemyLocation);
    }

    final static double getDanger(double guessFactor, Segmentations currentSegmentation) {
	double danger = 0;
	Iterator<Segmentations> iterator = _hitSegmentations.iterator();
	while (iterator.hasNext()) {
	    Segmentations hitSegmentation = iterator.next();
	    double distance = 0;
	    distance += Math.abs(currentSegmentation.AbsoluteLateralVelocitySegment - hitSegmentation.AbsoluteLateralVelocitySegment);
	    distance += Math.abs(currentSegmentation.AdvancingVelocitySegment - hitSegmentation.AdvancingVelocitySegment);
	    distance += Math.abs(currentSegmentation.DistanceSegement - hitSegmentation.DistanceSegement);
	    distance += Math.abs(currentSegmentation.LastAbsoluteLateralVelocitySegment - hitSegmentation.LastAbsoluteLateralVelocitySegment);
	    danger += 1 / distance / (0.02 + Math.abs(guessFactor - hitSegmentation.GuessFactor));
	}
	return danger;
    }

    public static double offset(double distance) {
	return (Math.PI / 2 - 0.4) + (0.4 / 475) * distance;
    }

    public static double wallSmoothing(Point2D.Double botLocation, double angle, int orientation) {
	do {
	    angle += orientation * 0.05;
	} while (!_battlefieldRectangle.contains(project(botLocation, angle, WALLSTICK)));
	return angle;
    }
}