package tcf;
import robocode.*;
import java.awt.Graphics2D;
import java.awt.Color;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.LinkedList;

//==============================================================================
//
//	Class REACH
//
//==============================================================================

/**
 * Reach - a robot by Tim Foden
 */
public class Reach extends AdvancedRobot
{
	public final static boolean	TC_DODGE = false;
	public final static boolean	TC_SHOOT = false;

	//==========================================================================

	public Opponent	getTarget()
	{
		return m_target;
	}

	//==========================================================================
	
	public Vec2d	location()
	{
		return new Vec2d(getX(), getY());
	}

	//==========================================================================

	/**
	 * run: Reach's default behavior
	 */
	public void run()
	{
		setColors( Color.yellow, Color.magenta, Color.blue );
		
		setAdjustGunForRobotTurn( true );
		setAdjustRadarForGunTurn( true );
		setAdjustRadarForRobotTurn( true );

		setTurnRadarRight( Double.POSITIVE_INFINITY );

		while(true)
		{
			execute();

			doUpdates();
			doTargetting();
			if( !TC_DODGE )
				doFire();
			doRadarCheck();
			if( !TC_SHOOT )
				doMovement();
		}
	}

	//==========================================================================

	private void doUpdates()
	{
		// call update tick for all opponents.
		for( Enumeration<Opponent> e = m_opponents.elements(); e.hasMoreElements(); )
		{
			Opponent opp = e.nextElement();
			opp.doTickUpdate();
		}
	}

	//==========================================================================

	private void doTargetting()
	{
		if( m_target == null )
			return;

		boolean fired = m_target.doTickTargetUpdate();
		if( fired )
		{
			// fire all virtual guns.
			// for now, we just fire the ones on the target.
			m_target.fireVirtualGuns();
		}
	}

	//==========================================================================

	private void doFire()
	{
//		if( m_target == null )
//			return;
	}

	//==========================================================================

	private void	doRadarCheck()
	{
		// let a full sweep occur before we start to optimise.
		if( getTime() > 8 )
		{
			// find oldest scan of alive robot.  
			Opponent	oldest = null;
			for( Enumeration<Opponent> e = m_opponents.elements(); e.hasMoreElements(); )
			{
				Opponent	opp = e.nextElement();
				
				if( opp.m_isAlive &&
					(oldest == null ||
					 oldest.m_curScan.getTime() > opp.m_curScan.getTime()) )
				{
					oldest = opp;
				}
			}

			if( oldest != null )
			{
				double	oppHeading = oldest.m_curPos.minus(location()).getHeading();
				double	radarHeading = getRadarHeading();
				double	angle = Utils.normAngle180(oppHeading - radarHeading);
				angle += Math.signum(angle) * 22.5;
				if( oldest.m_curScan.getTime() + 8 < getTime() )
					angle = 360;
				setTurnRadarRight( angle );
			}
			else
				setTurnRadarRight( 360 );
		}
	}

	//==========================================================================

	private void	doMovement()
	{
		// movement.
		if( getDistanceRemaining() == 0 )//&& getVelocity() == 0 )
		{
			int	range = 301;//((getRoundNum() % 3) / 10) * 100 + 201;
/*			if( m_target != null && (getRoundNum() / 10) % 2 == 0 )
			{
				if( m_target.m_curScan.getDistance() > 350 )
					range = 401;
				else if( m_target.m_curScan.getDistance() > 700 )
					range = 501;
			}*/
			double distToMove = Utils.rand(range) - range / 2;
			setAhead( distToMove );
		}


		{
			double	heading = getHeading();
			if( getDistanceRemaining() < 0 )
				heading += 180;
			double	da = calcDistAhead(location(), heading);
				
			if( da < 50 )
				setAhead( -getDistanceRemaining() );
		}

		if( m_target != null )
		{
			Vec2d	delta	= m_target.m_curPos.minus(location());
			double	heading = delta.getHeading() + 90;
			double	turn	= Utils.normAngle180(heading - getHeading());
			
			double	dist = m_target.m_curScan.getDistance();
			if( dist < 100 )
			{
				double	bearing = m_target.m_curScan.getBearing();
//				System.out.println( "close target bearing " + bearing );
				if( Math.abs(bearing) < 90 )
				{
//					System.out.println( "backing up" );
					setAhead( -50 );
					turn = Utils.normAngle180(turn - 45);
				}
				else
				{
//					System.out.println( "moving forwards" );
					setAhead( 50 );
					turn = Utils.normAngle180(turn + 45);
				}
			}
			else //if( dist < 450 )
			{

				double	da = calcDistAhead(location(), getHeading());
				double	db = calcDistAhead(location(), getHeading() + 180);

				double	dr = getDistanceRemaining();
				if( da + db < 250 )
				{
					dr = -dr;
					if( dr < 0 )
						turn = Utils.normAngle180(turn - 15);
					else
						turn = Utils.normAngle180(turn + 15);
				}
/*
				double	dr  = getDistanceRemaining();
				double	dir = dr < 0 ? -1 : 1;
				Vec2d	dv  = Vec2d.fromHeading(getHeading() + turn);
				Vec2d	me  = location();
				Vec2d	pos = me.plus(dv.mult(dir * 150));
				double	buffer = 60;

				if( pos.dist(0, 0) < buffer )
				{
					pos = pos.plus(me.unit().mult(buffer));
				}
				else if( pos.dist(0, getBattleFieldHeight()) < buffer )
				{
					pos = pos.plus(me.minus(0, getBattleFieldHeight()).unit().mult(buffer));
				}
				else if( pos.dist(getBattleFieldWidth(), 0) < buffer )
				{
					pos = pos.plus(me.minus(getBattleFieldWidth(), 0).unit().mult(buffer));
				}
				else if( pos.dist(getBattleFieldWidth(), getBattleFieldHeight()) < buffer )
				{
					pos = pos.plus(me.minus(getBattleFieldWidth(),
											 getBattleFieldHeight()).unit().mult(buffer));
				}
				
				buffer = 45;
				if( pos.x < buffer || pos.y < buffer ||
					pos.x > getBattleFieldWidth()  - buffer ||
					pos.y > getBattleFieldHeight() - buffer )
				{
					// need to turn away from wall.
					if( pos.x < buffer )
						pos.x = buffer;
					else if( pos.x > getBattleFieldWidth()  - buffer )
						pos.x = getBattleFieldWidth()  - buffer;
				
					if( pos.y < buffer )
						pos.y = buffer;
					else if( pos.y > getBattleFieldHeight() - buffer )
						pos.y = getBattleFieldHeight() - buffer;
				}

				double	newHeading = pos.minus(location()).getHeading();
				if( dr < 0 )
					newHeading += 180;
				turn = Utils.normAngle180(newHeading - getHeading());
*/
			}

			setTurnRight( turn );
		}
	}

	//==========================================================================

	public void	predictMyFuturePosition( double futureTime, Vec2d pos )
	{
		double	remaining = Math.abs(getDistanceRemaining());
		
//		System.out.printf( "rem %g  vel %g  delta-t %g....\n",
//				getDistanceRemaining(), getVelocity(), futureTime );

		Vec2d	me  = location();
		Vec2d	dir = Vec2d.fromHeading(getHeading());

		// distance is covered by manipulating the velocity.  the velocity goes
		// through 3 stages: acceleration, coasting, deceleration.

		double	spd = getVelocity();
		if( spd < 0 )
		{
			spd = -spd;
			dir = dir.negate();
		}

		final double ACCEL = MyRules.ACCELERATION;
		final double DECEL = MyRules.DECELERATION;

//		System.out.printf( "predict future pos: time %g   cur spd %g\n",
//			futureTime, spd );
		
		double	totalDist = 0;
//		System.out.printf( "future time %g  dist %g   remaining %g\n",
//			futureTime, totalDist, remaining );
			
		// time and distance to stop:
		double	stopTime = spd / DECEL;
		double	stopDist = spd * stopTime - 0.5 * DECEL * stopTime * stopTime;

		if( stopDist < remaining && spd < MyRules.MAX_VELOCITY )
		{
			// in acceleration phase.
			// there are 2 possiblilities...
			// 1. we get up to full speed, and then declerate from full speed.
			// 2. we have to start declerating before we get to full speed.
			//
			// max spd (v): v = As / u(A-D)  where A == accel, D == decel (D is -ve)
			//double	v = ACCEL * remaining / (spd * (ACCEL + DECEL));
			// -- the above equation is absolute crap... how did I work it out?!
			//
			// more correct is the following...
			// v = sqrt((2ADs + Du^2) / (D - A))
			double	num = 2 * ACCEL * (-DECEL) * remaining + (-DECEL) * spd * spd;
			double	den = (-DECEL) + ACCEL;
			double	v = Math.sqrt(num / den);
			
			// clamp v to max speed.
			v = Math.min(v, MyRules.MAX_VELOCITY);

			// work out how far we go.
			double	dist = (v * v - spd * spd) / (2 * ACCEL);
			double	accelTime = Math.min(v - spd, futureTime);
//			System.out.printf( "accel dist %g  time %g\n", dist, accelTime );
			totalDist += dist;
			remaining -= dist;
			futureTime -= accelTime;
			spd = v;

			// update stop time and distance.
			stopTime = spd / DECEL;
			stopDist = spd * stopTime - 0.5 * DECEL * stopTime * stopTime;
		}
	
		if( spd == MyRules.MAX_VELOCITY )
		{
			// in crusing phase.
			double	cruiseDist = remaining - stopDist;
			double	cruiseTime = Math.min(cruiseDist / spd, futureTime);
			double	dist = spd * cruiseTime;
//			System.out.printf( "cruise dist %g  time %g\n", dist, cruiseTime );
			totalDist += dist;
			remaining -= dist;
			futureTime -= cruiseTime;
		}
		
		if( stopDist >= remaining - 0.01 )
		{
			// in deceleration phase.
			futureTime = Math.min(futureTime, stopTime);
			double	dist = spd * futureTime - 0.5 * DECEL * futureTime * futureTime;
//			System.out.printf( "decel dist %g\n", dist );
			dist = Math.max(dist, 0);
			totalDist += dist;
		}

		pos.set( me.plus(dir.mult(totalDist)) );

//		System.out.printf( "    dist %g\n", totalDist );
		pos.set( me );
	}

	//==========================================================================
	/**
	 * onScannedRobot: What to do when you see another robot
	 */
	public void onScannedRobot(ScannedRobotEvent e)
	{
		Opponent	opp = m_opponents.get(e.getName());
		if( opp == null )
		{
			// found new opponent.
			opp = new Opponent(this);
			opp.m_name = e.getName();
			
			m_opponents.put( opp.m_name, opp );
			m_aliveCount++;
		}
		else if( !opp.m_isAlive )
		{
			// new round of a match... opponent is alive again!  :)
			opp.m_isAlive = true;
			opp.m_prevScan = null;
			opp.m_curScan = null;
			opp.m_prevPos = null;
			opp.m_curPos = null;
			m_aliveCount++;
		}
	
		opp.onScannedRobot( e );
		
		m_scanCount++;

		if( m_target == null ||
			m_target.m_name.equals(e.getName()) ||
			m_target.m_curScan.getDistance() > e.getDistance() )
		{
			m_target = opp;
			m_lastTargetTime = getTime();
			onTargetUpdated();
//			System.out.printf( "targ %s  time %d   dist %g\n", e.getName(), e.getTime(), e.getDistance() );
		}
	}

	//==========================================================================

	private void onTargetUpdated()
	{
		if( m_target == null )
			return;

		m_target.matchPattern( getTime() );
	}

	//==========================================================================
	/**
	 * onHitByBullet: What to do when you're hit by a bullet
	 */
	public void onHitByBullet(HitByBulletEvent e)
	{
		//turnLeft(90 - e.getBearing());
	}

	//==========================================================================

	public void onBulletHit( BulletHitEvent e )
	{
		Opponent	opp = m_opponents.get(e.getName());
		if( opp == null )
		{
			// we haven't scanned this bot yet... ignore it for now.
			//	(this should never happen)
			return;
		}

		opp.onBulletHit( e );
	}

	//==========================================================================

	public void onSkippedTurn( SkippedTurnEvent e )
	{
		System.out.printf( "!!!!! SKIPPED TURN !!!!! time: %d\n", getTime() );
	}

	//==========================================================================

	public void onRobotDeath( RobotDeathEvent e )
	{
		Opponent	opp = m_opponents.get(e.getName());
		if( opp != null )
		{
			opp.m_isAlive = false;
			m_aliveCount--;
			
			opp.onRobotDeath( e );
		}

		if( m_target == opp )
			m_target = null;
	}

	//==========================================================================

	public void onDeath( DeathEvent evt )
	{
		// we died... clean up opponent info.
		for( Enumeration<Opponent> e = m_opponents.elements(); e.hasMoreElements(); )
		{
			Opponent	opp = e.nextElement();
			opp.onDeath( evt );
		}
	}

	//==========================================================================

	public void onPaint( Graphics2D g )
	{
		Color[]	c = new Color[] { Color.cyan, Color.magenta, Color.magenta, Color.yellow,
								  Color.green, Color.white, Color.white, Color.white,
								  Color.white };
		Enumeration<Opponent> e = m_opponents.elements();
		for( Opponent opp; e.hasMoreElements(); )
		{
			opp = e.nextElement();
			if( !opp.m_isAlive )
				continue;

			double	t2 = opp.m_curScan.getTime();
			double	t1 = opp.m_prevScan == null ? t2 : opp.m_prevScan.getTime();

			for( VirtualBullet b : opp.m_bullets )
			{
				Vec2d	p1 = b.positionAtTime(t1);
				int		x1 = (int)Math.round(p1.x);
				int		y1 = (int)Math.round(p1.y);
				Vec2d	p2 = b.positionAtTime(t2);
				int		x2 = (int)Math.round(p2.x);
				int		y2 = (int)Math.round(p2.y);
				g.setColor( c[b.m_id] );
				g.drawLine( x1, y1, x2, y2 );
			}
	
			if( opp.m_prevPos != null )
			{
				int		x = (int)Math.round(opp.m_prevPos.x);
				int		y = (int)Math.round(opp.m_prevPos.y);
				g.setColor( Color.lightGray );
				g.drawOval( x - 20, y - 20, 40, 40 );
			}
		
			if( opp.m_curPos != null )
			{
				int		x = (int)Math.round(opp.m_curPos.x);
				int		y = (int)Math.round(opp.m_curPos.y);
				g.setColor( Color.white );
				g.drawOval( x - 20, y - 20, 40, 40 );
			}

			{
				Vec2d	pos = new Vec2d();
				opp.predictFuturePositionCircular( getTime() - t2, pos );
				int		x = (int)Math.round(pos.x);
				int		y = (int)Math.round(pos.y);
				g.setColor( Color.yellow );
				g.drawOval( x - 20, y - 20, 40, 40 );
			}
		}
	}

	//==========================================================================

	private double	distToEdge( double x, double y )
	{
		return Math.min(Math.min(x, getBattleFieldWidth()  - x),
						Math.min(y, getBattleFieldHeight() - y));
	}

	//==========================================================================

	private double	calcDistAhead( final Vec2d v, double heading )
	{
		return calcDistAhead(v, heading, 75);
	}

	//==========================================================================

	private double	calcDistAhead( final Vec2d v, double heading, double buffer )
	{
		heading = Utils.normAngle360(heading);
		double	s = Math.sin(Utils.deg2rad(heading));
		double	c = Math.cos(Utils.deg2rad(heading));

		double	dist = Double.MAX_VALUE;
		double	buf = 75;

		if( c > 0 )
		{
			// cross top of arena
			double	dy = getBattleFieldHeight() - buffer - v.y;
			dist = Math.min(dist, dy / c);
		}
		else if( c < 0 )
		{
			// cross bottom of arena
			double	dy =  buffer - v.y;
			dist = Math.min(dist, dy / c);
		}
				
		if( s > 0 )
		{
			// cross right of arena
			double	dx = getBattleFieldWidth() - buffer - v.x;
			dist = Math.min(dist, dx / s);
		}
		else if( s < 0 )
		{
			// cross left of arena
			double	dx = buffer - v.x;
			dist = Math.min(dist, dx / s);
		}
	
		return dist;
	}

	//==========================================================================

	volatile int		m_scanCount = 0;
	volatile int		m_aliveCount = 0;
	volatile long		m_lastTargetTime = 0;
	
	Opponent			m_target;

	static Hashtable<String, Opponent>	m_opponents = new Hashtable<String, Opponent>();
}

//==============================================================================
//
//	Class OPPONENT
//
//==============================================================================
//
//	for the real target, we need to do targetting some time before we actually
//	fire, in order to give the turret time to aim in the correct direction.
//
//	for virtual guns/bullets, we can just target and fire all in one go.
//
//	~~~~~
//
//	for scan update (doScanUpdate).
//
//	for each tick (for all opponents) (doTickUpdate).
//		1. work out power we want to fire with.
//
//	for each tick (with the target oppononet only) (boolean doFire).
//		1. choose the gun most likely to hit (ask the guns for their hit
//			probabilities).
//		2. work out new target position (as the prediction of our fire position
//			may have changed.
//		3. turn turret to hit the gun's projected future position of this
//			opponent.
//		4. if it's possible to fire the gun, do so now (return true).
//
//	for when the main (real) gun fires (doFireVirtual).
//		1. work out targets for all guns.
//		2. fire virtual guns (create virtual bullets).
//
//==============================================================================
	
class Opponent
{
	public Reach				m_r = null;

	public String				m_name = null;
	public boolean				m_isAlive = true;
	public Vec2d				m_curPos = null;
	public Vec2d				m_prevPos = null;
	public double				m_energy;

	public double				m_lastFireTime;
	public double				m_lastBulletPower;
	public double				m_nextTimeCanFire;

	public ScannedRobotEvent	m_curScan  = null;
	public ScannedRobotEvent	m_prevScan = null;

	public int					m_velIdx  = 0;
	public int					m_velSize = 0;
	public double[]				m_velHist = new double[1000];

	public long					m_targetPatTime;
	public int					m_targetPatIdx  = -1;
	public int					m_targetPatLen;
	public double				m_targetPatCorr;

	public double				m_firePower = 3.0;
	public Vec2d[]				m_targetPos;

	public int					m_bestGun;
	public ArrayList<VirtualGun>m_guns = new ArrayList<VirtualGun>();
	public List<VirtualBullet>	m_bullets = new LinkedList<VirtualBullet>();

	//==========================================================================

	public Opponent( Reach r )
	{
		m_r = r;

		m_guns.add( new HeadOnGun(r, this) );
		m_guns.add( new LinearGun(r, this, 1.0) );
		m_guns.add( new LinearGun(r, this, 0.5) );
		m_guns.add( new CircularGun(r, this) );
		m_guns.add( new VelocityPatternGun(r, this) );
		m_guns.add( new CircularGuessFactorGun(r, this, 1.0) );
		m_guns.add( new CircularGuessFactorGun(r, this, 0.75) );
		m_guns.add( new CircularGuessFactorGun(r, this, 0.5) );
		m_guns.add( new CircularGuessFactorGun(r, this, 0.25) );
		
		m_targetPos = new Vec2d[m_guns.size()];
	}

	//==========================================================================

	public void onScannedRobot( ScannedRobotEvent e )
	{
		m_prevScan = m_curScan;
		m_curScan = e;

		double	oppHeading = m_r.getHeading() + e.getBearing();
		m_prevPos = m_curPos;
		m_curPos = m_r.location().plus(Vec2d.polar(oppHeading, e.getDistance()));

		updateVirtualBullets();

		if( m_prevScan != null )
		{
			long	startTime = m_prevScan.getTime();
			long	rangeTime = e.getTime() - startTime;
			double	startVel  = m_prevScan.getVelocity();
			double	rangeVel  = e.getVelocity() - startVel;

			for( long deltaTime = 1; deltaTime < rangeTime; deltaTime++ )
			{
				// interpolate velocity...
				double	vel = startVel + rangeVel * deltaTime / rangeTime;
				addToOppVelHist( vel );
			}

		}

		addToOppVelHist( e.getVelocity() );

		updateTarget( -1 );

		if( m_energy > e.getEnergy() )
		{
			// something has drained energy from this bot.
			processEnergyDrain( m_energy - e.getEnergy() );
		}

		m_energy = e.getEnergy();
/*
		if( m_targetPatIdx >= 0 &&
			m_prevScan != null &&
			m_prevScan.getTime() > m_targetPatTime )
		{
			// pattern is now too old to be useful.
			m_targetPatIdx = -1;
		}*/
	}

	//==========================================================================

	public void onBulletHit( BulletHitEvent e )
	{
		Bullet	bullet = e.getBullet();
		double	bulletDamage  = MyRules.getBulletDamage(bullet.getPower()) +
								MyRules.getBulletHitBonus(bullet.getPower());
		m_energy -= bulletDamage;
	
		if( m_energy > e.getEnergy() )
		{
			// something has drained energy from this bot.
			processEnergyDrain( m_energy - e.getEnergy() );
		}

		m_energy = e.getEnergy();
	}

	//==========================================================================

	public void onRobotDeath( RobotDeathEvent e )
	{
		// this robot has died... cancel all its virtual bullets.
		m_bullets.clear();
		m_prevScan = null;
		m_curScan = null;
		m_prevPos = null;
		m_curPos = null;
	}

	//==========================================================================

	public void onDeath( DeathEvent e )
	{
		// my robot has died... cancel all our virtual bullets.
		m_bullets.clear();
		m_prevScan = null;
		m_curScan = null;
		m_prevPos = null;
		m_curPos = null;
	}

	//==========================================================================

	/**
	 *	chooseFirePower
	 *
	 *	Chooses which power to use when firing.
	 */
	public void doTickUpdate()
	{
		chooseFirePower();
//		System.out.printf( "fire power: %g\n", m_firePower );
	}

	//==========================================================================

	/**
	 *	doTickTargetUpdate
	 *
	 *	This routine gets called every tick for the current target opponent only.
	 *	It's main use is to aim the gun in the correct direction, and to fire
	 *	as soon as it's possible to do so.
	 *
	 *	@return	True if fired the gun, else false.
	 */
	public boolean doTickTargetUpdate()
	{
		chooseBestGun();
		if( m_bestGun < 0 )
			return false;

		updateTarget( m_bestGun );
		return aimAndFireGun();
	}

	//==========================================================================

	private void	chooseFirePower()
	{
		//	1. work out power we want to fire with.  if this has changed, invalidate
		//		all the target positions.
		ScannedRobotEvent	e = m_curScan;
		if( e == null )
			return;

		double	dist = e.getDistance();
		double	firePower = 3;
		if( dist > 100 )
		{
			if( m_r.getEnergy() > 55 )
				firePower = 3;
			else if( m_r.getEnergy() > 25 )
				firePower = 2;
			else if( m_r.getEnergy() > 10 )
				firePower = 1;
			else if( m_r.getEnergy() > 2.5 )
				firePower = 0.5;
			else
				firePower = 0.1;
		}
		else
			firePower = MyRules.MAX_BULLET_POWER;

		if( dist > 750 )//|| getEnergy() < 10 )
			firePower = Math.min(0.5, firePower);
			
		if( m_r.TC_SHOOT )
			firePower = MyRules.MAX_BULLET_POWER;

		// clamp fire power to valid range.
		firePower = Math.max(MyRules.MIN_BULLET_POWER, Math.min(firePower,
							 MyRules.MAX_BULLET_POWER));
							
		if( firePower != m_firePower )
		{
			m_firePower = firePower;
		}
	}

	//==========================================================================
	
	/**
	 *	updateTarget
	 *
	 *	@param	gun		Index of gun to update the target for.  Pass in -1 to
	 *					target all guns.
	 */
	private void updateTarget( int gun )
	{
		ScannedRobotEvent	e = m_curScan;

		// work out where opp will be by the time we can fire + time we can get
		//	 bullet to them.
//		double	timeToFire = m_r.getGunHeat() / m_r.getGunCoolingRate();
		double	timeToFire = Math.ceil(m_r.getGunHeat() / m_r.getGunCoolingRate());
//		double	timeForBullet = e.getDistance() / MyRules.getBulletSpeed(m_firePower);

		Vec2d	myFirePos = new Vec2d();
		m_r.predictMyFuturePosition( timeToFire, myFirePos );

		// add any additional time since the scan.
		timeToFire += m_r.getTime() - e.getTime();


		for( int i = 0; i < m_guns.size(); i++ )
		{
			if( gun > 0 && i != gun )
				continue;

			Vec2d	target = new Vec2d();
			if( m_guns.get(i).predict(myFirePos, m_firePower, target) )
			{
				// limit target to be in the arena.
				if( target.x < 19 )
					target.x = 19;
				if( target.x > m_r.getBattleFieldWidth() - 19 )
					target.x = m_r.getBattleFieldWidth() - 19;
				if( target.y < 19 )
					target.y = 19;
				if( target.y > m_r.getBattleFieldHeight() - 19 )
					target.y = m_r.getBattleFieldHeight() - 19;
					
				m_targetPos[i] = target;
			}
			else
			{
				m_targetPos[i] = null;
			}
		}
	}

	//==========================================================================

	/**
	 *	chooseBestGun
	 *
	 *	Looks through the array of guns, and selects the one with the highest
	 *	probablity to hit.
	 */
	private void chooseBestGun()
	{
		int		bestIdx = -1;
		double	bestProb = 0;
		for( int i = 0; i < m_targetPos.length; i++ )
		{
			if( m_targetPos[i] == null )
			{
				System.out.print( "-- " );
				continue;
			}

			double	prob = getBulletHitProbability(i);
			System.out.printf( "%2.0f ", prob * 100 );
			if( bestIdx < 0 || bestProb < prob )
			{
				bestIdx  = i;
				bestProb = prob;
			}
		}
		System.out.println( " best " + bestIdx + " " + m_name );

		m_bestGun = bestIdx;
	}

	//==========================================================================

	public boolean	aimAndFireGun()
	{
		if( m_bestGun < 0 )
			return false;

//		System.out.printf( "best targ: %d (%.4f)\n", bestIdx, bestProb );
		Vec2d	target = m_targetPos[m_bestGun];
		if( target != null )
		{
			double	fireHeading = target.minus(m_r.location()).getHeading();
			double	delta = Utils.normAngle180(fireHeading - m_r.getGunHeading());
			m_r.setTurnGunRight( delta );
		}

		if( m_r.getGunHeat() == 0 )//&&
//			Math.abs(getGunTurnRemaining())  <  1 )//&&
//			Math.abs(getVelocity())          == 8 &&
//			Math.abs(getDistanceRemaining()) >  5 )
		{
			if( Reach.TC_SHOOT && m_r.getEnergy() > 0 ||
				m_r.getEnergy() - m_firePower >= 0.01 )
			{
				m_r.setFire( m_firePower );
//				System.out.println( "gun turn rem " + getGunTurnRemaining() +
//					"   scan age " + (getTime() - m_target.m_curScan.getTime()));
				return true;
			}
		}
	
		return false;
	}

	//==========================================================================

	public void fireVirtualGuns()
	{
		// add virtual bullets to target.
		Vec2d	me = m_r.location();
		for( int i = 0; i < m_targetPos.length; i++ )
		{
			if( m_targetPos[i] == null )
				continue;

			double	bulletSpeed = MyRules.getBulletSpeed(m_firePower);

			Vec2d	vel = m_targetPos[i].minus(me).unit().mult(bulletSpeed);
			double	targDist = m_curScan.getDistance();

			VirtualBullet	b = new VirtualBullet();
			b.m_victim   = this;
			b.m_position = me;
			b.m_target   = m_targetPos[i];
			b.m_velocity = vel;
			b.m_power    = m_firePower;
			b.m_dist     = targDist;
			b.m_id       = i;
			b.m_fireTime = m_r.getTime();
			b.m_hitTime  = b.m_fireTime + targDist / bulletSpeed;
			b.m_hit      = false;

//			System.out.printf( "add bullet (%.0f %.0f) v(%.3f %.3f) %d  %.1f %.0f-%.0f\n",
//					me.x, me.y, vel.x, vel.y, i, bulletSpeed, b.m_fireTime, b.m_wallTime );
			m_bullets.add( b );
		}
	}

	//==========================================================================

	private void	processEnergyDrain( double drain )
	{
		// something else has drained energy from this bot.
		if( drain >= MyRules.MIN_BULLET_POWER &&
			drain <= MyRules.MAX_BULLET_POWER )
		{
			// probably shot a bullet... check last time fired.
			if( m_nextTimeCanFire <= m_r.getTime() )
			{
				double	gunCoolingTime = MyRules.getGunHeat(drain) / m_r.getGunCoolingRate();
				m_lastFireTime = m_r.getTime() - 1;
				m_nextTimeCanFire = m_lastFireTime + gunCoolingTime;
				m_lastBulletPower = drain;
//				System.out.printf( "%s shot: time %d  power %g\n", e.getName(),
//						getTime(), drain );
			}
			else
			{
				// couldn't have fired yet... must be something else.
			}
		}
	}

	//==========================================================================

	public double	getBulletHitProbability( int idx )
	{
		return m_guns.get(idx).probabilityToHit(m_firePower);
	}

	//==========================================================================

	public void updateVirtualBullets()
	{
		// scan the virtual bullet list, and see if any of them hit us, or
		//	have gone past us.
		//
		// if a bullet hits us, we update its probability to be higher.
		// if a bullet has passed us, we update is probaility to be lower.
		
		// we assume we have moved in a straight line from our previous
		//	position to our current one.
		
		ScannedRobotEvent	s1 = m_curScan;
		ScannedRobotEvent	s0 = m_prevScan == null ? m_curScan : m_prevScan;

		Vec2d	p1 = m_curPos;
		Vec2d	p0 = m_prevPos == null ? m_curPos : m_prevPos;

		double	deltaT = s1.getTime() - s0.getTime();
		Vec2d	deltaP = p1.minus(p0);
		Vec2d	deltaV = deltaP.mult(1.0 / deltaT);
//		System.out.printf( "target (%.1f %.1f)-(%.1f %.1f) %d-%d\n",
//				p0.x, p0.y, p1.x, p1.y, s0.getTime(), s1.getTime() );

		long	curTime = s1.getTime();

		final int		BOT_RADIUS	= 20;
		for( Iterator<VirtualBullet> i = m_bullets.listIterator(); i.hasNext(); )
		{
			VirtualBullet b = i.next();

//			if( b.m_hitTime > curTime )
//				continue;
				
			Vec2d	b0 = b.positionAtTime(s0.getTime());
			Vec2d	b1 = b.positionAtTime(s1.getTime()).minus(deltaP);
			Vec2d	deltaB = b1.minus(b0);
//			System.out.printf( "bullet (%.1f %.1f)-(%.1f %.1f)\n", b0.x, b0.y, b1.x, b1.y );
		
			boolean	hit = false;
			if( b0.dist(p0) < BOT_RADIUS || b1.dist(p0) < BOT_RADIUS )
			{
				hit = true;
//				System.out.printf( "%d hit end\n", b.m_id );
			}
			else if( Math.abs(Utils.distFromLine(b0, deltaB, p0)) < BOT_RADIUS )
			{
				// might have touched... need to check where...
				double	t = Utils.closestPtOnLine(b0, deltaB, p0);
				hit = t >= 0 && t <= 1;
				if( hit )
				{
					double	hitT = s0.getTime() + (s1.getTime() - s0.getTime()) * t;
					Vec2d	hitB = b.positionAtTime(hitT);
					Vec2d	hitP = p0.plus(deltaP.mult(t));
					double	dist = hitB.dist(hitP);
					double	dist2 = Utils.distFromLine(b0, deltaB, p0);
//					System.out.printf( "%d hit mid line (%.3f) dist %g, %g\n", b.m_id, t, dist, dist2 );
				}
			}
		
			if( hit )
			{
				// bullet hit the target.
				b.m_hit = true;
				System.out.printf( "bullet hit [%d] (%6.1f, %6.1f)\n", b.m_id, b0.x, b0.y );

				// update stats (upwards).
//				m_bulletStats.get(b.m_id).learnValue( b.m_dist, 1.0 );
				m_guns.get(b.m_id).learn( b );

				// remove bullet from the list.
				i.remove();
			}
			else if( //deltaP.dot(p0.minus(b0)) < 0 ||	// target passed
					 b.m_hitTime <= curTime )		// time's up.
			{
				// target is now behind the path of the bullet, or bullet has
				//	run out of time (hit a wall).
				b.m_hit = false;
//				System.out.printf( "bullet missed (%6.1f, %6.1f) %d\n", b0.x, b0.y, b.m_id );
				
				// update stats (downwards).
//				m_bulletStats.get(b.m_id).learnValue( b.m_dist, 0.0 );
				m_guns.get(b.m_id).learn( b );
								
				// remove the bullet from the list.
				i.remove();
			}
		}
	}

	//==========================================================================

	public double	getVelHist( int idx )
	{
		return m_velHist[(m_velIdx + idx) % m_velHist.length];
	}

	//==========================================================================

	public void addToOppVelHist( double vel )
	{
		m_velIdx = (m_velIdx + m_velHist.length - 1) % m_velHist.length;
//		m_velHist[m_velIdx] = (byte)Math.round(vel * 8);
		m_velHist[m_velIdx] = vel;
//		System.out.println( "add vel " + vel + " --> " + m_velHist[m_velIdx] );
		if( m_velSize < m_velHist.length )
			m_velSize++;
	}

	//==========================================================================

	public void	predictFuturePositionLinear( double futureTime, Vec2d pos )
	{
		ScannedRobotEvent	e = m_curScan;

		double	futureDist = e.getVelocity() * futureTime;
		pos.set( m_curPos.plus(Vec2d.polar(e.getHeading(), futureDist)) );
	}

	//==========================================================================

	public void	predictFuturePositionCircular( double futureTime, Vec2d pos )
	{
		double	distMoved = m_curScan.getVelocity() * futureTime;
		predictFuturePositionCircFromDist( distMoved, pos );
	}

	//==========================================================================
	
	public void	predictFuturePositionCircFromDist( double distMoved, Vec2d pos )
	{
		ScannedRobotEvent	e = m_curScan;
		ScannedRobotEvent	p = m_prevScan;

		double	curvature = 0.00001;	// assume going straight

		if( p != null && Math.abs(e.getVelocity()) > 1 )
		{
			double	deltaT = e.getTime() - p.getTime();
			double	deltaAng = e.getHeadingRadians() - p.getHeadingRadians();
			double	turnRate = Utils.normAnglePi(deltaAng) / deltaT;

			curvature = turnRate / e.getVelocity();
			if( Math.abs(curvature) < 0.00001 )
				curvature = 0.00001;
		}

		double	radius = 1 / curvature;
//		System.out.println( "k " + curvature + "  radius " + radius );
		double	heading1 = e.getHeadingRadians() - Math.PI / 2;
		double	heading2 = heading1 + curvature * distMoved;
//		System.out.println( "h1 " + heading1 + "  h2 " + heading2 );

		Vec2d	arcMove = Vec2d.polarRadians(heading2, radius).minus(
						  Vec2d.polarRadians(heading1, radius));

		pos.set( m_curPos.plus(arcMove) );
	}

	//==========================================================================

	public void	predictFuturePositionCircularGF( double futureTime, double gf, Vec2d pos )
	{
		boolean	ahead = true;//m_curScan.getVelocity() >= 0;
		double	maxDist = Utils.calcMaxDist(m_curScan.getVelocity(), ahead,  futureTime);
		double	minDist = Utils.calcMaxDist(m_curScan.getVelocity(), !ahead, futureTime);
/*
		if( m_curScan.getVelocity() < 0 )
		{
			double	temp = minDist;
			minDist = maxDist;
			maxDist = temp;
		}
*/
		double	distMoved = minDist + (maxDist - minDist) * gf;
		predictFuturePositionCircFromDist( distMoved, pos );
	}

	//==========================================================================

	public void matchPattern( long curTime )
	{
		m_targetPatTime = curTime;
		m_targetPatIdx  = -1;
		m_targetPatLen  = 0;
		m_targetPatCorr = 0;

		long	start = System.nanoTime();

		int	minTime = (int)Math.ceil(m_curScan.getDistance() /
							MyRules.getBulletSpeed(MyRules.MAX_BULLET_POWER));

		int		nMults = 0;
		for( int i = minTime; i < m_velSize; i++ )
		{
			// length left
			int	maxLen = m_velSize - i;
			if( maxLen < m_targetPatLen )
				// can't find a long sequence now.
				break;

			double	sum = 0;
			for( int j = 0; j < maxLen; j++ )
			{
				double	delta = getVelHist(j) - getVelHist(i + j);
				sum += delta * delta;
				nMults++;

				if( sum / maxLen >= 1.5 )
					// it's impossible to find a good enough correlation now...
					break;

				double	corr = sum / (j + 1);
				if( m_targetPatLen <= j && corr < 1.5  )
				{
					m_targetPatIdx  = i;
					m_targetPatLen  = j + 1;
					m_targetPatCorr = corr;
				}
			}
		}

		long	elapsed = System.nanoTime() - start;

		System.out.printf( "n%d (%d) %d [%d,%d] %g: ",
							nMults, elapsed / 1000,
							m_targetPatTime, m_targetPatIdx,
							m_targetPatLen,  m_targetPatCorr );
		for( int i = 0; i < m_targetPatLen && i < 20; i++ )
		{
			System.out.printf( "%2.0f ", getVelHist(m_targetPatIdx + i) );
		}
		System.out.println();

	}

	//==========================================================================

	public int		getTargetPatternIndex()
	{
		return m_targetPatIdx;
	}

	//==========================================================================

	public double	calcPatternDist( double patTime )
	{
		double	totalDist = 0;
		int	idx = m_targetPatIdx;
		while( idx > 0 && patTime >= 1 )
		{
			double	u = getVelHist(idx);
			double	v = getVelHist(idx - 1);
//			System.out.printf( "%3d", u );
			double	dist = (u + v) * 0.5;
			totalDist += dist;
			
			idx--;
			patTime--;
		}
	
		return totalDist;
	}
}

//==============================================================================
//
//	Class VIRTUAL BULLET
//
//==============================================================================

class VirtualBullet
{
	public Opponent	m_victim;	// victim we hope to hit.
	public Vec2d	m_position;	// initial position of bullet.
	public Vec2d	m_target;	// final position of bullet.
	public Vec2d	m_velocity;	// velocity of bullet.
	public double	m_power;	// power of bullet.
	public double	m_dist;		// dist of target at time of bullet creation.
	public int		m_id;		// id of virtual gun (index for modifying stats).
	public double	m_fireTime;	// time when this bullet was fired.
	public double	m_hitTime;	// time when bullet will get to target position.
	public boolean	m_hit;		// true if bullet hit victim, else false.
	
	public Vec2d	positionAtTime( double time )
	{
		double	deltaTime = time - m_fireTime;
		return m_position.plus(m_velocity.mult(deltaTime));
	}
}

//==============================================================================
//
//	Interface TARGETTING
//
//==============================================================================

interface Targetting
{
	public boolean predict( Reach r, Vec2d myFirePos, Opponent opp, double power, Vec2d target );
}

//==============================================================================
//
//	Class HEAD-ON TARGET
//
//==============================================================================

class HeadOnTarget implements Targetting
{
	public boolean predict(
		Reach		r,
		Vec2d		myFirePos,
		Opponent	opp,
		double		power,
		Vec2d		target )
	{
		target.set( opp.m_curPos );

		if( opp.m_curScan.getEnergy() > 0 )
		{
			double	offsTime = r.getTime() - opp.m_curScan.getTime();
			opp.predictFuturePositionCircular(offsTime, target);
		}

		return true;
	}
}

//==============================================================================
//
//	Class LINEAR TARGET
//
//==============================================================================

class LinearTarget implements Targetting
{
	LinearTarget( double proportion )
	{
		m_proportion = proportion;
	}

	public boolean predict(
		Reach		r,
		Vec2d		myFirePos,
		Opponent	opp,
		double		power,
		Vec2d		target )
	{
		ScannedRobotEvent	e = opp.m_curScan;
			
		// work out where opp will be by the time we can fire + time we can get
		//	 bullet to them.
		double	bulletSpeed = MyRules.getBulletSpeed(power);
		double	timeToFire = r.getGunHeat() / r.getGunCoolingRate();
		double	timeForBullet = e.getDistance() / bulletSpeed;

		// add any additional time since the scan.
		timeToFire += r.getTime() - e.getTime();

//		System.out.printf( "targetting: time %g\n", timeToFire + timeForBullet );
		for( int loopCount = 0; loopCount < 10; loopCount++ )
		{
			double		oppTime = timeToFire + timeForBullet;

			opp.predictFuturePositionLinear(oppTime * m_proportion, target);

			double	newFireTime = target.dist(myFirePos) / bulletSpeed;
			double	errTime = newFireTime - timeForBullet;
//			System.out.println( "targ err " + errTime );
			if( Math.abs(errTime) < 0.01 )
			{
				break;
			}
		
			timeForBullet = newFireTime;
		}

		return true;
	}

	private double	m_proportion;
}

//==============================================================================
//
//	Class CIRCULAR TARGET
//
//==============================================================================

class CircularTarget implements Targetting
{
	public boolean predict(
		Reach		r,
		Vec2d		myFirePos,
		Opponent	opp,
		double		power,
		Vec2d		target )
	{
//		if( opp.m_prevScan == null )
//			return false;

		ScannedRobotEvent	e = opp.m_curScan;
			
		// work out where opp will be by the time we can fire + time we can get
		//	 bullet to them.
		double	bulletSpeed = MyRules.getBulletSpeed(power);
		double	timeToFire = r.getGunHeat() / r.getGunCoolingRate();
		double	timeForBullet = e.getDistance() / bulletSpeed;

		// add any additional time since the scan.
		timeToFire += r.getTime() - e.getTime();

//		System.out.printf( "targetting: time %g\n", timeToFire + timeForBullet );
		for( int loopCount = 0; loopCount < 10; loopCount++ )
		{
			double		oppTime = timeToFire + timeForBullet;

			opp.predictFuturePositionCircular(oppTime, target);

			double	newFireTime = target.dist(myFirePos) / bulletSpeed;
			double	errTime = newFireTime - timeForBullet;
//			System.out.println( "targ err " + errTime );
			if( Math.abs(errTime) < 0.01 )
			{
				break;
			}
		
			timeForBullet = newFireTime;
		}

		return true;
	}
}

//==============================================================================
//
//	Class VELOCITY PATTERN TARGET
//
//==============================================================================

class VelocityPatternTarget implements Targetting
{
	public boolean predict(
		Reach		r,
		Vec2d		myFirePos,
		Opponent	opp,
		double		power,
		Vec2d		target )
	{
//		if( r.getTarget() != opp )
//			return false;

		ScannedRobotEvent	e = opp.m_curScan;
			
		// work out where opp will be by the time we can fire + time we can get
		//	 bullet to them.
		double	timeToFire = r.getGunHeat() / r.getGunCoolingRate();
		double	timeForBullet = e.getDistance() / MyRules.getBulletSpeed(power);

		// add any additional time since the scan.
		timeToFire += r.getTime() - e.getTime();

		double	targX = 0;
		double	targY = 0;

//		System.out.printf( "VELPAT targetting: time %g\n", timeToFire + timeForBullet );
		Vec2d	oppPos = new Vec2d();
		for( int loopCount = 0; loopCount < 10; loopCount++ )
		{
			double		oppTime = timeToFire + timeForBullet;
			opp.predictFuturePositionCircular( oppTime, oppPos );

			if( opp.getTargetPatternIndex() >= oppTime )
			{
				double	patDist = opp.calcPatternDist(oppTime);
//				double	dist = m_target.m_curScan.getVelocity() * oppTime;
//				System.out.printf( "pat dist %g  opp dist %g   time %g\n",
//						patDist, dist, oppTime );
				opp.predictFuturePositionCircFromDist(patDist, oppPos);
			}

			target.x = oppPos.x;
			target.y = oppPos.y;

			double	newFireDX = target.x - myFirePos.x;
			double	newFireDY = target.y - myFirePos.y;
			double	newFireTime = Math.hypot(newFireDX, newFireDY) /
										MyRules.getBulletSpeed(power);

			double	errTime = newFireTime - timeForBullet;
//			System.out.printf( "targ err %.5f\n", errTime );
			if( Math.abs(errTime) < 0.01 )
			{
				break;
			}
		
			timeForBullet = newFireTime;
		}

		return true;
	}
}

//==============================================================================
//
//	Class CIRCULAR GUESS FACTOR TARGET
//
//==============================================================================

class CircularGuessFactorTarget implements Targetting
{
	CircularGuessFactorTarget( double proportion )
	{
		m_proportion = proportion;
	}

	public boolean predict(
		Reach		r,
		Vec2d		myFirePos,
		Opponent	opp,
		double		power,
		Vec2d		target )
	{
//		if( opp.m_prevScan == null )
//			return false;

		ScannedRobotEvent	e = opp.m_curScan;
			
		// work out where opp will be by the time we can fire + time we can get
		//	 bullet to them.
		double	bulletSpeed = MyRules.getBulletSpeed(power);
		double	timeToFire = r.getGunHeat() / r.getGunCoolingRate();
		double	timeForBullet = e.getDistance() / bulletSpeed;

		// add any additional time since the scan.
		timeToFire += r.getTime() - e.getTime();

//		System.out.printf( "targetting: time %g\n", timeToFire + timeForBullet );
		for( int loopCount = 0; loopCount < 10; loopCount++ )
		{
			double		oppTime = timeToFire + timeForBullet;

			opp.predictFuturePositionCircularGF(oppTime, m_proportion, target);

			double	newFireTime = target.dist(myFirePos) / bulletSpeed;
			double	errTime = newFireTime - timeForBullet;
//			System.out.println( "targ err " + errTime );
			if( Math.abs(errTime) < 0.01 )
			{
				break;
			}
		
			timeForBullet = newFireTime;
		}

		return true;
	}

	private double	m_proportion;
}

//==============================================================================
//
//	Interface FUNCTION
//
//==============================================================================

interface Function
{
	public double	getValue( double in );
	public void		learnValue( double in, double value );
}

//==============================================================================
//
//	Class SIMPLE SEGMENTATION
//
//==============================================================================

class SimpleSegmentedFunction implements Function
{
	SimpleSegmentedFunction( double min, double max, int bins, double beta, double initialValue )
	{
		m_min = min;
		m_max = max;
		m_range = max - min;
		m_bins = bins;
		m_bin = new double[bins];
		for( int i = 0; i < m_bins; i++ )
			m_bin[i] = initialValue;
		m_beta = beta;
	}

	//==========================================================================

	public double	getBeta()
	{
		return m_beta;
	}

	//==========================================================================

	public void	setBeta( double beta )
	{
		m_beta = beta;
	}

	//==========================================================================

	public void	setBinValue( int bin, double value )
	{
		m_bin[bin] = value;
	}

	//==========================================================================

	public double	getBinValue( int bin )
	{
		return m_bin[bin];
	}

	//==========================================================================

	public double	getValue( double in )
	{
		int	bin = (int)Math.floor((in - m_min) / m_range);
		if( bin < 0 )
			bin = 0;
		else if( bin >= m_bins )
			bin = m_bins - 1;
			
		return m_bin[bin];
	}

	//==========================================================================

	public void	learnValue( double in, double value )
	{
		int	bin = (int)Math.floor((in - m_min) / m_range);
		if( bin < 0 )
			bin = 0;
		else if( bin >= m_bins )
			bin = m_bins - 1;
			
		m_bin[bin] = m_bin[bin] * (1 - m_beta) + value * m_beta;
	}

	//==========================================================================

	private double		m_min;
	private double		m_max;
	private double		m_range;
	private int			m_bins;
	private double[]	m_bin;
	private double		m_beta;
}

//==============================================================================
//
//	Class VIRTUAL GUN
//
//==============================================================================

abstract class VirtualGun
{
	public VirtualGun( Reach r, Opponent opp, Targetting targ )
	{
		m_r = r;
		m_opp = opp;
		m_targetting = targ;
	}

	public boolean predict( Vec2d myFirePos, double power, Vec2d target )
	{
		return m_targetting.predict(m_r, myFirePos, m_opp, power, target);
	}

	abstract public void	learn( VirtualBullet b );
	abstract public double	probabilityToHit( double power );

	protected Reach			m_r;
	protected Opponent		m_opp;
	protected Targetting	m_targetting;
}

//==============================================================================
//
//	Class DISTANCE SEGMENTED GUN
//
//==============================================================================

class DistanceSegmentedGun extends VirtualGun
{
	public DistanceSegmentedGun( Reach r, Opponent opp, Targetting targ )
	{
		super( r, opp, targ );
		m_stats = new SimpleSegmentedFunction(0, 1200, 8, 0.05, 0.2);
	}

	public void learn( VirtualBullet b )
	{
		if( b.m_hit )
		{
			m_stats.learnValue( b.m_dist, 1.0 );
		}
		else
		{
			m_stats.learnValue( b.m_dist, 0.0 );
		}
	}

	public double	probabilityToHit( double power )
	{
		double	dist = m_opp.m_curScan.getDistance();
		return m_stats.getValue(dist);
	}

	protected Function	m_stats;
}

//==============================================================================
//
//	Class DISTANCE POWER SEGMENTED GUN
//
//==============================================================================

class DistancePowerSegmentedGun extends VirtualGun
{
	public DistancePowerSegmentedGun( Reach r, Opponent opp, Targetting targ )
	{
		super( r, opp, targ );
		m_stats = new SimpleSegmentedFunction[5];
		for( int i = 0; i < 5; i++ )
			m_stats[i] = new SimpleSegmentedFunction(0, 1200, 8, 0.05, 0.2);
	}

	public void learn( VirtualBullet b )
	{
		int	powerIdx = getPowerIndex(b.m_power);

		if( b.m_hit )
		{
			m_stats[powerIdx].learnValue( b.m_dist, 1.0 );
		}
		else
		{
			m_stats[powerIdx].learnValue( b.m_dist, 0.0 );
		}
	}

	public double	probabilityToHit( double power )
	{
		double	dist = m_opp.m_curScan.getDistance();
		int		powerIdx = getPowerIndex(power);
		return m_stats[powerIdx].getValue(dist);
	}

	private int getPowerIndex( double power )
	{
		if( power > 2.5 )
			return 0;
		else if( power > 1.5 )
			return 1;
		else if( power > 0.75 )
			return 2;
		else if( power > 0.25 )
			return 3;
		else
			return 4;
	}

	protected Function[]	m_stats;
}

//==============================================================================
//
//	Class HEAD-ON GUN
//
//==============================================================================

class HeadOnGun extends DistanceSegmentedGun
//class HeadOnGun extends DistancePowerSegmentedGun
{
	public HeadOnGun( Reach r, Opponent opp )
	{
		super( r, opp, new HeadOnTarget() );
		m_disabledStats = new SimpleSegmentedFunction(0, 1200, 8, 0.05, 0.2);
	}

	//==========================================================================

	public void learn( VirtualBullet b )
	{
		if( m_opp.m_curScan.getEnergy() > 0 )
			super.learn(b);
		else
		{
			if( b.m_hit )
			{
				m_disabledStats.learnValue( b.m_dist, 1.0 );
			}
			else
			{
				m_disabledStats.learnValue( b.m_dist, 0.0 );
			}
		}
	}

	//==========================================================================

	public double	probabilityToHit( double power )
	{
		double	dist = m_opp.m_curScan.getDistance();
		if( m_opp.m_curScan.getEnergy() > 0 )
			return super.probabilityToHit(dist);
		else
			return m_disabledStats.getValue(dist);
	}

	protected Function m_disabledStats;
}

//==============================================================================
//
//	Class LINEAR GUN
//
//==============================================================================

class LinearGun extends DistanceSegmentedGun
//class LinearGun extends DistancePowerSegmentedGun
{
	public LinearGun( Reach r, Opponent opp, double factor )
	{
		super( r, opp, new LinearTarget(factor) );
	}
}

//==============================================================================
//
//	Class CIRCULAR GUN
//
//==============================================================================

class CircularGun extends DistanceSegmentedGun
//class CircularGun extends DistancePowerSegmentedGun
{
	public CircularGun( Reach r, Opponent opp )
	{
		super( r, opp, new CircularTarget() );
	}
}

//==============================================================================
//
//	Class VELOCITY PATTERN GUN
//
//==============================================================================

class VelocityPatternGun extends DistanceSegmentedGun
//class VelocityPatternGun extends DistancePowerSegmentedGun
{
	public VelocityPatternGun( Reach r, Opponent opp )
	{
		super( r, opp, new VelocityPatternTarget() );
	}
}

//==============================================================================
//
//	Class GUESS FACTOR GUN
//
//==============================================================================

class CircularGuessFactorGun extends DistanceSegmentedGun
//class CircularGuessFactorGun extends DistancePowerSegmentedGun
{
	public CircularGuessFactorGun( Reach r, Opponent opp, double gf )
	{
		super( r, opp, new CircularGuessFactorTarget(gf) );
	}
}

//==============================================================================
//
//	Class VEC2D
//
//==============================================================================

class Vec2d
{
	public double	x;
	public double	y;

	//==========================================================================

	public	Vec2d()
	{
		x = 0;
		y = 0;
	}

	//==========================================================================

	public Vec2d( double x, double y )
	{
		this.x = x;
		this.y = y;
	}

	//==========================================================================

	public Vec2d( final Vec2d v )
	{
		x = v.x;
		y = v.y;
	}

	//==========================================================================

	static public Vec2d	fromHeadingRadians( double radians )
	{
		return new Vec2d(Math.sin(radians), Math.cos(radians));
	}

	//==========================================================================

	static public Vec2d	fromHeading( double degrees )
	{
		return fromHeadingRadians(Utils.deg2rad(degrees));
	}

	//==========================================================================

	static public Vec2d polarRadians( double radians, double radius )
	{
		return new Vec2d(radius * Math.sin(radians),
						 radius * Math.cos(radians));
	}

	//==========================================================================

	static public Vec2d polar( double degrees, double radius )
	{
		return polarRadians(Utils.deg2rad(degrees), radius);
	}

	//==========================================================================

	public void	set( final Vec2d v )
	{
		x = v.x;
		y = v.y;
	}

	//==========================================================================

	public double	mag()
	{
		return Math.hypot(x, y);
	}

	//==========================================================================

	public double	dist( final Vec2d v )
	{
		double	dx = v.x - x;
		double	dy = v.y - y;
		return Math.sqrt(dx * dx + dy * dy);
	}

	//==========================================================================

	public double	dist( double x, double y )
	{
		double	dx = x - this.x;
		double	dy = y - this.y;
		return Math.sqrt(dx * dx + dy * dy);
	}

	//==========================================================================

	public Vec2d	unit()
	{
		double	len = mag();
		return new Vec2d(x / len, y / len);
	}

	//==========================================================================

	public double	dot( final Vec2d v )
	{
		return x * v.x + y * v.y;
	}

	//==========================================================================

	public double	getHeadingRadians()
	{
		return Math.atan2(x, y);
	}

	//==========================================================================

	public double	getHeading()
	{
		return Utils.rad2deg(getHeadingRadians());
	}

	//==========================================================================

	public Vec2d	plus( final Vec2d in )
	{
		return new Vec2d(x + in.x, y + in.y);
	}

	//==========================================================================

	public Vec2d	plus( double x, double y )
	{
		return new Vec2d(this.x + x, this.y + y);
	}

	//==========================================================================

	public Vec2d	minus( final Vec2d in )
	{
		return new Vec2d(x - in.x, y - in.y);
	}

	//==========================================================================

	public Vec2d	minus( double x, double y )
	{
		return new Vec2d(this.x - x, this.y - y);
	}

	//==========================================================================

	public Vec2d	negate()
	{
		return new Vec2d(-x, -y);
	}

	//==========================================================================

	public Vec2d	mult( double s )
	{
		return new Vec2d(x * s,	y * s);
	}

	//==========================================================================

	public Vec2d	scale( double sx, double sy )
	{
		return new Vec2d(x * sx, y * sy);
	}
}

//==============================================================================
//
//	Class UTILS
//
//==============================================================================

class Utils
{
	public static double	deg2rad( double deg )
	{
		return deg * Math.PI / 180;
	}

	//==========================================================================

	public static double	rad2deg( double rad )
	{
		return rad * 180 / Math.PI;
	}

	//==========================================================================

	public static double	normAngle180( double deg )
	{
		while( deg < -180 )
			deg += 360;
		while( deg > 180 )
			deg -= 360;
		return deg;
	}

	//==========================================================================

	public static double	normAngle360( double deg )
	{
		while( deg < 0 )
			deg += 360;
		while( deg >= 360 )
			deg -= 360;
		return deg;
	}

	//==========================================================================

	public static double	normAnglePi( double rad )
	{
		while( rad < -Math.PI )
			rad += Math.PI * 2;
		while( rad > Math.PI )
			rad -= Math.PI * 2;
		return rad;
	}

	//==========================================================================

	public static double	normAngle2Pi( double rad )
	{
		while( rad < 0 )
			rad += Math.PI * 2;
		while( rad >= Math.PI * 2 )
			rad -= Math.PI * 2;
		return rad;
	}

	//==========================================================================

	public static int	rand( int limit )
	{
		return (int)Math.floor(Math.random() * limit);
	}

	//==========================================================================

	public static double calcMaxDist( double spd, boolean ahead, double time )
	{
		if( !ahead )
		{
			spd = -spd;
		}
	
		double	dist = 0;

		// accel of 2 available.
		while( spd < 0 && time > 0 )
		{
			double	dt = Math.min(1, time);
			spd  += MyRules.DECELERATION;
			dist += spd * dt;
			time -= dt;
		}
	
		// accel of 1 available.
		while( spd < MyRules.MAX_VELOCITY && time > 0 )
		{
			double	dt = Math.min(1, time);
			spd  =  Math.min(MyRules.MAX_VELOCITY, spd + 1);
			dist += spd * dt;
			time -= dt;
		}
	
		dist += spd * time;
		return ahead ? dist : -dist;
	}

	//==========================================================================

	public static double distFromLine( final Vec2d l, final Vec2d v, final Vec2d p )
	{
		Vec2d	n = new Vec2d(v.y, -v.x).unit();
		return p.minus(l).dot(n);
	}

	//==========================================================================

	public static double closestPtOnLine( final Vec2d l, final Vec2d v, final Vec2d p )
	{
		// P from AB
		// Q is closest pt on AB
		// (P-Q).(B-A) == 0 then Q is closest pt.
		// Q = A + t.(B-A)
		// (P-(A+t.(B-A)).(B-A)
		// use AB for const B-A, and AP for P-A.
		// (AP + tAB).AB == AP.AB + tAB.AB
		// t = -AP.AB / AB.AB == PA.AB / AB.AB

		double		den = v.dot(v);
		if( den == 0 )
			return 0;

		Vec2d	u = p.minus(l);
		double	num = u.dot(v);
		double	t = num / den;
		return t;
	}
}

//==============================================================================

class MyRules	// because there is no Rules class in 1.1.3 (for RoboRumble)
{
	public static final double	MIN_BULLET_POWER = 0.1;
	public static final double	MAX_BULLET_POWER = 3.0;

	public static final double	ACCELERATION = 1.0;
	public static final double	DECELERATION = 2.0;
	
	public static final double	MAX_VELOCITY = 8.0;

	public static double	getBulletSpeed( double power )
	{
		return 20 - power * 3;
	}
	
	public static double	getBulletDamage( double power )
	{
		return 4 * power;
	}
	
	public static double	getBulletHitBonus( double power )
	{
		return Math.max(0, 2 * (power - 1));
	}
								
	public static double	getGunHeat( double power )
	{
		return 1 + power / 5;
	}
}

//==============================================================================
