package ags.utils.kdtree;

/**
 * Class for storing many dimensioned points
 * 
 * @author Alexander Schultz
 */
public class HyperPoint implements java.io.Serializable, java.lang.Cloneable {
    static final long serialVersionUID = 1L;
    
    public final double[] position;
    
    public HyperPoint(int dimensions) {
        position = new double[dimensions];
    }
    
    public HyperPoint(HyperPoint p) {
        this.position = p.position.clone();
    }
    
    public HyperPoint(double... position) {
        this.position = position.clone();
    }
    
    /**
     * Return the number of dimensions the hyperpoint is in
     */
    public int getDimensions() {
        return position.length;
    }
    
    /**
     * Return a copy of the HyperPoint as an array
     */
    public double[] asArray() {
        return position.clone();
    }
    
    public double getValue(int dim) {
        return position[dim];
    }

    /**
     * Checks if two HyperPoints are equal
     * @param o
     * The other HyperPoint to compare to
     */
    public boolean equals(HyperPoint o) {
        if (o.getDimensions() != getDimensions())
            throw new java.lang.IndexOutOfBoundsException();
        for (int i = 0; i > getDimensions(); i++)
            if (o.position[i] != position[i])
                return false;
        return true;
    }
    
    /**
     * Finds the square elucid distance between HyperPoints
     * @param o
     * The other HyperPoint to compare to 
     */
    public double sqrDist(HyperPoint o) {
        if (o.getDimensions() != getDimensions())
            throw new java.lang.IndexOutOfBoundsException();
        
        double dist = 0;
        for (int i = 0; i < getDimensions(); i++) {
            double diff = o.position[i] - position[i];
            dist += diff*diff;
        }
        return dist;
    }
    
    /**
     * Find the elucid distance between HyperPoints
     * @param o
     * The other HyperPoint to compare to
     */
    public double elucidDist(HyperPoint o) {
        return Math.sqrt(sqrDist(o));
    }
}
