package romz.component.targeting.base;

import static romz.math.RobotMath.cos;
import static romz.math.RobotMath.distance;
import static romz.math.RobotMath.heading;
import static romz.math.RobotMath.limit;
import static romz.math.RobotMath.min;
import static romz.math.RobotMath.sin;
import robocode.Rules;
import robocode.util.Utils;
import romz.component.radar.Radar;
import romz.component.targeting.Targeting;
import romz.model.Situation;
import romz.model.advice.TargetingAdvice;
import romz.model.robot.Robot;

public abstract class AbstractTargeting implements Targeting {

	/** The maximum distance at which we fire with maximum power */
	public static final double MAX_POWER_DISTANCE = 300;
	
	public Situation situation;
	
	@Override
	public TargetingAdvice getTargetingAdvice(Situation situation) {
		this.situation = situation;
		TargetingAdvice targetingAdvice = new TargetingAdvice();
		if (enemyTraced(situation)) {
			targetingAdvice.fire = selectFire();
			targetingAdvice.firePower = selectFirePower();
			targetingAdvice.fireAngle = selectFireAngle(targetingAdvice.firePower);
		}
		return targetingAdvice;
	}

	private boolean enemyTraced(Situation situation) {
		return situation.enemy != null && situation.enemy.last != null && situation.enemy.timeScanned > situation.time - Radar.RESCAN_DELAY;
	}
	
	private double selectFirePower() {
		double firePower = 0;
		if (situation.hero.energy > 0.1) {
			if (situation.enemy.energy <= 4) {
				firePower = situation.enemy.energy / 4;
			} else if (situation.enemy.energy <= 16) {
				firePower = (situation.enemy.energy + 2) / 6;
			} else {
				firePower = Rules.MAX_BULLET_POWER * (MAX_POWER_DISTANCE / situation.enemy.distance);
			}
			firePower = min(firePower, (situation.hero.energy - 0.1) / 8);
			firePower = limit(0.1, firePower, Rules.MAX_BULLET_POWER);
		}
		return firePower;
	}
	
	protected abstract double selectFireAngle(double firePower);
	
	protected abstract boolean selectFire();
	
	protected double circularAcceleratedAngle(double firePower) {
		double bulletVelocity = Rules.getBulletSpeed(firePower);
		double enemyX = situation.enemy.x;
		double enemyY = situation.enemy.y;
		double enemyHeading = situation.enemy.heading;
		double enemyTurnRate = situation.enemy.turnRate;
		double enemyVelocity = situation.enemy.velocity;
		double enemyAcceleration = situation.enemy.acceleration;
		int t = 0;
		do {
			t++;
			enemyHeading = Utils.normalRelativeAngleDegrees(enemyHeading + enemyTurnRate);
			enemyVelocity = min(Rules.MAX_VELOCITY, enemyVelocity + enemyAcceleration);
			enemyX = limit(Robot.HALF_ROBOT_SIZE, enemyX + enemyVelocity * sin(enemyHeading), situation.battleField.width - Robot.HALF_ROBOT_SIZE);
			enemyY = limit(Robot.HALF_ROBOT_SIZE, enemyY + enemyVelocity * cos(enemyHeading), situation.battleField.height - Robot.HALF_ROBOT_SIZE);
		} while (t * bulletVelocity < distance(situation.hero.x, situation.hero.y, enemyX, enemyY));
		double fireAngle = heading(situation.hero.x, situation.hero.y, enemyX, enemyY);
		return Utils.normalAbsoluteAngleDegrees(fireAngle);
	}
	
}
