package djc;
import robocode.*;
import java.util.Enumeration;

/**
 * Gun - an interface for shooting bad guys.
 */
public abstract class Gun
{
    /* ********************************************************************************** */
    /*                                   CONSTANTS                                        */
    /* ********************************************************************************** */
    /** Numerator used to find FirePower: FP_NUMERATOR / targetDist = GunFirePower */
    public static final double FP_NUMERATOR = 400.0;

    /* ********************************************************************************** */
    /*                                MEMBER VARIABLES                                    */
    /* ********************************************************************************** */
    public StrategyBot self = null;
    public GunManager gm = null;
    public String name = "GUN";

    public Gun(StrategyBot _s, GunManager _gm) 
    {
	self = _s;
	gm = _gm;
    }

    /** A method that must be implemented */
    public abstract void reset();
    /** A method that must be implemented */
    public abstract void onScannedRobot(ScannedRobotEvent e);
    /** A method that must be implemented */
    public abstract void setGunRotation();

    /**
     * findBestTarget
     *  @return Target that is best to attack
     */
    public Target findBestTarget()
    {
	return null;
    }

    /**
     * findNearestTarget
     *
     *  @return Target that is closest to self or null if none found.
     */
    public final Target findNearestTarget()
    {
	Enumeration e = self.targetList.elements();
	Target closestTarget = null;
	Target currentTarget = null;
	double closestDistance = Double.MAX_VALUE;
	double currentDistance = Double.MAX_VALUE;
	Coordinate selfPos = new Coordinate(self.getX(),
					    self.getY());

	while(e.hasMoreElements()) {
	    currentTarget = (Target)e.nextElement();
	    double dT = self.getTime() - 
		currentTarget.lastTimePosition;
	    // Check currentTarget is alive and position current enough
	    if (currentTarget.isAlive && dT < TacticalStrategy.POSITION_STALENESS) {
		if (closestTarget == null) {
		    closestTarget = currentTarget;
		    closestDistance = currentTarget.position.distanceFrom(selfPos);
		} else {
		    currentDistance = currentTarget.position.distanceFrom(selfPos);
		    if(currentDistance < closestDistance) {
			closestTarget = currentTarget;
			closestDistance = currentDistance;
		    }
		}
	    }  // currentTarget isAlive and position current enough
	} // end while(e.hasMoreElements())

	return closestTarget;
    }

    /**
     * findWeakestTarget
     *
     *  @return Target that is closest to self or null if none found.
     */
    public final Target findWeakestTarget()
    {
	Enumeration e = self.targetList.elements();
	Target lowestTarget = null;
	Target currentTarget = null;
	double lowestEnergy = Double.MAX_VALUE;
	double currentEnergy = Double.MAX_VALUE;

	while(e.hasMoreElements()) {
	    currentTarget = (Target)e.nextElement();
	    double dT = self.getTime() - 
		currentTarget.lastTimePosition;
	    // Check currentTarget is alive and position current enough
	    if (currentTarget.isAlive && dT < TacticalStrategy.POSITION_STALENESS) {
		if (lowestTarget == null) {
		    lowestTarget = currentTarget;
		    lowestEnergy = currentTarget.energy;
		} else {
		    currentEnergy = currentTarget.energy;
		    if(currentEnergy < lowestEnergy) {
			lowestTarget = currentTarget;
			lowestEnergy = currentEnergy;
		    }
		}
	    }  // currentTarget isAlive and position current enough
	} // end while(e.hasMoreElements())

	return lowestTarget;
    }

    /**
     * findOneShotKillTarget
     *
     *  @return Target that is closest to self or null if none found.
     */
    public final Target findOneShotKillTarget()
    {
	Enumeration e = self.targetList.elements();
	Target closestTarget = null;
	Target currentTarget = null;
	double closestDistance = Double.MAX_VALUE;
	double currentDistance = Double.MAX_VALUE;
	double anticipatedShotPower = 0;
	Coordinate selfPos = new Coordinate(self.getX(),
					    self.getY());

	while(e.hasMoreElements()) {
	    currentTarget = (Target)e.nextElement();
	    double dT = self.getTime() - 
		currentTarget.lastTimePosition;
	    // Check currentTarget is alive and position current enough
	    if (currentTarget.isAlive && dT < TacticalStrategy.POSITION_STALENESS) {
		currentDistance = currentTarget.position.distanceFrom(selfPos);
		anticipatedShotPower = self.currentGun.computeFirePower(currentDistance, currentTarget);
		if(MathHelper.getBulletDamage(anticipatedShotPower) > currentTarget.energy) {
		    if (closestTarget == null) {
			closestTarget = currentTarget;
			closestDistance = currentTarget.position.distanceFrom(selfPos);
		    } else {
			if(currentDistance < closestDistance) {
			    closestTarget = currentTarget;
			    closestDistance = currentDistance;
			}
		    }
		}
	    }  // currentTarget isAlive and position current enough
	} // end while(e.hasMoreElements())

	return closestTarget;
    }

    /**
     * Computes the FirePower to use when shooting a target.
     *
     * @author Dan Cieslak
     */
    public double computeFirePower(double targetDist, Target t)
    {
	double gunPowerLevel = FP_NUMERATOR / targetDist;
	gunPowerLevel = Math.max(MathHelper.MIN_SHOT_ENERGY, gunPowerLevel);
	gunPowerLevel = Math.min(MathHelper.MAX_SHOT_ENERGY, gunPowerLevel);
	return gunPowerLevel;
    }

}
