package djc;
import robocode.*;
import java.util.Hashtable;
import java.util.Enumeration;

/**
 * TargetingStrategyManager 
 *
 * Manages current targeting strategy for a target.
 */
public class TargetingStrategyManager extends Object
{
    /* ********************************************************************************** */
    /*                                   CONSTANTS                                        */
    /* ********************************************************************************** */
    public static final String STATIONARY = "STATIONARY";
    public static final String LINEAR = "LINEAR";
    public static final String CIRCULAR = "CIRCULAR";
    public static final String LINEAR_AVG_VEL = "LINEAR_AVG_VEL";           // Experimental
    public static final String CIRCULAR_AVG_VEL = "CIRCULAR_AVG_VEL";       // Experimental
    public static final String POSITION_HISTORY = "POSITION_HISTORY";       // Experimental
    public static final String FIRE_RESPONSE_DIST = "FIRE_RESPONSE_DIST";   // Experimental
    public static final String FIRE_RESPONSE_POS = "FIRE_RESPONSE_POS";     // Experimental
    public static final String JIGGLE = "JIGGLE";                           // Experimental
    public static final String PATTERN = "PATTERN";                         // Experimental
    public static final double DEFAULT_RATE = 1.0 / 10.0;
    public static final int GRACE_PERIOD = 4;
    public static final int NUM_PREDICTIVE_METHODS = 5;

    /* ********************************************************************************** */
    /*                                MEMBER VARIABLES                                    */
    /* ********************************************************************************** */
    /** The initial strategy to use by default. */
    //public String INITIAL = FIRE_RESPONSE_DIST;
    //public String INITIAL = LINEAR;
    //public String INITIAL = CIRCULAR;
    public String INITIAL = CIRCULAR_AVG_VEL;
    public Hashtable availableStrategies = new Hashtable();
    public Hashtable decodeStrategies = new Hashtable();
    Target t = null;

    // By targeting method
    private static final int TARGET_MODES = 9;
    private double timesFired[] = new double[TARGET_MODES];
    private double timesHit[] = new double[TARGET_MODES];
    private double energyLost[] = new double[TARGET_MODES];
    private double energyGained[] = new double[TARGET_MODES];

    /**
     * TargetingStrategyManager constructor
     * 
     * Currently adds the following TargetingStrategies:
     * <ul>
     *   <li> StationaryTacticalStrategy </li>
     *   <li> LinearTacticalStrategy </li>
     *   <li> CircularTacticalStrategy </li>
     *   <li> LinearAvgVelTacticalStrategy </li>
     *   <li> CircularAvgVelTacticalStrategy </li>
     * </ul>
     *
     * @param theTarget - who I work for.
     */
    public TargetingStrategyManager(Target theTarget)
    {
	t = theTarget;
	TargetingStrategy aStrategy, initStrategy;
	int i=0;

	// Add all known TargetingStrategies to availableStrategies.
	aStrategy = new StationaryTargetingStrategy(t);
	availableStrategies.put(aStrategy.name, aStrategy);
	decodeStrategies.put(aStrategy.name, new Integer(i++));
	
	aStrategy = new LinearTargetingStrategy(t);
	availableStrategies.put(aStrategy.name, aStrategy);
	decodeStrategies.put(aStrategy.name, new Integer(i++));

	aStrategy = new CircularTargetingStrategy(t);
	availableStrategies.put(aStrategy.name, aStrategy);
	decodeStrategies.put(aStrategy.name, new Integer(i++));

	aStrategy = new LinearAvgVelTargetingStrategy(t);
	availableStrategies.put(aStrategy.name, aStrategy);
	decodeStrategies.put(aStrategy.name, new Integer(i++));

	aStrategy = new CircularAvgVelTargetingStrategy(t);
	availableStrategies.put(aStrategy.name, aStrategy);
	decodeStrategies.put(aStrategy.name, new Integer(i++));

	//aStrategy = new FireResponseDistTargetingStrategy(t);
	//availableStrategies.put(aStrategy.name, aStrategy);
	//decodeStrategies.put(aStrategy.name, new Integer(i++));

	// JiggleTargeting used only in aimHere if target is jiggling
	//aStrategy = new JiggleTargetingStrategy(t);
	//availableStrategies.put(aStrategy.name, aStrategy);
	decodeStrategies.put(JIGGLE, new Integer(i++));

	// PatternTargeting used only in aimHere if target is pattern-matched
	//aStrategy = new PatternTargetingStrategy(t);
	//availableStrategies.put(aStrategy.name, aStrategy);
	decodeStrategies.put(PATTERN, new Integer(i++));

	// Set initial strategies
	initStrategy = (TargetingStrategy)availableStrategies.get(INITIAL);
	if(initStrategy == null) initStrategy = aStrategy;
	t.currentTargetStrategy = initStrategy;

	for(i=0;i<TARGET_MODES;i++) {
	    timesFired[i] = 0;
	    timesHit[i] = 0;
	    energyLost[i] = 0;
	    energyGained[i] = 0;
	}
    }

    /**
     * Change current strategy to newStrategyName.  If this
     * is not found, go back to the INITIAL.
     *
     * @param newStrategyName
     */
    public void setStrategy(String newStrategyName)
    {
	if (newStrategyName != t.currentTargetStrategy.name) {
	    /*t.self.out.println("Switching from " + t.currentTargetStrategy.name + " to " +
			       newStrategyName + " for " + t.name);
	    */
	    TargetingStrategy newStrategy;
	    newStrategy =  (TargetingStrategy)availableStrategies.get(newStrategyName);
	    if(newStrategy == null) newStrategy =  (TargetingStrategy)availableStrategies.get(INITIAL);
	    t.currentTargetStrategy = newStrategy;
	}
    }

    public void updateCompletedBullet(AdvancedBullet ab)
    {
	double shotPower = ab.b.getPower();
	if(ab.targetMode.startsWith(GunManager.BUTTERFLY)) {
	    t.self.gunManager.shots[GunManager.BUTTERFLYID]++;
	    t.self.gunManager.energyLost[GunManager.BUTTERFLYID] += shotPower;
	    if(ab.b.getVictim() != null) {
		t.self.gunManager.energyGained[GunManager.BUTTERFLYID] += MathHelper.getShotEnergyGain(shotPower);
		t.self.gunManager.damageDone[GunManager.BUTTERFLYID] += MathHelper.getBulletDamage(shotPower);
	    }
	} else { // Assume AIMED
	    t.self.gunManager.shots[GunManager.AIMEDID]++;
	    t.self.gunManager.energyLost[GunManager.AIMEDID] += shotPower;
	    if(ab.b.getVictim() != null) {
		t.self.gunManager.energyGained[GunManager.AIMEDID] += MathHelper.getShotEnergyGain(shotPower);
		t.self.gunManager.damageDone[GunManager.AIMEDID] += MathHelper.getBulletDamage(shotPower);
	    }
	    String aimedMode = ab.targetMode.substring(GunManager.AIMED.length() + 1);
	    Integer iI = (Integer)decodeStrategies.get(ab.targetMode);
	    if(iI != null) {
		int i = iI.intValue();
		timesFired[i] += 1;
		energyLost[i] += ab.b.getPower();
		if(ab.b.getVictim() != null) {
		    if(ab.targetName == ab.b.getVictim()) {
			timesHit[i] += 1;
			energyGained[i] += MathHelper.getShotEnergyGain(ab.b.getPower());
		    }
		} else {
		    t.self.out.println("Ouch!");
		    //decideStrategyChange();
		}
	    }
	}

	if(ab.b.getVictim() == null) { // missed shot
	    t.self.gunManager.decideGunChange();
	}
    }

    public void decideStrategyChange()
    {
	Enumeration e = availableStrategies.elements();
	TargetingStrategy ts;
	Integer iI;
	int i;
	String bestStratName = "";
	double bestStratHitRate = 0.0;
	double curStratHitRate = 0.0;

	while(e.hasMoreElements()) {
	    ts = (TargetingStrategy)e.nextElement();
	    iI = (Integer)decodeStrategies.get(ts.name);
	    if(iI != null) {
		i = iI.intValue();
		if(timesFired[i] <= GRACE_PERIOD) curStratHitRate = DEFAULT_RATE;
		else curStratHitRate = timesHit[i] / timesFired[i];
		if(curStratHitRate > bestStratHitRate) {
		    bestStratHitRate = curStratHitRate;
		    bestStratName = ts.name;
		}
	    }
	}
	setStrategy(bestStratName);
    }
}
