/*
 * Written by Kinsen Choy
 * Version 1.8c
 */

package kinsen.melee;
import kinsen.melee.Guns.GunAvoidance;
import kinsen.melee.Guns.SimpleGuns;
import kinsen.melee.Movement.GridMovement;
import robocode.AdvancedRobot;
import robocode.BulletHitEvent;
import robocode.DeathEvent;
import robocode.HitByBulletEvent;
import robocode.HitRobotEvent;
import robocode.RobotDeathEvent;
import robocode.ScannedRobotEvent;
import robocode.WinEvent;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.util.ArrayList;

/*
 * Angsaichmophobia - Definition:
 *						Angstrom: One ten-billionth of a meter
 *						Aichmophobia: Fear of sharp or pointed objects
 *						Definitions from The Phrontistery <http://phrontistery.info>
 *					- Description:
 *						An advanced robot that is specialized for melees but performs well in duels also.
 *						Movement: General simple bullets avoidance similar to anti-gravity with bullet type weighting.
 *						Targeting: Fires at closest robot or robots that are disabled using simple guns and virtual bullets.
 */
public class Angsaichmophobia extends AdvancedRobot
{
	// Constants for challenges
	private static final boolean USE_GUN = true;
	private static final boolean USE_MOVEMENT = true;
	private static final boolean SET_FIRE_AMOUNT = false;
	private static final double FIRE_AMOUNT = -1;

	// Contains enemy details
	private static ArrayList enemies = new ArrayList();
	// Contains previous enemy energy (for bullet detection)
	private static ArrayList enemyPreviousEnergy = new ArrayList();
	// Contains enemy names
	private static ArrayList enemyNames = new ArrayList();
	// Focused enemy name
	private String currentEnemyName = null;
	// Focused enemy distance
	private double currentEnemyDistance = Double.POSITIVE_INFINITY;
	// Focused enemy object
	private Details enemy;
	// Focused enemy gun classes
	private static SimpleGuns enemyGuns;
	// Focused enemy index
	private int enemyIndex = -1;
	// Time focused enemy changed
	private long enemyChange = -1;

	// Robots that have died
	private ArrayList deadRobots = new ArrayList();

	// Radar variables
	// Time when radar sweep started
	private long sweepStart = 0;
	// Enemy sweep began with
	private String sweepBeginEnemy = null;
	// Last enemy scanned
	private String sweepLastScannedName = null;
	// Is doing a radar sweep
	private boolean blnRadarSweep = true;
	// Last fire time
	private long lastFiredTime = 0;
	// Last time an enemy was scanned
	private long lastScan = 0;

	// Guns for each robot
	private static ArrayList myGuns = new ArrayList();
	// Bullets that were fired
	private ArrayList bullets = new ArrayList();
	// Type of bullets fired
	private ArrayList bulletTypes = new ArrayList();
	// Enemy bullet was fired at
	private ArrayList bulletFiredAt = new ArrayList();

	// Robot movement
	private static GridMovement robotMovement;
	// Bullet avoidance
	private static GunAvoidance bulletAvoidance;
	// Bullets that have been fired by enemies (assumes head-on)
	private ArrayList enemyBullets = new ArrayList();
	// Type of bullets that have been fired
	private ArrayList enemyBulletTypes = new ArrayList();
	// Distance of closest bullets
	private ArrayList enemyBulletDistance = new ArrayList();

	// Constants
	private static final double MIN_DISTANCE_DIFF = 125;
	private static final double MIN_ENERGY_DIFF = 25;
	// Distance from wall to start wall smoothing
	private static final double MINIMUM_DISTANCE = 200;
	private static final int WALL_SMOOTHING_DISTANCE = 300;

	private static double battleFieldHeight, battleFieldWidth;

	public void run()
	{
		// Check if it can use newer functions
		try
		{
			if (AdvancedRobot.class.getMethod("setBodyColor", new Class[]{Color.class}) == null)
			{
				// Use older function
				setColors(new Color(100, 200, 120), new Color(150, 100, 125), new Color(75, 80, 100));
			}
			else
			{
				// Use newer functions
				setBodyColor(new Color(100, 200, 120));
				setGunColor(new Color(150, 100, 125));
				setRadarColor(new Color(75, 80, 100));

				setBulletColor(Color.white);
				setScanColor(new Color(120, 255, 120));
			}
		}
		catch (java.lang.NoSuchMethodException e)
		{
			// It could not set any colors so use older function by default
			setColors(new Color(100, 200, 120), new Color(150, 100, 125), new Color(75, 80, 100));
		}

		// Have robocode auto-adjust gun or radar
		setAdjustGunForRobotTurn(true);
		setAdjustRadarForGunTurn(true);
		setAdjustRadarForRobotTurn(true);

		// If first round then initialize once
		if (getRoundNum() == 0)
		{
			battleFieldHeight = getBattleFieldHeight();
			battleFieldWidth = getBattleFieldWidth();
			// Give arena boundaries to utilities class
			Utils.setBounds(battleFieldWidth, battleFieldHeight);
			for (int i = 0; i < getOthers(); i++)
			{
				myGuns.add(new SimpleGuns());
			}

			// Create new instance of movement class
			bulletAvoidance = new GunAvoidance();
			robotMovement = new GridMovement();
		}

		robotMovement.setVariables(this, enemyBullets, enemyBulletTypes, enemyBulletDistance, bulletAvoidance);

		// Start scanning for enemies
		turnRadarRight(360);

		while (true)
		{
			// If there was a radar slip or there are no enemies within radar range then it has to look for them
			if (getTime() - lastScan > 8)
			{
				setTurnRadarRight(Double.POSITIVE_INFINITY);
				move(100, 1, 0, true);
				execute();
			}
			else
			{
				// Always do movement to avoid bullets
				doMovement();
				execute();
			}
		}
	}

	/*
	 * onScannedRobot: Saves details to arraylist then tells robot to move
	 */
	public void onScannedRobot(ScannedRobotEvent e)
	{
		if (deadRobots.indexOf(e.getName()) != -1)
			return;
		// Saves details
		String enemyName = e.getName();
		enemyIndex = enemyNames.indexOf(enemyName);

		double enemyBearing = e.getBearing();
		double enemyHeading = e.getHeading();
		double enemyDistance = e.getDistance();
		double myHeading = getHeading();
		double absoluteBearing = enemyBearing + myHeading;
		double myX = getX();
		double myY = getY();
		double enemyX = getX() + Math.sin(Math.toRadians(absoluteBearing)) * enemyDistance;
		double enemyY = getY() + Math.cos(Math.toRadians(absoluteBearing)) * enemyDistance;
		double enemyEnergy = e.getEnergy();
		double enemyVelocity = e.getVelocity();

		lastScan = getTime();

		// Create details for enemy and me
		Details enemyDetails = new Details(enemyX, enemyY, enemyVelocity, enemyHeading, enemyEnergy);
		Details myDetails = new Details(myX, myY, getVelocity(), getHeading(), getEnergy());

		// Check if enemy exists
		if (enemyIndex == -1)
		{
			// New enemy -- add details and name
			enemies.add(enemyDetails);
			enemyNames.add(enemyName);
			enemyPreviousEnergy.add(new Double(enemyEnergy));
			enemyIndex = enemies.size() - 1;
		}
		else
		{
			enemyPreviousEnergy.set(enemyIndex, new Double(((Details) enemies.get(enemyIndex)).getEnergy()));
			enemies.set(enemyIndex, enemyDetails);
		}

		SimpleGuns currentGuns = (SimpleGuns) myGuns.get(enemyIndex);
		currentGuns.recordData(enemyDetails, myDetails, getTime());

		// If no other enemy then set to scanned
		if (enemy == null)
		{
			enemy = enemyDetails;
			currentEnemyDistance = enemyDistance;
			currentEnemyName = enemyName;
			enemyChange = getTime();
			enemyGuns = currentGuns;
		}
		else if (currentEnemyName.equals(enemyName))
		{
			// These have to be set even in case it was detected by on hit
			enemy = enemyDetails;
			currentEnemyDistance = enemyDistance;
			enemyGuns = currentGuns;
		}
		else
		{
			// Check if new enemy is much closer or if new enemy has much less energy
			if (enemyDistance < MINIMUM_DISTANCE || currentEnemyDistance - enemyDistance > MIN_DISTANCE_DIFF ||
				enemyEnergy == 0 || enemy.getEnergy() - enemyEnergy > MIN_ENERGY_DIFF)
			{
				if (!(currentEnemyDistance < MINIMUM_DISTANCE || enemy.getEnergy() == 0) && getTime() - enemyChange >= 16)
				{
					// Switch enemies
					enemy = enemyDetails;
					currentEnemyDistance = enemyDistance;
					currentEnemyName = enemyName;
					enemyChange = getTime();
					enemyGuns = currentGuns;
				}
			}
		}

		sweepLastScannedName = enemyName;

		// For movement
		robotMovement.recordData(enemyDetails, myDetails, getTime());

		doEvents();
	}

	/*
	 * doEvents: Handles the radar, gun, and robot movement
	 */
	private void doEvents()
	{
		doRadar();
		doGun();
		doMovement();
		execute();
	}

	/*
	 * doRadar: Handles radar
	 */
	private void doRadar()
	{
		// Focused
		// Sweep
		// If firing then start sweep
		if ((lastFiredTime > getTime() - 2) || (lastFiredTime < getTime() - 16) && !blnRadarSweep)
		{
			blnRadarSweep = true;
			sweepStart = getTime();
			sweepBeginEnemy = currentEnemyName;
		}

		// Degrees to turn gun
		double radarTurn = 0;

		// If only one other left or has scanned original enemy again
		if (getOthers() == 1 || (sweepLastScannedName.equals(currentEnemyName) && getTime() - sweepStart > 2))
		{
			blnRadarSweep = false;
			radarTurn = focusRadar();
			if (getOthers() > 1)
			{
				// Increase focus
				radarTurn += (radarTurn >= 0)? 14: -14;
			}
		}
		else
			radarTurn = 360;

		// Set gun turn
		setTurnRadarRight(radarTurn);
	}

	/*
	 * focusRadar: Focuses radar on best enemy
	 */
	private double focusRadar()
	{
		// Amount to turn to radar
		double turnAmount = Utils.normalRelativeAngle(Utils.angleFromPoint(getX(), getY(), enemy.getX(), enemy.getY()) - getRadarHeading());
		// Go past a little
		turnAmount += (turnAmount >= 0)? 1: -1;
		return turnAmount;
	}

	/*
	 * doGun: Handles gun turning and firing
	 */
	private void doGun()
	{
		int i = 0;
		// Check if any bullets are unneeded
		while (i < bullets.size())
		{
			// Actual bullet
			robocode.Bullet currentBullet = (robocode.Bullet) bullets.get(i);
			Details enemyFiredAt = (Details) enemies.get(((Integer) bulletFiredAt.get(i)).intValue());
			double bulletDistance = Utils.distance(currentBullet.getX(), currentBullet.getY(), enemyFiredAt.getX(), enemyFiredAt.getY());
			// Next bullet location
			double nextX = currentBullet.getX() + Math.sin(Math.toRadians(currentBullet.getHeading())) * currentBullet.getVelocity();
			double nextY = currentBullet.getY() + Math.cos(Math.toRadians(currentBullet.getHeading())) * currentBullet.getVelocity();
			double nextBulletDistance = Utils.distance(nextX, nextY, enemyFiredAt.getX(), enemyFiredAt.getY());
			// Check if bullet is inactive or it is getting farther away
			if (!currentBullet.isActive() || (bulletDistance < nextBulletDistance && bulletDistance > 60))
			{
				// Record bullet hit/miss
				SimpleGuns guns = (SimpleGuns) myGuns.get(((Integer) bulletFiredAt.get(i)).intValue());
				boolean bulletHit = (currentBullet.getVictim() != null);
				if (deadRobots.indexOf((String) enemyNames.get(((Integer) bulletFiredAt.get(i)).intValue())) == -1)
					guns.recordBullet((byte) ((Integer) bulletTypes.get(i)).intValue(), bulletHit);
				// Remove bullet data
				bullets.remove(i);
				bulletTypes.remove(i);
				bulletFiredAt.remove(i);
				i--;
			}
			i++;
		}
		double enemyDistance = Utils.distance(getX(), getY(), enemy.getX(), enemy.getY());
		// Fire power is enough to kill other robot times number of others halved
		double firePower;
		if (getOthers() == 1)
			firePower = Math.max(Math.min(enemy.getEnergy() / 4 * Math.min(Math.max(300 / enemyDistance, 0.25), 1), 3), 0.1);
		else
			firePower = Math.max(Math.min(enemy.getEnergy() / 4 * (getOthers() + 1) / 2, 3), 0.1);
		// If this is running out of energy and there are not too many robots, conserve
		if (enemyDistance > 70 && getEnergy() < 30)
			firePower = Math.min(firePower, getEnergy() / 10);
		// If enemy is too close then fire full power
		if (enemyDistance <= 60)
			firePower = 3;

		// If fixed amount
		if (SET_FIRE_AMOUNT)
			firePower = FIRE_AMOUNT;

		// Calculate target and fire power
		enemyGuns.calculateGun(firePower);
		Point fireAt = enemyGuns.getFireAt();
		firePower = enemyGuns.getFirePower();
		// Amount to turn gun
		double turnAmount = 0;
		if (fireAt != null)
			turnAmount = Utils.normalRelativeAngle(Utils.angleFromPoint(getX(), getY(), fireAt.getX(), fireAt.getY()) - getGunHeading());
		setTurnGunRight(turnAmount);
		robocode.Bullet firedBullet = null;
		// If it should use gun
		if (USE_GUN)
		{
			// Check if a bullet should be fired
			boolean willHaveMore = getEnergy() - firePower > enemy.getEnergy() + 1;
			if ((enemy.getEnergy() == 0 && willHaveMore) || getEnergy() > 8 || willHaveMore || enemyDistance < 180)
			{
				if (firePower > 0 && Math.abs(turnAmount) < 2)
					firedBullet = setFireBullet(firePower);
			}
		}

		// If a bullet was fired then record it
		if (firedBullet != null)
		{
			lastFiredTime = getTime();
			bullets.add(firedBullet);
			bulletTypes.add(new Integer(enemyGuns.getGunType()));
			bulletFiredAt.add(new Integer(enemyIndex));
			enemyGuns.firedBullet(true);
//			Utils.println(Utils.convertGunType(enemyGuns.getGunType()));
		}
	}

	/*
	 * doMovement: Handles robot's movement
	 */
	private void doMovement()
	{
		checkEnemyFiring();
		if (USE_MOVEMENT)
			robotMovement.calculateMovement();
	}

	/*
	 * checkEnemyFiring: Determines if an enemy fired a bullet
	 */
	private void checkEnemyFiring()
	{
		// Check for bullet fired
		// Check enemy that was scanned
		int scannedIndex = enemyNames.indexOf(sweepLastScannedName);
		if (scannedIndex != -1)
		{
			// Check if robot is not dead
			if (deadRobots.indexOf((String) enemyNames.get(scannedIndex)) == -1)
			{
				Details current = (Details) enemies.get(scannedIndex);
				double energyChange = ((Double) enemyPreviousEnergy.get(scannedIndex)).doubleValue() - current.getEnergy();
				// Check if previous energy - new energy is > 0 to detect bullets
				if (energyChange > 0)
				{
					// Set previous energy to current so that it does not trigger multiple bullets
					enemyPreviousEnergy.set(scannedIndex, new Double(current.getEnergy()));
					// Make sure that maximum power is 3
					double firePower = Math.min(energyChange, 3);
					long fireTime = getTime() - 1;
					// Calculate bullet headings
					bulletAvoidance.calculateGun(firePower);
					ArrayList locations = bulletAvoidance.getAllFireAt();
					// Add possible bullets
					for (int j = 0; j < locations.size(); j++)
					{
						// Add bullet
						Point currentLocation = (Point) locations.get(j);
						double fireHeading = Utils.angleFromPoint(current.getX(), current.getY(), currentLocation.getX(), currentLocation.getY());
						Bullet firedBullet = new Bullet(current.getX(), current.getY(), fireHeading, firePower, fireTime);
						enemyBullets.add(firedBullet);
						enemyBulletTypes.add(new Integer(j));
					}
				}
			}
		}
	}

	/*
	 * move: Actually moves the robot and the wall smoothing if needed
	 */
	public void move(double travelDistance, int direction, double turnAmount, boolean useWallSmoothing)
	{
		// Variables needed for wall smoothing
		double myHeading = getHeading();
		double myX = getX();
		double myY = getY();
		double myTurn = turnAmount;
		double newX = myX + Math.sin(Math.toRadians(myHeading + myTurn)) * WALL_SMOOTHING_DISTANCE * direction;
		double newY = myY + Math.cos(Math.toRadians(myHeading + myTurn)) * WALL_SMOOTHING_DISTANCE * direction;
		double originalX = newX;
		double originalY = newY;

		double positiveTurn = myTurn;
		double negativeTurn = 360;

		// Wall smoothing
		if (useWallSmoothing)
		{
			if (Utils.robotOutOfBounds(newX, newY))
			{
				while (Utils.robotOutOfBounds(newX, newY))
				{
					myTurn += 3;
					newX = myX + Math.sin(Math.toRadians(myHeading + myTurn)) * WALL_SMOOTHING_DISTANCE * direction;
					newY = myY + Math.cos(Math.toRadians(myHeading + myTurn)) * WALL_SMOOTHING_DISTANCE * direction;
				}
				positiveTurn = myTurn;

				myTurn = 0;
				newX = originalX;
				newY = originalY;
				while (Utils.robotOutOfBounds(newX, newY))
				{
					myTurn -= 3;
					newX = myX + Math.sin(Math.toRadians(myHeading + myTurn)) * WALL_SMOOTHING_DISTANCE * direction;
					newY = myY + Math.cos(Math.toRadians(myHeading + myTurn)) * WALL_SMOOTHING_DISTANCE * direction;
				}
				negativeTurn = myTurn;
			}

			if (positiveTurn < Math.abs(negativeTurn))
				myTurn = positiveTurn;
			else
				myTurn = negativeTurn;
		}

		myTurn = Utils.normalRelativeAngle(myTurn);

		if (travelDistance != 0)
		{
			setTurnRight(myTurn);
			// Have it slow down when doing large turns
			if (myTurn > 10)
				setAhead(travelDistance / (myTurn / 10));
			else
				setAhead(travelDistance);
		}
		else
			// If not moving then stop robot
			setAhead(0);
	}

	/*
	 * onBulletHit: Record new details about enemy if it does not have any enemy
	 */
	public void onBulletHit(BulletHitEvent e)
	{
		int index = enemyNames.indexOf(e.getName());
		if (index != -1)
		{
			Details previousDetails = (Details) enemies.get(index);
			Details newEnemyDetails = new Details(previousDetails.getX(), previousDetails.getY(), previousDetails.getVelocity(),
													previousDetails.getHeading(), e.getEnergy());
			enemies.set(index, newEnemyDetails);
		}
	}

	/*
	 * onHitRobot: Record new details if it has not changed enemies often
	 */
	public void onHitRobot(HitRobotEvent e)
	{
		if (enemyChange < getTime() - 16 && getOthers() > 1)
		{
			currentEnemyName = e.getName();
			currentEnemyDistance = 0;
			double absoluteBearing = e.getBearing() + getHeading();
			double enemyX = getX() + Math.sin(Math.toRadians(absoluteBearing)) * 50;
			double enemyY = getY() + Math.cos(Math.toRadians(absoluteBearing)) * 50;
			enemy = new Details(enemyX, enemyY, 0, 0, e.getEnergy());
			enemyChange = getTime();
		}
	}

	/*
	 * onHitByBullet: When hit by a bullet then record type of bullet
	 */
	public void onHitByBullet(HitByBulletEvent e)
	{
		// Distance of closest bullet
		double closestDistance = Double.POSITIVE_INFINITY;
		// Index number of closest bullet
		int closestBullet = -1;
		for (int i = 0; i < enemyBulletDistance.size(); i++)
		{
			// Check if bullet is closer
			double currentDistance = ((Double) enemyBulletDistance.get(i)).doubleValue();
			if (currentDistance < closestDistance)
			{
				closestDistance = currentDistance;
				closestBullet = i;
			}
		}

		// If there was a close bullet then record it
		if (closestBullet != -1 && closestDistance < 80)
			bulletAvoidance.recordHit(((Integer) enemyBulletTypes.get(closestBullet)).intValue());
	}

	/*
	 * onRobotDeath: Record that the robot is dead and remove from current enemy if it was the robot
	 */
	public void onRobotDeath(RobotDeathEvent e)
	{
		// Add name to list of dead robots
		deadRobots.add(e.getName());
		// The robot we were focusing on has died, switch robots and spin radar
		if (currentEnemyName.equals(e.getName()))
			enemy = null;
	}

	/*
	 * onDeath: Call clean up
	 */
	public void onDeath(DeathEvent e)
	{
		removeOldData();
	}

	/*
	 * onWin: Continue avoiding any bullets
	 */
	public void onWin(WinEvent e)
	{
		removeOldData();

		// Victory dance
		while (true)
		{
			setTurnRadarRight(focusRadar());
			setTurnGunRight(Utils.normalRelativeAngle(Utils.angleFromPoint(getX(), getY(), enemy.getX(), enemy.getY()) - getGunHeading()));
			setFire(Math.random() * 3);
			move(3 * ((getTime() / 5) % 10 + 1), 1, Math.cos(Math.toRadians(getTime() * 15)) * 30, true);
			execute();
		}
	}

	/*
	 * removeOldData: Calls gun targeting to clean up
	 */
	private void removeOldData()
	{
		for (int i = 0; i < myGuns.size(); i++)
		{
			((SimpleGuns) myGuns.get(i)).newRound();
		}
		robotMovement.newRound();
	}

	/*
	 * onPaint: Debugging
	 */
	public void onPaint(Graphics2D g)
	{
		// Show enemy
		long currentTime = getTime();
		g.setColor(Color.yellow);
		if (enemy != null)
		{
			g.drawOval((int) (enemy.getX() - 20), (int) (enemy.getY() - 20), 40, 40);
		}
		// Show bullets to avoid
		g.setColor(Color.green);
		for (int i = 0; i < enemyBullets.size(); i++)
		{
			Bullet currentBullet = (Bullet) enemyBullets.get(i);
			g.drawOval((int) (currentBullet.getCurrentX(currentTime) - 5), (int) (currentBullet.getCurrentY(currentTime) - 5), 10, 10);
		}
		// Show virtual bullets
		if (enemyGuns != null)
			enemyGuns.debug(g);
		robotMovement.debug(g);
		// Print to screen the output
		g.setColor(Color.white);
		Utils.drawText(g);
	}
}