package marksteam;
import robocode.*;
import java.util.*;
import java.awt.geom.*;
import java.awt.Color;

public class Phoenix extends AdvancedRobot
{	
	final double PI = Math.PI;					//constant
	public double firePower;					//power of shot Phoenix will be using
	public double dodgeDistance = 50;			//distance to move when bulletFired event occurs
	public int direction = 1;
	
	public double oldEnergy = 100; 				//last recorded energy of enemy bot
	public double newEnergy = 100;				//current energy of enemy bot - refreshes when bulletFired event occurs
	public int missCount = 0;					//keeps track of how accurate tracking is	
		
	public double lastEnemyHeading;				//last recorded heading of enemy - checks for circular movement	
	opponentBot target;							//create new opponentBot object

	
	public void run()
	{
		target = new opponentBot();
		target.distance = 1000000;				//start searching for new target immediately
		
		setColors(Color.red, Color.black, Color.red);
		
		/**set radar and gun to turn independently of robot**/
		setAdjustGunForRobotTurn(true);
		setAdjustRadarForGunTurn(true);
					
		turnRadarRightRadians(2*PI);			//turn the radar a full rotation
		
		/**bulletFired event created here - tests to see if bullet has been fired**/
		addCustomEvent(
			new Condition("bulletFired"){
            	public boolean test(){
                	double delta = oldEnergy - newEnergy;
                	oldEnergy = newEnergy;
                
					/**if energy has changed, a bullet has probably been fired**/
					return 0.0 < delta && delta <= 3.0;                	
           		 };
        	}
		);
		
		while(true){
			/**turns Phoenix perpendicular with enemy bot to allow for more effective dodging**/
			setTurnRightRadians(target.bearing + (PI/2));
			
			if(getOthers() > 1)					//for melee battles, use constant avoidance rather than dodging
				doEvade();
			
			doFirePower();						//select the fire power
			doScan();							//Oscillate the scanner over the bot
			doGun();							//move the gun to predict where the enemy will be
			if (getGunHeat() == 0 && getEnergy() > 0.1)				//if gun is capable of firing, fire
				fire(firePower);
			execute();							//execute all commands
		}
	}
	
	/** 
	  *	this method is only used in battles with more than one opponent, as dodging is not
	  * very effective with multiple enemy bots (we tend to get blasted by 'corners', for instance)
	  */	
	public void doEvade() 
	{
		if(getTime()%20 == 0){ 												//every ten ticks of the game timer:
			direction *= -1;												//reverse direction
			setAhead(direction*(100 + Math.random()*50));					//move in that direction
		}
	}
	
	
			
	
	/**this method selects an appopriate bullet power to use based on distance from target**/
	/**Note: for 1v1 battles, especially against spinbot, power 3 will suffice**/
	private void doFirePower()
	{
		if(target.movementType.equals("linear") && target.distance > 50)
			firePower = 2;
		else
			firePower = 3;
						
		if (target.velocity == 0)
			firePower = 3;
			
		if(getEnergy() <= 3.0)
			firePower = 0.1;
	}
		
		
	/**This method points the radar directly at the opponent bot, so the tracking algorithms can work**/	
	private void doScan()
	{
		double radarMove;
		
		if (getTime() - target.lastScanTime > 4){				//if Phoenix hasn't detected any enemy bots,
			radarMove = 4*PI;									//rotate the radar to find a target
		}else{
			/**set radarMove to keep radar on current target**/
			radarMove = getRadarHeadingRadians() - (PI/2 - Math.atan2(target.y - getY(), target.x - getX()));
			
			/**the following code causes the rader to oscillate backwards and forwards to ensure the target is not lost**/
			radarMove = normalRelativeAngle(radarMove);
			if (radarMove < 0)
				radarMove -= PI/10;
			else
				radarMove += PI/10;
		}
		
		setTurnRadarLeftRadians(radarMove);				//turn the radar the calculated amount
	}
	
	
	/**this method sets the targeting of the main gun in occordance with the tracking method to be used**/
	private void doGun()
	{
		long time, nextTime;
		Point2D.Double p = new Point2D.Double(target.x, target.y);
		
		/**the following loop performes 10 iterations to determine exactly how long it will take
		   for a bullet to reach the target at the speed set in doFirepower**/
		for (int i = 0; i < 10; i++){
       		nextTime = (int) Math.round((Point2D.distance(getX(), getY(), p.x, p.y)/(20 - (3*firePower))));
			time = getTime() + nextTime;
        	p = target.predictPosition(time);
		}
		
		/**turns the gun for the next shot based on targeting provided by the enemy class**/
		double gunMove = getGunHeadingRadians() - (PI/2 - Math.atan2(p.y - getY(), p.x -  getX()));
		setTurnGunLeftRadians(normalRelativeAngle(gunMove));
	}

		
				
	/**
     * onCustomEvent handler
     */
    public void onCustomEvent(CustomEvent e)
    {
    	if(getOthers() > 1)					//if this is a melee battle, we want constant dodging 
			return;
	   			   
	  	/**if bullet has been fired, dodge either side**/
		if (e.getCondition().getName().equals("bulletFired")) {
   	    	if (Math.random() <= 0.5)
            	setAhead(dodgeDistance + Math.random()*30);
            else
                setBack(dodgeDistance + Math.random()*30);
        }
    }
		
	public void onRobotDeath(RobotDeathEvent e)
	{
		if (e.getName() == target.name){
			out.println(target.name + " eliminated.");
			target.distance = 10000; 			//search for a new target
		}
	}
	
	/**
	 * onWin:  Do a victory dance
	 */	
	public void onWin(WinEvent e) {
		out.println("All opponents eliminated. Combat successful.");
		for (int i = 0; i < 50; i++)
		{
			turnRight(30);
			turnLeft(30);
		}
	}
		
	public void onHitByBullet(HitByBulletEvent e)
	{
		if(getEnergy() < 30)
			out.println("Energy levels critical.");
	}
		
		
	public void onHitWall(HitWallEvent e)
    {
		/**move clear of wall so there is enough room to dodge incoming bullets**/
        setTurnRightRadians(normalRelativeAngle(e.getBearingRadians() + PI));
        ahead(150);
    }
	
	
	/**
     * onScannedRobot: Focus on opponent robot and fire with lead for linear and circular movement
     */
	public void onScannedRobot(ScannedRobotEvent e)
	{
		/**if closer target has been found, or existing target is still in view**/
		if((e.getDistance() < target.distance) || (e.getName() == target.name)){
			if(e.getName() != target.name)
				out.println("Targeting: " + e.getName());
				
			/**refresh enemy energy reading**/
			newEnergy = e.getEnergy();
		
			/**calculate absolute bearing to target bot**/
			double absoluteBearing = (getHeadingRadians() + e.getBearingRadians())%(2*PI);
		
			/**set all information about target**/
			target.name = e.getName();
			double w = normalRelativeAngle(e.getHeadingRadians() - target.heading);
			target.angularVelocity = w/(getTime() - target.lastScanTime);
			
			if(Math.abs(target.angularVelocity) > 0.0001)
				target.movementType = "circular";
			else
				target.movementType = "linear";
		
			target.x = getX() + Math.sin(absoluteBearing)*e.getDistance(); 	//x coordinate of target
			target.y = getY() + Math.cos(absoluteBearing)*e.getDistance(); 	//y coordinate of target
			
			target.bearing = e.getBearingRadians();
			target.heading = e.getHeadingRadians();
			target.lastScanTime = getTime();								//game time at which this scan occured
			target.velocity = e.getVelocity();
			target.distance = e.getDistance();
		}
	}	
	
	
	public void onBulletMissed(BulletMissedEvent e)
	{
		missCount++;
		
		if(missCount > 4){
			if(target.targetingAccurate)
				target.targetingAccurate = false;
			else
				target.targetingAccurate = true;
		}
	}
	
	
	public void onBulletHit(BulletHitEvent e)
	{
		missCount = 0;
	}
		
		
	/*restricts an angle to the range -PI/2 <= Angle <= PI/2**/
	private double smallestToParallel(double angle)
	{
		if (angle > PI/2.0)
			angle -= PI;
		if (angle < -PI/2.0)
			angle += PI;
		return angle;
	}
	
	/*restricts an angle to the range -PI <= Angle <= PI**/
	private double normalRelativeAngle(double angle)
	{
		if (angle > PI)
			angle -= 2*PI;
		if (angle < -PI)
			angle += 2*PI;
		return angle;
	}
	
}


/*
 * object to define essential information about enemy bot, with useful methods for
 * calculating distance from Phoenix
 */
class opponentBot
{
	String name;
	String movementType;
	
	public double bearing;				
	public double heading;				
	public long lastScanTime;						//Time the last scan occured
	public double velocity;		
	public double x, y;								//x and y co-ordinates of enemy bot
	public double distance;							//distance to enemy boy
	public double angularVelocity;
	
	public boolean targetingAccurate = true;		//if targeting is not effective, use basic
	
	
	/*
	 * Method for predicting the next position of the enemy bot. Uses both circular
	 * and linear prediction methods. May add other methods if there is time (rapid movement?)...
	 */
	public Point2D.Double predictPosition(long newScanTime)
	{
		double changeInTime = newScanTime - lastScanTime;
		double newX, newY;
		
		if(movementType.equals("circular") && targetingAccurate){
			/**if angularVelocity is significant enough to imply circular movement, use circular targeting**/
			double radius = velocity/angularVelocity;
			double changeInHeading = changeInTime*angularVelocity;
			
			newX = x + Math.cos(heading)*radius - Math.cos(heading + changeInHeading)*radius;
			newY = y + Math.sin(heading + changeInHeading)*radius - Math.sin(heading)*radius;
		} else if (movementType.equals("linear") && targetingAccurate){
			/**if angularVelocity is insignificant, use simple linear targeting**/
			newX = x + Math.sin(heading)*velocity*changeInTime;
			newY = y + Math.cos(heading)*velocity*changeInTime;
		} else {
			/**if neither circular nor linear targeting is working, use simple targeting**/
			newX = x;
			newY = y;
		}
		
		return new Point2D.Double(newX, newY);
	}
}
			
			
		