package tzu.gun;

import java.util.*;
import tzu.util.*;
import tzu.intel.*;
import tzu.strategy.*;
import robocode.*;

/**
 * Provides one method, calcFireSolution, which determines
 * the best aiming technique to use, calls it, and returns a
 * FireSolution object which can be used to aim at (and hopefully
 * hit) the given target.
 */

public class AimManager extends AbstractManager {

    static final int PREDICT_TRIAL_SHOTS    = 20;
    static final int PATTERN_TRIAL_SHOTS    = 15;
    static final int ESTIMATE_TRIAL_SHOTS   = 5;

    static final double MIN_PREDICT         = 8.0;
    static final double MIN_PATTERN         = 4.0;
    static final double MIN_ESTIMATE        = 1.0;
    static final double GOOD_HIT_PERCENTAGE = 40.0;
    static Random random                    = new Random();
    static int  enemyCount;


    public AimManager(AdvancedRobot ar, TargetStrategyInterface tsi) {
        super(ar, tsi);
        enemyCount = ar.getOthers();
    }


    /** Overriden to do nothing. */
    public void reinitialize() {}
    /** Overriden to do nothing. */
    public void takeTurn() {}


    /**
     * Calculate the best fire solution to maximize our chances of hitting
     * this target.
     */
    public FireSolution calcFireSolution(Bot target, double firePower) {

        int     randomNumber        = random.nextInt(100);
        double  randomDouble        = random.nextDouble() * 2.0;
        Reaction reaction           = target.getPreviousReaction();

        double  predictPct  = Math.max(target.getPredictHitPct(), MIN_PREDICT);
        double  estimatePct = Math.max(target.getEstimateHitPct(), MIN_ESTIMATE);
        double  patternPct  = Math.max(target.getPatternHitPct(), MIN_PATTERN);

        int     predictShotCount    = target.getPredictAimShotAtCount();
        int     patternShotCount    = target.getPatternAimShotAtCount();
        int     estimateShotCount   = target.getEstimateAimShotAtCount();

        Motion bullet = new Motion(
                myRobot.getX(),
                myRobot.getY(),
                0, firePower);

        /*
         * Allow a number of trial shots to establish
         * a hit percentage for this aiming technique.
         */
        if (reaction            !=  null                &&
            predictShotCount    <   PREDICT_TRIAL_SHOTS &&
           (predictShotCount    <=  patternShotCount    ||
            predictShotCount    <=  estimateShotCount)) {

            return PredictAim.calculate(target, reaction, bullet);
        }


        /*
         * Allow a number of trial shots to establish
         * a hit percentage for this aiming technique.
         */
        if (target.getTimePlayed() > 250            &&
            patternShotCount < PATTERN_TRIAL_SHOTS  &&
            patternShotCount <= estimateShotCount)   {

            return PatternAim.calculate(new Motion(target), bullet);
        }


        /*
         * Allow a number of trial shots to establish
         * a hit percentage for this aiming technique.
         * Skip if in a melee situation.
         */
        if (enemyCount == 1 &&
            estimateShotCount < ESTIMATE_TRIAL_SHOTS) {
            return EstimateAim.calculate(new Motion(target), bullet);
        }


        //-------------------------------------------------------------------
        // After the trial shots have been fired and hit percentages
        // established, use the aiming technique that works best.  Also use
        // the other techniques occasionally, so we do not get locked into
        // using just one aiming algorithm.
        //
        // However, if we have an aiming algorithm that works REALLY well,
        // don't bother trying the other techniques.
        //-------------------------------------------------------------------


        /*
         * When predicted aiming works best:
         */
        if (reaction   != null         &&
            predictPct >= patternPct   &&
            predictPct >= estimatePct) {

            if (predictPct < GOOD_HIT_PERCENTAGE) {

                if (patternPct >= estimatePct &&
                    randomNumber <= patternPct) {

                    if (estimatePct / patternPct <= randomDouble) {
                        return PatternAim.calculate(new Motion(target), bullet);
                    }
                    return EstimateAim.calculate(new Motion(target), bullet);
                }

                if (estimatePct > patternPct &&
                    randomNumber <= estimatePct) {

                    if (patternPct / estimatePct <= randomDouble) {
                        return EstimateAim.calculate(new Motion(target), bullet);
                    }
                    return PatternAim.calculate(new Motion(target), bullet);
                }
            }
            return PredictAim.calculate(target, reaction, bullet);
        }


        /*
         * When pattern aiming works best:
         */
        if (patternPct >= estimatePct) {

            if (patternPct < GOOD_HIT_PERCENTAGE) {

                if (predictPct >= estimatePct &&
                    reaction != null &&
                    randomNumber <= predictPct) {

                    if (estimatePct / predictPct < randomDouble) {
                        return PredictAim.calculate(target, reaction, bullet);
                    }
                    return EstimateAim.calculate(new Motion(target), bullet);
                }

                if (estimatePct > predictPct && randomNumber <= estimatePct) {

                    if (predictPct / estimatePct <
                        randomDouble || reaction == null) {

                        return EstimateAim.calculate(new Motion(target), bullet);
                    }
                    return PredictAim.calculate(target, reaction, bullet);
                }
            }
            return PatternAim.calculate(new Motion(target), bullet);
        }


        /*
         * When estimated aiming works best:
         */

        if (estimatePct < GOOD_HIT_PERCENTAGE) {

            if (predictPct >= patternPct &&
                reaction != null &&
                randomNumber <= predictPct) {

                if (patternPct / predictPct < randomDouble) {
                    return PredictAim.calculate(target, reaction, bullet);
                }
                return PatternAim.calculate(new Motion(target), bullet);
            }

            if (patternPct > predictPct && randomNumber < patternPct) {

                if (predictPct / patternPct <
                    randomDouble || reaction == null) {

                    return PatternAim.calculate(new Motion(target), bullet);
                }
                return PredictAim.calculate(target, reaction, bullet);
            }
        }
        return EstimateAim.calculate(new Motion(target), bullet);
    }
}