package ags.lunartwins.util;

import robocode.*;
import ags.lunartwins.LunarBase;

/**
 * Quick representation of an enemy robot
 * 
 * @author Alexander Schultz
 */
public class EnemyRobot implements java.io.Serializable { 
    private static final long serialVersionUID = 1L;
    public String name;
    public RelativePoint relativelocation;
    public AbsolutePoint location;
    public RelativePoint velocity;
    public double angularvelocity;
    public long time;
    
    public EnemyRobot() {}
    
    private EnemyRobot(EnemyRobot old) {
        name = old.name;
        relativelocation = old.relativelocation;
        location = old.location;
        velocity = old.velocity;
        angularvelocity = old.angularvelocity;
        time = old.time;
    }
    
    public void onScannedRobotEvent(LunarBase bot, ScannedRobotEvent e) {
        name = e.getName();
        relativelocation = RelativePoint.fromDM(e.getBearingRadians()+bot.getHeadingRadians(), e.getDistance());
        location = AbsolutePoint.fromXY(bot.getX(), bot.getY()).addRelativePoint(relativelocation);
        // Rolling average angular velocity
        if (velocity != null) {
            angularvelocity = (angularvelocity*3 + velocity.getDirDist(e.getHeadingRadians())) / 4;
        }
        velocity = RelativePoint.fromDM(e.getHeadingRadians(), e.getVelocity());
        time = e.getTime();
    }
    
    public void predictTick() {
        velocity = velocity.incrementDirMag(-angularvelocity, 0);
        location = location.addRelativePoint(velocity);
        relativelocation = relativelocation.addRelativePoint(velocity);
        
        if (processWallCollision(location, velocity.getDirection()))
            velocity.setDirectionMagnitude(velocity.getDirection(), 0);
        time++;
    }
    
    public AbsolutePoint predict(AbsolutePoint from, double bulletpower) {
        long start = time;
        //AbsolutePoint oldl = location;
        while ((location.distance(from)-ROBOT_SIZE) > ((time-start)*(20-3*bulletpower)))
            predictTick();
        //AbsolutePoint newl = location;
        //location = oldl;
        return location;
    }
    
    public Object clone() {
        return new EnemyRobot(this);
    }
    
    private static final int ROBOT_SIZE = 18;
    
    public static boolean processWallCollision(AbsolutePoint goal, AbsolutePoint start) {
        return processWallCollision(goal, RelativePoint.fromPP(goal, start).getDirection());
    }
    
    public static boolean processWallCollision(AbsolutePoint location, double direction) {
        boolean hitWall = false;
        double fixx = 0, fixy = 0;
    
        if (Math.round(location.getX()) > LunarBase.bfwidth - ROBOT_SIZE) {
            hitWall = true;
            fixx = LunarBase.bfwidth - ROBOT_SIZE - location.getX();
        }
        if (Math.round(location.getX()) < ROBOT_SIZE) {
            hitWall = true;
            fixx = ROBOT_SIZE - location.getX();
        }
        if (Math.round(location.getY()) > LunarBase.bfheight - ROBOT_SIZE) {
            hitWall = true;
            fixy = LunarBase.bfheight - ROBOT_SIZE - location.getY();
        }
        if (Math.round(location.getY()) < ROBOT_SIZE) {
            hitWall = true;
            fixy = ROBOT_SIZE - location.getY();
        }
        
        double tanHeading = Math.tan(direction);
        double fixxtanHeading = fixx / tanHeading;
        double fixytanHeading = fixy * tanHeading;
        
        // if it hits bottom or top wall
        if (fixx == 0) {
            fixx = fixytanHeading;
        } // if it hits a side wall
        else if (fixy == 0) {
            fixy = fixxtanHeading;
        } // if the robot hits 2 walls at the same time (rare, but just in case)
        else if (Math.abs(fixxtanHeading) > Math.abs(fixy)) {
            fixy = fixxtanHeading;
        } else if (Math.abs(fixytanHeading) > Math.abs(fixx)) {
            fixx = fixytanHeading;
        }
        
        location.setLocation(location.x+fixx, location.y+fixy);
        
        return hitWall;
    }
    
    public static EnemyRobot gotScan(EnemyRobot old, LunarBase bot, ScannedRobotEvent e) {
        if (old == null)
            old = new EnemyRobot();
        old.onScannedRobotEvent(bot, e);
        return old;
    }
}
