/*
 * Copyright (c) 2004 Marcin Fuszara
 */  
package fushi.PvP1;
import robocode.*;

/* This movement makes the robot move around the center of the arena
 * and stay away from the opponent by moving on the other side of
 * the battlefield;
 */
public class CenterOrbitalMovement extends Movement
{

   // margin from the borders of arena and its center beyond which the robot does not move
   static final double margin = 20;
   // minimum angle [radians] to enemy that the robot tries to keep
   static final double minEnemyAngleDistance = Math.toRadians( 120 );
   // maximum change in angle [radians]
   static final double maxAngleChange = Math.toRadians( 50 );
   // maximum change in distance to arena center
   static final double maxCenterDistanceChange = 20;

   // robot for which this movement is used
   AdvancedRobot robot;
   // robot's body
   Body body;
   // enemy to avoid
   Enemy enemy;
   
   // data about the battlefield
   Vector2d arenaSize;
   Vector2d arenaCenter;
   // minimum distance from center of arena at which the robot moves
   double minCenterDistance;
   
   // if this value is set to true, next run method will select a new destination point
   boolean selectNewDestination = true;
   
   /* at construction provide robot, its body
    * and enemy in one-on-one battle
    */
   public CenterOrbitalMovement( AdvancedRobot robot, Body body, Enemy enemy )
   {
      this.robot = robot;
      this.body = body;
      this.enemy = enemy;
      initialize();
   }

   /* prepare data about the battlefield
    */
   void initialize()
   {
      arenaSize = new Vector2d( robot.getBattleFieldWidth(), robot.getBattleFieldHeight() );
      arenaCenter = new Vector2d( arenaSize );
      arenaCenter.div( 2 );
      
      // determine minimum and maximum distances from arena center
      minCenterDistance = 100;
   }
   
   /* prepare orders
    */
   public void run()
   {
         
      // allow robot to finish it's previous move
      if ( body.destinationClose() )
         selectNewDestination = true;
         
      // let the robot reach its previous destination
      if ( !selectNewDestination )
         return;

      // determine enemy position
      Vector2d enemyPosition = new Vector2d();
      // check if enemy position is known
      if ( enemy.getLogLength() == 0 ) {
         // enemy not scanned at all; assume random position
         enemyPosition.x = Util.randomDouble( 0, arenaSize.x );
         enemyPosition.y = Util.randomDouble( 0, arenaSize.y );
      } else {
         // get the latest known enemy position
         enemyPosition = new Vector2d( enemy.getData( 0 ).position );
      }
      // get the robot position
      Vector2d robotPosition = new Vector2d( robot.getX(), robot.getY() );
      
      // determine angles at which the robot and the enemy are located
      double enemyAngle = getAngleToCenter( enemyPosition );
      double robotAngle = getAngleToCenter( robotPosition );
   
      // choose new angle
      double newAngle;   
         
      // check if current angle to enemy is not too small
      double enemyAngleRelative = Util.normalRelativeAngle( robotAngle - enemyAngle );
      if ( Math.abs( enemyAngleRelative ) < minEnemyAngleDistance ) {  // enemy too close
         double angleChange = Util.randomDouble( 0, maxAngleChange );
         if ( enemyAngleRelative > 0 )    // try to escape enemy
            newAngle = robotAngle + angleChange;
         else
            newAngle = robotAngle - angleChange;
      } else {     // enemy angle distance ok
         double angleChange = Util.randomDouble( -maxAngleChange, maxAngleChange );
         newAngle = robotAngle + angleChange;
      }
 
      // count robot's distance to arena center
      double centerDistance = Vector2d.distance( robotPosition, arenaCenter );
      
      // count maximum distance to arena center based on newAngle
      double maxCenterDistance = Math.abs( Math.sin( newAngle ) ) * arenaSize.x +
                                 Math.abs( Math.cos( newAngle ) ) * arenaSize.y + margin;
      
      // choose new distance to arena center
      double newCenterDistance = centerDistance + 
         Util.randomDouble( -maxCenterDistanceChange, maxCenterDistanceChange );
      if ( newCenterDistance > maxCenterDistance )
         newCenterDistance = maxCenterDistance - margin;
      else if ( newCenterDistance < minCenterDistance )
         newCenterDistance = minCenterDistance + margin;
      
      // select new destination based on newAngle and newCenterDistance
      Vector2d destination = new Vector2d();
      destination.setFromAngleDistance( newAngle, newCenterDistance );
      destination.add( arenaCenter );
      
      // make order
      body.goTo( destination );
           
      
      // do not select new destination next frame
      selectNewDestination = false;
   }
   
   /* when hit enemy, change destination
    */
   public void onHitRobot( HitRobotEvent e )
   {
      selectNewDestination = true;      
   }
   
   /* given point position on the arena determines the angle to center of arena
    */
   double getAngleToCenter( Vector2d point )
   {
      // calculate point position relative to arena center
      Vector2d pointRelative = new Vector2d( point );
      pointRelative.sub( arenaCenter );
      // determine angle from the center
      return pointRelative.angle();
   }
   
}