/*
 * Copyright (c) 2004 Marcin Fuszara
 */  
package fushi.PvP1;
import robocode.*;
import java.util.LinkedList;

/* this pattern matcher is to be used with the enemy class;
 * it records the enemy's history, allows to find matches
 * in the history and predict future movement;
 */
public class PatternMatcher
{
   // maximum length of pattern data log
   static final int maxPatternLogLength = 384;
    
   // constants used in grading matches
   static final double minSpeed = -8;
   static final double maxSpeed = 8;
   static final double minHeadingDelta = Math.toRadians( -10 ); 
   static final double maxHeadingDelta = Math.toRadians( 10 );
   // ranges used to normalize grading
   static final double speedSpan = maxSpeed - minSpeed;
   static final double headingDeltaSpan = maxHeadingDelta - minHeadingDelta;
     
   // a single record of data for pattern matching
   public class PatternData
   {
      public double speed;          // robot velocity
      public double headingDelta;   // change in heading
      
      public PatternData( double speed, double headingDelta )
      {
         this.speed = speed;
         this.headingDelta = headingDelta;
      }
   }
   // log of pattern data
   LinkedList patternLog;
   
   // enemy for which this matcher records the history
   Enemy enemy;

   /* at construction provide the enemy to record
    */
   public PatternMatcher( Enemy enemy )
   {
      this.enemy = enemy;
      
      // create a log for pattern matching
      patternLog = new LinkedList();
   }
   
   /* get data at specified index;
    * \return null if data at given index do not exist
    */
   public PatternData getData( int index )
   {
      if ( index < 0 || index >= patternLog.size() )
         return null;
      else
         return (PatternData)patternLog.get( index );
   }
   
   /* returns the length of patternLog
    */
   public int getLogLength()
   {
      return patternLog.size();
   }
   
   /* a match type is a record returned from a method
    * findMatch;
    */
   public class Match
   {
      public int index;   //< index at which the match was found
      public double rank; //< match rank in range 0..1
      
      public Match( int index, double rank )
      {
         this.index = index;
         this.rank = rank;
      }
   }
   
   /* finds a best match of the last 'length' records
    * in the whole patternLog;
    * \return the best match; null if finding a match was not possible;
    */
   public Match findMatch( int length )
   {
      int logLength = patternLog.size(); 
      if ( logLength < 2 * length ) // there will be no sequence long enough to match
         return null;
         
      // find index of pattern beginning
      int patternBegin = logLength - length;
      
      double maxRank = -1;
      int maxRankIndex = 0;
      
      double rank;
      int i;  //< index in log
      int j;  //< index in pattern
      PatternData patternRecord, logRecord; // compared records
      double speedDiff, headingDeltaDiff;   // differences in values
      
      for ( i = 0; i < patternBegin - length - 1; ++i ) {
         rank = 0;
         // count rank of match at index i
         for ( j = 0; j < length; ++j ) {
            logRecord = (PatternData)patternLog.get( i + j );
            patternRecord = (PatternData)patternLog.get( patternBegin + j );
            // compute values differences
            speedDiff = Math.abs( logRecord.speed - patternRecord.speed );
            headingDeltaDiff = Math.abs( logRecord.headingDelta - patternRecord.headingDelta );
            
            // normalize differences;
            speedDiff /= speedSpan;
            headingDeltaDiff /= headingDeltaSpan;
            // remultiply differences to get square error
            speedDiff *= speedDiff;
            headingDeltaDiff *= headingDeltaDiff;
            // sum up differences to the rank
            rank += speedDiff + headingDeltaDiff;
         }
         // normalize rank
         rank /= length * 2;
         rank = 1 - rank;
         
         // check if better than found earlier
         if ( rank > maxRank ) {
            maxRank = rank;
            maxRankIndex = i;
         }
      }
      
      // return found match
      return new Match( maxRankIndex, maxRank );
   }
   
   /* assuming that the enemy log is not empty, uses the last two entries
    * to update the pattern log;
    * if the last two entries do not come from consecutive frames
    * the missing frames in the pattern log are interpolated;
    */
   public void updatePatternLog()
   {
      if ( enemy.getLogLength() < 2 ) // not enough data to update the matcher
         return;
         
      // get the last two observed data records; data0 is the older one
      Enemy.Data data0 = enemy.getData( 1 );
      Enemy.Data data1 = enemy.getData( 0 );
      
      // determine the number of frames between the two last records
      long numFrames = data1.time - data0.time;
      
      if ( numFrames < 1 ) // something is wrong!
         return;
         
      // determine the difference in heading and speed
      double speedDifference = data1.speed - data0.speed;
      double headingDifference = Util.normalRelativeAngle( data1.heading - data0.heading );
      
      // calculate deltas distributed for each frame
      double speedDelta = speedDifference / numFrames;
      double headingDelta = headingDifference / numFrames;
      
      // add frames to pattern log
      for ( int i = 0; i < numFrames; ++i )
         patternLog.addLast( new PatternData( data0.speed + (i+1)*speedDelta, headingDelta ) );
         
      // if the log is too long, cut the oldest frames
      int numRemove = patternLog.size() - maxPatternLogLength;
      while( numRemove > 0 ) {
         patternLog.removeFirst();
         --numRemove;
      }
   }
 
  
}