package pez.mini;
import robocode.*;
import robocode.util.Utils;
import java.awt.geom.*;
import java.awt.Color;
import java.util.*;
import java.util.zip.*;
import java.io.*;

// VertiLeach, by PEZ. Stays close to you and follows your vertical movements
//
// This code is released under the RoboWiki Public Code Licence (RWPCL), datailed on:
// http://robowiki.net/?RWPCL
//
// Code home page: http://robowiki.net/?VertiLeach/Code
//
// $Id: VertiLeach.java,v 1.10 2003/12/28 19:53:12 peter Exp $

public class VertiLeach extends AdvancedRobot {
    private static final double MAX_VELOCITY = 8;
    private static final double DEFAULT_DISTANCE = 300;
    private static final double DEFAULT_BULLET_POWER = 2.8;
    private static final double WALL_MARGIN = 50;
    private static final int DIRECTION_SEGMENTS = 3;
    private static final int VERTICAL_SEGMENTS = 9;
    private static final int AIM_FACTORS = 23;
    private static Point2D robotLocation = new Point2D.Double();
    private static Point2D lastRobotLocation = new Point2D.Double();
    private static Point2D enemyLocation = new Point2D.Double();
    private static String enemyName = "";
    private static double enemyDistance;
    private static double enemyEnergy;
    private static double enemyAbsoluteBearing;
    private static double lastEnemyAbsoluteBearing;
    private static double enemyYNormal;
    private static double lastEnemyYNormal;
    private static int bearingDirection;
    private static double yOffset;
    private static double midY;
    private static int[][][] aimFactors;
    private int[] currentAimFactors;
    private int fullLeadHits;
    private static boolean isNormalMove;
    private double direction = 0.15;
    private Point2D robotDestination = new Point2D.Double();
    private int timeSinceDirectionChange;

    public void run() {
        setColors(Color.red.darker().darker(), Color.green.darker().darker(), Color.gray.darker().darker());
        setAdjustGunForRobotTurn(true);
        setAdjustRadarForGunTurn(true);
	midY = getBattleFieldHeight() / 2;

        do {
            turnRadarRightRadians(Double.POSITIVE_INFINITY); 
        } while (true);
    }

    public void onScannedRobot(ScannedRobotEvent e) {
        if (enemyName == "") {
            enemyName = e.getName();
            restoreFactors();
        }
        lastRobotLocation.setLocation(robotLocation);
        robotLocation.setLocation(getX(), getY());
	lastEnemyAbsoluteBearing = enemyAbsoluteBearing;
        enemyAbsoluteBearing = getHeadingRadians() + e.getBearingRadians();
        enemyDistance = e.getDistance();
        enemyLocation.setLocation(vectorToLocation(enemyAbsoluteBearing, enemyDistance, robotLocation));

	lastEnemyYNormal = enemyYNormal;
	double enemyY = enemyYNormal = enemyLocation.getY();
	if (enemyY > midY) {
	    enemyYNormal = midY - (enemyY - midY);
	}

	double eDelta = enemyEnergy - e.getEnergy();
	if (eDelta >= 0.1 && eDelta <= 3.0) {
	    double maxOffset = 175 + (DEFAULT_DISTANCE - Math.min(DEFAULT_DISTANCE, enemyDistance));
	    yOffset = 2 * maxOffset * Math.random() - maxOffset;
	    if (aimVerticalSegment() < 1) {
		yOffset *= 0.5 * Math.abs(yOffset) * sign(midY - enemyY);
	    }
	}
        enemyEnergy = e.getEnergy();

	if (getTime() > 8) {
	    if (isNormalMove) {
		move();
	    }
	    else {
		trickHeadOnMove();
	    }
	    goTo(robotDestination);

	    double deltaBearing = Utils.normalRelativeAngle(absoluteBearing(lastRobotLocation, enemyLocation) -
		    lastEnemyAbsoluteBearing);
	    if (deltaBearing < 0) {
		bearingDirection = -1;
	    }
	    else if (deltaBearing > 0) {
		bearingDirection = 1;
	    }
	    double bulletPower = (enemyDistance < 2 * DEFAULT_DISTANCE / 3) ? 3.0 : DEFAULT_BULLET_POWER;
	    bulletPower = Math.min(getEnergy() / 5, Math.min(enemyEnergy / 4, bulletPower));

	    currentAimFactors = aimFactors[aimDirectionSegment()][aimVerticalSegment()];

	    setTurnGunRightRadians(Utils.normalRelativeAngle(
			enemyAbsoluteBearing +
			maxEscapeAngle(bulletPower) * bearingDirection * mostVisitedFactor() -
			getGunHeadingRadians()));

	    setFire(bulletPower);

	    if (bulletPower >= DEFAULT_BULLET_POWER) {
		Wave wave = new Wave();
		wave.wTime = getTime();
		wave.wBulletPower = bulletPower;
		wave.wGunLocation.setLocation(robotLocation);
		wave.currentTargetLocation = enemyLocation;
		wave.wAimFactors = currentAimFactors;
		wave.wBearing = enemyAbsoluteBearing;
		wave.wBearingDirection = bearingDirection;
		addCustomEvent(wave);
	    }
	}

        setTurnRadarRightRadians(Utils.normalRelativeAngle(enemyAbsoluteBearing - getRadarHeadingRadians()) * 2);
    }

    public void onCustomEvent(CustomEvent e) {
	Wave wave = (Wave)(e.getCondition());
	wave.wAimFactors[(int)Math.min(AIM_FACTORS - 1, Math.max(0, Math.round(((((
	    wave.wBearingDirection * Utils.normalRelativeAngle(absoluteBearing(wave.wGunLocation, wave.currentTargetLocation) - wave.wBearing)) /
	    maxEscapeAngle(wave.wBulletPower)) + 1) / 2) * AIM_FACTORS)))]++;
    }

    public void onWin(WinEvent e) {
        saveFactors();
    }

    public void onDeath(DeathEvent e) {
        saveFactors();
    }

    static final int MAX_FULL_LEAD_HITS = 2;
    public void onHitByBullet(HitByBulletEvent e) {
	if (!isNormalMove && enemyDistance > 200) {
	    // Check if we are hit with full lead aim
	    // Adapted from Axe's Musashi: http://robowiki.net/?Musashi
	    long fullLeadTime = (long)((enemyDistance * 1.2) / e.getVelocity());
	    if (timeSinceDirectionChange > fullLeadTime) {
		fullLeadHits++;
		isNormalMove = fullLeadHits >= MAX_FULL_LEAD_HITS;
	    }
	}
    }

    private void move() {
	direction = 1;
	for (int i = 0; i < 2; i++) {
	    double X = enemyLocation.getX() + direction * sign(getX() - enemyLocation.getX()) * DEFAULT_DISTANCE;
	    robotDestination = new Point2D.Double(Math.max(WALL_MARGIN, Math.min(getBattleFieldWidth() - WALL_MARGIN, X)),
		Math.max(WALL_MARGIN, Math.min(getBattleFieldHeight() - WALL_MARGIN, enemyLocation.getY() + yOffset)));
	    if (Math.abs(X - robotDestination.getX()) < 100) {
		break;
	    }
	    else {
		direction *= -1;
	    }
	}
    }

    private void trickHeadOnMove() {
	double tries = 0;
	do {
	    robotDestination = vectorToLocation(enemyAbsoluteBearing + Math.PI + direction,
		    enemyDistance * (1.05 - tries / 100.0), enemyLocation);
	    tries++;
	} while (tries < 120 && !(new Rectangle2D.Double(WALL_MARGIN, WALL_MARGIN,
	    getBattleFieldWidth() - WALL_MARGIN * 2, getBattleFieldHeight() - WALL_MARGIN * 2)).contains(robotDestination));
	if (tries > 40) {
	    direction *= -1;
	    timeSinceDirectionChange = -1;
	}
	timeSinceDirectionChange++;
    }

    void goTo(Point2D destination) {
        double angle = Utils.normalRelativeAngle(absoluteBearing(robotLocation, destination) - getHeadingRadians());
	double turnAngle = Math.atan(Math.tan(angle));
        int forward = angle == turnAngle ? 1 : - 1;
        setTurnRightRadians(turnAngle);
        setAhead(robotLocation.distance(destination) * forward);
    }

    static double maxEscapeAngle(double bulletPower) {
	return Math.asin(MAX_VELOCITY / bulletVelocity(bulletPower));
    }

    double mostVisitedFactor() {
        int mostVisited = (AIM_FACTORS - 1) / 2;
        for (int i = 0; i < AIM_FACTORS; i++) {
            if (currentAimFactors[i] > currentAimFactors[mostVisited]) {
                mostVisited = i;
            }
        }
	return ((mostVisited + 0.5) / AIM_FACTORS) * 2 - 1;
    }

    private int aimVerticalSegment() {
	int segment = Math.min((int)(enemyYNormal / (midY / VERTICAL_SEGMENTS)), VERTICAL_SEGMENTS - 1);
	return segment;
    }

    private int aimDirectionSegment() {
        double yDelta = enemyYNormal - lastEnemyYNormal;
        if (yDelta < 0) {
            return 0;
        }
        else if (yDelta > 0) {
            return 2;
        }
        return 1;
    }

    static Point2D vectorToLocation(double angle, double length, Point2D sourceLocation) {
        return new Point2D.Double(sourceLocation.getX() + Math.sin(angle) * length,
            sourceLocation.getY() + Math.cos(angle) * length);
    }

    private static double bulletVelocity(double power) {
        return 20 - 3 * power;
    }

    private static int sign(double v) {
        return v > 0 ? 1 : -1;
    }

    private static double absoluteBearing(Point2D source, Point2D target) {
        return Math.atan2(target.getX() - source.getX(), target.getY() - source.getY());
    }

    void restoreFactors() {
        try {
            ZipInputStream zipin = new ZipInputStream(new
                FileInputStream(getDataFile(enemyName)));
            zipin.getNextEntry();
            aimFactors = (int[][][])(new ObjectInputStream(zipin)).readObject();
        }
        catch (Exception e) {
            aimFactors = new int[DIRECTION_SEGMENTS][VERTICAL_SEGMENTS][AIM_FACTORS];
        }
    }

    void saveFactors() {
        try {
            ZipOutputStream zipout = new ZipOutputStream(new RobocodeFileOutputStream(getDataFile(enemyName)));
            zipout.putNextEntry(new ZipEntry("aimFactors"));
            ObjectOutputStream out = new ObjectOutputStream(zipout);
            out.writeObject(aimFactors);
            out.flush();
            zipout.closeEntry();
            out.close();
        }
        catch (IOException e) {
        }
    }

    class Wave extends Condition {
	long wTime;
	double wBulletPower;
	Point2D wGunLocation = new Point2D.Double();
	Point2D currentTargetLocation;
	double wBearing;
	double wBearingDirection;
	int[] wAimFactors;

	public boolean test() {
	    if (bulletVelocity(wBulletPower) * (getTime() - wTime) > wGunLocation.distance(currentTargetLocation)) {
		removeCustomEvent(this);
		return true;
	    }
	    return false;
	}
    }
}
